//
// NAME: CIDLib_Queue.Hpp
//
// DESCRIPTION:
//
//  This is the header for the queue collection template. It implements the
//  TQueue class template, which provides LIFO and priority based storage of
//  its elements by value.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 02/08/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// -----------------------------------------------------------------------------
//  Forward reference the queue and queue cursor classes
// -----------------------------------------------------------------------------
template <class TElem> class TQueueCursor;
template <class TElem> class TQueue;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TQueueNode
//  PREFIX: node
// -----------------------------------------------------------------------------
template <class TElem> class TQueueNode : public TSLstNode
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TQueueNode( const   TElem&                  objData
                    ,       tCIDLib::EQPriorities   ePriority = tCIDLib::EQPrio_0) :

            __ePriority(ePriority)
            , __objData(objData)
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp(L"TQueueNodeOf");
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        ~TQueueNode()
        {
        }


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::EQPriorities ePriority()
        {
            return __ePriority;
        }

        const TElem& objCur() const
        {
            return __objData;
        }

        TElem& objCur()
        {
            return __objData;
        }


    protected   :
        // ---------------------------------------------------------------------
        //  The queue and its cursor are our friends
        // ---------------------------------------------------------------------
        friend class TQueue<TElem>;
        friend class TQueueCursor<TElem>;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TQueueNode();

        TQueueNode(const TQueueNode<TElem>&);

        tCIDLib::TVoid operator=(const TQueueNode<TElem>&);


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __ePriority
        //      The priority of this node.
        //
        //  __pobjData
        //      A pointer to the data object of this node.
        // ---------------------------------------------------------------------
        tCIDLib::EQPriorities   __ePriority;
        TElem                   __objData;


        // ---------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // ---------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TQueueNode<TElem>,TSLstNode)
};

template <class TElem> tCIDLib::TBoolean TQueueNode<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TQueueNode<TElem>::clsThis;



// -----------------------------------------------------------------------------
//   CLASS: TQueue
//  PREFIX: que
// -----------------------------------------------------------------------------
template <class TElem> class TQueue : public TCollection<TElem>
{
    public  :
        // ---------------------------------------------------------------------
        //  Nested typedefs for the cursor and node types used by a queue.
        // ---------------------------------------------------------------------
        typedef class TQueueCursor<TElem> TCursor;
        typedef TQueueNode<TElem>         TNode;


        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TQueue(const tCIDLib::EMTStates eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __evWait(tCIDLib::EEventState_Reset)
            , __llstQueue(0)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        TQueue( const   tCIDLib::TCard4     c4MaxElems
                , const tCIDLib::EMTStates  eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __evWait(tCIDLib::EEventState_Reset)
            , __llstQueue(c4MaxElems)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        ~TQueue() {}


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid Add(const TElem& objNew)
        {
            TLocker lockQueue(this);
            Put(objNew);
            _c4IncSerialNum();
        }

        tCIDLib::TBoolean bFlushCur()
        {
            TLocker lockQueue(this);

            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Save the next node
            TNode* const pnodeNext = (TNode*)__pnodeCur->pnodeNext();

            // Now flush the iterator node
            __llstQueue.FlushNode(__pnodeCur);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Put the next node back into the iterator
            __pnodeCur = pnodeNext;

            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bIsEmpty() const
        {
            TLocker lockQueue(this);
            return __llstQueue.bIsEmpty();
        }

        tCIDLib::TBoolean bNext()
        {
            TLocker lockQueue(this);
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TNode*)__pnodeCur->pnodeNext();
            return (__pnodeCur != 0);
        }
        
        tCIDLib::TBoolean bResetIter()
        {
            TLocker lockQueue(this);
            __pnodeCur = (TNode*)__llstQueue.pnodeHead();
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            TLocker lockQueue(this);
            return __llstQueue.c4ElemCount();
        }

        tCIDLib::TCard4 c4MaxElemCount() const
        {
            TLocker lockQueue(this);
            return __llstQueue.c4MaxElemCount();
        }

        tCIDLib::TVoid Flush()
        {
            TLocker lockQueue(this);
            __pnodeCur = 0;
            __llstQueue.Flush();
            _c4IncSerialNum();
        }

        TElem& objCur() const
        {
            TLocker lockQueue(this);
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }

        TElemCursor<TElem>* pcursNew() const
        {
            TLocker lockQueue(this);
            return new TCursor(this);
        }


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid FlushAt(TCursor& cursAt)
        {
            TLocker lockQueue(this);

            // Make sure the cursor belongs to this queue
            if (!cursAt.bIsCursoring(*this))
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NotMyCursor
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , cursAt.clsIsA()
                    , clsIsA()
                );
            }

            // Get the node to flush
            TNode* pnodeToFlush = cursAt.pnodeCur();

            // Make sure the cursor's valid
            if (!pnodeToFlush)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Bump up the cursor
            cursAt.bNext();

            // If the flush node is the iterator node, then move it up also
            if (pnodeToFlush == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // And flush the node
            __llstQueue.FlushNode(pnodeToFlush);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // But we kept the passed cursor up to date by bumping it up
            cursAt._c4SerialNum(c4SerialNum());
        }

        TElem objGetNext(const tCIDLib::TCard4 c4MilliSecs = 0)
        {
            // Use a faux block to handle the temporary lock
            tCIDLib::TCard4 c4Count;
            {
                TLocker lockQueue(this);

                // Get the current element count
                c4Count = __llstQueue.c4ElemCount();

                //
                //  If no elements available and the user did not want to wait
                //  around for anything, we just return 0 now.
                //
                if (!c4Count && !c4MilliSecs)
                {
                    facCIDLib.ThrowErr
                    (
                        __FILE__
                        , __LINE__
                        , kCIDErrs::errcCol_IsEmpty
                        , tCIDLib::ESev_APIFailed
                        , tCIDLib::EClass_Timeout
                        , clsIsA()
                    );
                }

                // If none available, then reset the wait sem so we will block
                if (!c4Count)
                    __evWait.Reset();
            }

            //
            //  Wait for something. If there is stuff available, then
            //  its already posted so there's nothing to worry about because
            //  we'll just fall through.
            //
            __evWait.WaitFor(c4MilliSecs);

            // Lock the access mutex again now
            TLocker lockQueue(this);

            // Get the head node
            TNode* pnodeHead = (TNode*)__llstQueue.pnodeHead();

            // If the gotten node is the current iterator node, move it up
            if (pnodeHead == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Get the object out of it and then flush the node
            TElem objRet = pnodeHead->__objData;
            __llstQueue.FlushNode(pnodeHead);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Return the object
            return objRet;
        }

        TElem objGetHighestPrio(const tCIDLib::TCard4 c4MilliSecs = 0)
        {
            // Use a faux block to handle the temporary lock
            tCIDLib::TCard4 c4Count;
            {
                TLocker lockQueue(this);

                // Get the current element count
                c4Count = __llstQueue.c4ElemCount();

                //
                //  If no elements available and the user did not want to wait
                //  around for anything, we throw an exception.
                //
                if (!c4Count && !c4MilliSecs)
                {
                    facCIDLib.ThrowErr
                    (
                        __FILE__
                        , __LINE__
                        , kCIDErrs::errcCol_IsEmpty
                        , tCIDLib::ESev_APIFailed
                        , tCIDLib::EClass_Timeout
                        , clsIsA()
                    );
                }

                // If none available, then close the gate so we will block
                if (!c4Count)
                    __evWait.Reset();
            }

            //
            //  Wait for something. If there is stuff available, then its already
            //  posted so there's nothing to worry about because we'll just fall
            //  through.
            //
            __evWait.WaitFor(c4MilliSecs);

            // Lock the access queue again now
            TLocker lockQueue(this);

            //
            //  Get the highest priority node. We have to loop through and find the
            //  one with the highest priority.
            //
            tCIDLib::EQPriorities   eHighest    = tCIDLib::EQPrio_0;
            TNode*                  pnodeCur    = (TNode*)__llstQueue.pnodeHead();
            TNode*                  pnodeHighest= pnodeCur;
            while (pnodeCur)
            {
                if (pnodeCur->__ePriority > eHighest)
                {
                    eHighest = pnodeCur->__ePriority;
                    pnodeHighest = pnodeCur;
                }
                pnodeCur = (TNode*)pnodeCur->pnodeNext();
            }

            // If 0, then none, so log an error
            if (!pnodeHighest)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_Internal
                    , clsIsA()
                );
            }

            // Get a copy of the node data object
            TElem objRet = pnodeHighest->__objData;

            // If the gotten node is the current iterator node, move it up
            if (pnodeHighest == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Delete the node that we found.
            __llstQueue.FlushNode(pnodeHighest);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // And return the object
            return objRet;
        }

        tCIDLib::TVoid
        Put(const   TElem&                  objToPut
            ,       tCIDLib::EQPriorities   ePriority = tCIDLib::EQPrio_0)
        {
            TLocker lockQueue(this);

            //
            //  Add the new node and trigger the event to let the thread
            //  serving the queue wake up if he's waiting.
            //
            __llstQueue.AppendNode(new TNode(objToPut, ePriority));

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            __evWait.Trigger();
        }


    protected :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TQueueCursor<TElem>;

        friend tCIDLib::TBoolean bCompCollections
        (
            const   TQueue<TElem>&          col1
            , const TQueue<TElem>&          col2
            , const TObjEq<TElem>&          compToUse
        );

        friend TQueue<TElem>* pDupCollection
        (
            const   TQueue<TElem>&          colToDup
        );


        // ---------------------------------------------------------------------
        //  Hidden constructors and operators
        // ---------------------------------------------------------------------
        TQueue(const TQueue<TElem>& colToCopy) :

            TCollection<TElem>(colToCopy)
            , __evWait(tCIDLib::EEventState_Reset)
            , __llstQueue(colToCopy.c4MaxElemCount())
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        const TSLinkedList& _llstQueue() const
        {
            return __llstQueue;
        }

        TSLinkedList& _llstQueue()
        {
            return __llstQueue;
        }


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TQueue<TElem>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString   strTmp(L"TQueueOf");
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __kevWait
        //      This is an event semaphore that allows threads to wait
        //      for something to arrive.
        //
        //  __llstQueue
        //      This is a singly linked list that is the storage for the queue
        //      items. Queues are unidirectional, so we use a singly linked
        //      list for speed and efficiency.
        //
        //  __pnodeCur
        //      The current node for the internal iterator.
        // ---------------------------------------------------------------------
        TEvent              __evWait;
        TSLinkedList        __llstQueue;
        TNode*              __pnodeCur;


        // ---------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a queue of a particular type.
        // ---------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TQueue<TElem>,TCollection<TElem>)
};

template <class TElem> tCIDLib::TBoolean TQueue<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TQueue<TElem>::clsThis;



// -----------------------------------------------------------------------------
//   CLASS: TQueueCursor
//  PREFIX: curs
// -----------------------------------------------------------------------------
template <class TElem> class TQueueCursor : public TElemCursor<TElem>
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TQueueCursor(const TQueue<TElem>* const pcolToCursor) :

            TElemCursor<TElem>(pcolToCursor->c4SerialNum())
            , __pcolCursoring(pcolToCursor)
            , __pllstCursoring(&pcolToCursor->_llstQueue())
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
            __pnodeCur = (TQueueNode<TElem>*)__pllstCursoring->pnodeHead();
        }

        TQueueCursor(const TQueueCursor<TElem>& cursToCopy) :

            TElemCursor<TElem>(cursToCopy)
            , __pcolCursoring(cursToCopy.__pcolCursoring)
            , __pllstCursoring(cursToCopy.__pllstCursoring)
            , __pnodeCur(cursToCopy.__pnodeCur)
        {
            __CheckFirstOfType();
        }

        ~TQueueCursor() {}


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TQueueCursor<TElem>& operator=(const TQueueCursor<TElem>& cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            TParent::operator=(cursToAssign);
            __pcolCursoring = cursToAssign.__pcolCursoring;
            __pllstCursoring = cursToAssign.__pllstCursoring;
            __pnodeCur = cursToAssign.__pnodeCur;

            return *this;
        }


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bIsEmpty() const
        {
            return __pllstCursoring->bIsEmpty();
        }

        tCIDLib::TBoolean bIsValid() const
        {
            // If the serial number is off, its not valid
            if (c4SerialNum() != __pcolCursoring->c4SerialNum())
                return kCIDLib::False;
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bNext()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TQueueNode<TElem>*)__pnodeCur->pnodeNext();

            return (__pnodeCur != 0);
        }
        
        tCIDLib::TBoolean bReset()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            __pnodeCur = (TQueueNode<TElem>*)__pllstCursoring->pnodeHead();

            // Get our serial back in sync with the collection
            _c4SerialNum(__pcolCursoring->c4SerialNum());
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            return __pllstCursoring->c4ElemCount();
        }

        const TElem& objCur() const
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bIsCursoring(const TQueue<TElem>& colToCheck) const
        {
            return (__pcolCursoring == &colToCheck);
        }

        const TQueueNode<TElem>* pnodeCur() const
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }

        TQueueNode<TElem>* pnodeCur()
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TQueue<TElem>;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TQueueCursor();


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString   strTmp("TQueueCursorOf");
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __pcolCursoring
        //      This is a pointer to the queue collection we are cursoring.
        //
        //  __pllstCursoring
        //      This is the linked list we are cursoring. It is extracted from
        //      the queue during construction for fast access.
        //
        //  __pnodeCur
        //      This is the current node that we are on in our iteration. It is
        //      0 if we are at the end.
        // ---------------------------------------------------------------------
        const TQueue<TElem>*    __pcolCursoring;
        const TSLinkedList*     __pllstCursoring;
        TQueueNode<TElem>*      __pnodeCur;

        // ---------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a queue cursor of a particular type.
        // ---------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TQueueCursor<TElem>,TElemCursor<TElem>)
};

template <class TElem> tCIDLib::TBoolean TQueueCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TQueueCursor<TElem>::clsThis;

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Some template functions to make duplication and copying of queues
//  typesafe
// -----------------------------------------------------------------------------
template <class TElem> tCIDLib::TBoolean
bCompCollections(const  TQueue<TElem>&  col1
                , const TQueue<TElem>&  col2
                , const TObjEq<TElem>&  compToUse)
{
    // Check for comparison against self
    if (&col1 == &col2)
        return kCIDLib::True;

    // Lock both of the bags
    TLocker lockQueue1(&col1);
    TLocker lockQueue2(&col2);

    //
    //  First see if the linked lists behind them have equal element counts,
    //  max element counts, and default delete mode. Note that this does
    //  not compare the elements because the linked knows nothing about how
    //  to compare them.
    //
    if (col1.__llstQueue != col2.__llstQueue)
        return kCIDLib::False;

    // If the element count of either is 0, then they are both zero so equal
    if (!col1.c4ElemCount())
        return kCIDLib::True;

    //
    //  Ok so we have to compare the elements. So we just create cursors
    //  for both and compare each element. We know that they have the
    //  same number of elements and are locked so this safe.
    //
    TQueue<TElem>::TCursor curs1(&col1);
    TQueue<TElem>::TCursor curs2(&col2);
    do
    {
        if (!compToUse.bEqual(curs1.objCur(), curs2.objCur()))
            return kCIDLib::False;
    }   while (curs1.bNext() && curs2.bNext());
    return kCIDLib::True;
}

template <class TElem> tCIDLib::TVoid
CopyCollectionNodes(TQueue<TElem>& colDest, const TQueue<TElem>& colSource)
{
    // Check for copy to self
    if (&colDest == &colSource)
        return;

    // Lock both collections
    TLocker lockNew(&colDest);
    TLocker lockToDup(&colSource);

    if (colSource.c4ElemCount())
    {
        TQueue<TElem>::TCursor  cursSource(&colSource);
        do
        {
            TQueue<TElem>::TNode* pnodeCur = cursSource.pnodeCur();
            colDest.Put(pnodeCur->objCur(), pnodeCur->ePriority());
        }   while (cursSource.bNext());
    }
}

template <class TElem> TQueue<TElem>*
pDupCollection(const TQueue<TElem>& colToDup)
{
    // Lock the source collection
    TLocker lockToDup(&colToDup);

    // Create a new queue with the same max elem count
    TQueue<TElem>* const pcolNew = new TQueue<TElem>(colToDup);
    TJanitor<TQueue<TElem> > janCol(pcolNew);

    // If there are any elements in the source, copy them
    if (colToDup.c4ElemCount())
    {
        TQueue<TElem>::TCursor  cursDup(&colToDup);
        do
        {
            TQueue<TElem>::TNode* pnodeCur = cursDup.pnodeCur();
            pcolNew->Put(pnodeCur->objCur(), pnodeCur->ePriority());
        }   while (cursDup.bNext());
    }

    // Return a pointer to the new queue
    janCol.Orphan();
    return pcolNew;
}
