//
// NAME: CIDLib_ProcessRegistry.Hpp
//
// DESCRIPTION:
//
//  This is the header file for the CIDLib_ProcessRegistry.Cpp module. This
//  module provides a process registry for all CIDLib processes on the
//  workstation. This information is kept in a named shared memory segment
//  and is used by the metrics viewer and any other process that needs
//  to know what other CIDLib processes are running.
//
//  The actual implementation of the registry is hidden away in the
//  CIDKernel.DLl facility, and this module is just implemented in terms
//  of the kernel's more primitive services.
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 01/12/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TProcessRegistry;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TRegProcessInfo
//  PREFIX: rpi
// -----------------------------------------------------------------------------
class CIDLIBEXP TRegProcessInfo : public TObject, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TRegProcessInfo();

        TRegProcessInfo
        (
            const   TRegProcessInfo&        rpiToCopy
        );

        ~TRegProcessInfo();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TRegProcessInfo&        rpiToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TRegProcessInfo&        rpiToTest
        )   const;

        TRegProcessInfo& operator=
        (
            const   TRegProcessInfo&        rpiToAssign
        );


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TCard4 c4InfoCard() const;

        tCIDLib::EExitCodes eWaitForDeath
        (
            const   tCIDLib::TCard4         c4MilliSecs = kCIDLib::c4MaxWait
        )   const;

        tCIDLib::TProcessHandle hprocThis() const;

        tCIDLib::TProcessId pidThis() const;

        const TString& strInfoLine() const;

        const TString& strName() const;



    protected   :
        // --------------------------------------------------------------------
        //  Declare friends
        // --------------------------------------------------------------------
        friend  class TProcessRegistry;


        // ---------------------------------------------------------------------
        //  Hidden constructors and operators
        // ---------------------------------------------------------------------
        TRegProcessInfo
        (
            const   tCIDLib::TRawProcessEntry&
        );


        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __c4InfoCard
        //      This is the information cardinal value of the process
        //
        //  __hprocThis
        //      This is a handle for the process. It was created for this
        //      object and will be destroyed when this object is destroyed.
        //
        //  __pidThis
        //      This is the process id of the process
        //
        //  __strInfoLine
        //      This is the information line.
        //
        //  __strName
        //      This is the name of the process.
        // ---------------------------------------------------------------------
        tCIDLib::TCard4         __c4InfoCard;
        tCIDLib::TProcessHandle __hprocThis;
        tCIDLib::TProcessId     __pidThis;
        TString                 __strInfoLine;
        TString                 __strName;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRegProcessInfo,TObject)
};



// -----------------------------------------------------------------------------
//   CLASS: TProcessRegistry
//  PREFIX: preg
// -----------------------------------------------------------------------------
class CIDLIBEXP TProcessRegistry : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        TProcessRegistry() {}

        ~TProcessRegistry() {}


        // --------------------------------------------------------------------
        //  Public, static functions
        // --------------------------------------------------------------------
        static tCIDLib::TCard4 c4GetRegisteredProcs
        (
                    TCollection<TRegProcessInfo>&  colTarget
        );

        static tCIDLib::TCard4 c4InfoCard();

        static tCIDLib::TCard4 c4InfoCard
        (
            const   tCIDLib::TCard4         c4Card
        );

        static tCIDLib::TVoid SetProcessState
        (
            const   tCIDLib::EProcStates    eStatusToSet
        );

        static TString strInfoLine();

        static const TString strInfoLine
        (
            const   TString&                strNewLine
        );


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TProcessRegistry(const TProcessRegistry&);

        tCIDLib::TVoid operator=(const TProcessRegistry&);


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TProcessRegistry,TObject)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  TRegProcessInfo: Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TRegProcessInfo::operator!=(const TRegProcessInfo& rpiToTest) const
{
    return !operator==(rpiToTest);
}


// -----------------------------------------------------------------------------
//  TRegProcessInfo: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TCard4 TRegProcessInfo::c4InfoCard() const
{
    return __c4InfoCard;
}

inline tCIDLib::TProcessHandle TRegProcessInfo::hprocThis() const
{
    return __hprocThis;
}

inline tCIDLib::TProcessId TRegProcessInfo::pidThis() const
{
    return __pidThis;
}

inline const TString& TRegProcessInfo::strInfoLine() const
{
    return __strInfoLine;
}

inline const TString& TRegProcessInfo::strName() const
{
    return __strName;
}
