//
// NAME: CIDLib_ProcessRegistry.Cpp
// 
// DESCRIPTION: 
// 
//  This module implements the classes that provide access to the process
//  registry, which is maintained by the kernel facility.
//
// 
// AUTHOR: Dean Roddey
// 
// CREATE DATE: 01/12/94
// 
// COPYRIGHT: 1992..1997, 'CIDCorp
// 
// CAVEATS/GOTCHAS: 
// 


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TProcessRegistry,TObject)
RTTIData(TRegProcessInfo,TObject)


// ----------------------------------------------------------------------------
//  Local static data
//
//  __pmtxProcReg
//      This is our own mutex to use to lock the process registry. The
//      kernel implementation of the process registry has a TKrnlMutex which
//      it uses, but we need to use a TMutex for exception reasons. Since its
//      named, we can just open our own copy.
// ----------------------------------------------------------------------------
static TMutex*      __pmtxProcReg;



// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermProcessRegistry
//
// DESCRIPTION:
//
//	This function is called from the CIDLib.Dll's initialization.
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modInit is a temp module object for this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid
_InitTermProcessRegistry(const  tCIDLib::EInitTerm      eInitTerm
                        , const tCIDLib::EGlobalStates  eGlobals
                        , const TModule&                modInit
                        , const tCIDLib::TCard4         c4MaxChars
                        ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            // Create the process registry resource name
            TResourceName rsnProcReg
            (
                L"CIDCorp"
                , L"ProcRegistry"
                , L"MainRegistry"
            );

            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midPReg_MutexCreate);
            __pmtxProcReg = new TMutex(rsnProcReg);
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midPReg_MutexDelete);
            delete __pmtxProcReg;
            __pmtxProcReg = 0;
        }
    }
    
    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TProcessRegistry
//  PREFIX: preg
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TProcessRegistry: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TCard4
TProcessRegistry::c4GetRegisteredProcs(TCollection<TRegProcessInfo>& colTarget)
{
    tCIDLib::TCard4 c4Cnt = 0;
    try
    {
        // Validate the registry to get rid of dead entries
        TKrnlProcRegistry::ValidateRegistry();

        // Lock the registry
        TMtxLock mtxlReg(__pmtxProcReg, 5 Seconds);

        // Loop through the entries and get all used ones
        tCIDLib::TCard4 c4Index = 0;
        for (; c4Index < kCIDLib::c4MaxRegProc; c4Index++)
        {
            // Get a pointer to this entry
            const tCIDLib::TRawProcessEntry& Entry
                                        = TKrnlProcRegistry::preEntryAt(c4Index);
            if (Entry.bUsed)
            {
                colTarget.Add(TRegProcessInfo(Entry));
                c4Cnt++;
            }
        }

        #if CID_DEBUG_ON
        if (c4Cnt != TKrnlProcRegistry::c4ProcessCount())
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcPReg_BadEntryCount
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
            );
        }
        #endif
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPReg_GetRegistered
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return c4Cnt;
}


tCIDLib::TCard4 TProcessRegistry::c4InfoCard()
{
    try
    {
        return TKrnlProcRegistry::c4InfoCard();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPReg_ReadAccess
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , facCIDLib.strMsg(kCIDMsgs::midPReg_InfoCard)
        );
    }
}

tCIDLib::TCard4
TProcessRegistry::c4InfoCard(const tCIDLib::TCard4 c4Card)
{
    try
    {
        TKrnlProcRegistry::SetInfoCard(c4Card);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPReg_WriteAccess
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , facCIDLib.strMsg(kCIDMsgs::midPReg_InfoCard)
        );
    }
    return c4Card;
}


tCIDLib::TVoid
TProcessRegistry::SetProcessState(const tCIDLib::EProcStates eNewState)
{
    try
    {
        TKrnlMetricSystem::SetProcessState(eNewState);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPReg_WriteAccess
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , facCIDLib.strMsg(kCIDMsgs::midPReg_ProcessState)
        );
    }
}


TString TProcessRegistry::strInfoLine()
{
    try
    {
        return TString(TKrnlProcRegistry::pszInfoLine());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPReg_ReadAccess
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , facCIDLib.strMsg(kCIDMsgs::midPReg_InfoLine)
        );
    }
}

const TString TProcessRegistry::strInfoLine(const TString& strLine)
{
    try
    {
        TKrnlProcRegistry::SetInfoLine(strLine.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPReg_WriteAccess
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , facCIDLib.strMsg(kCIDMsgs::midPReg_InfoLine)
        );
    }
    return TString(strLine);
}




// ----------------------------------------------------------------------------
//   CLASS: TRegProcessInfo
//  PREFIX: rpi
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TRegProcessInfo: Constructors and Destructors
// ----------------------------------------------------------------------------

TRegProcessInfo::TRegProcessInfo(const tCIDLib::TRawProcessEntry& preToCopy) :

    __c4InfoCard(preToCopy.c4InfoCard)
    , __hprocThis(kCIDLib::hprocInvalid)
    , __pidThis(preToCopy.pidThis)
    , __strInfoLine(preToCopy.szInfoLine)
    , __strName(preToCopy.szProcName)
{
    __hprocThis = TKrnlProcRegistry::hprocFromId(__pidThis);
}

TRegProcessInfo::TRegProcessInfo(const TRegProcessInfo& rpiToCopy) :

    __c4InfoCard(rpiToCopy.__c4InfoCard)
    , __hprocThis(kCIDLib::hprocInvalid)
    , __pidThis(rpiToCopy.__pidThis)
    , __strInfoLine(rpiToCopy.__strInfoLine)
    , __strName(rpiToCopy.__strName)
{
    __hprocThis = TKrnlProcRegistry::hprocFromId(__pidThis);
}

TRegProcessInfo::TRegProcessInfo() :

    __c4InfoCard(0)
    , __hprocThis(kCIDLib::hprocInvalid)
    , __pidThis(kCIDLib::pidInvalid)
{
}

TRegProcessInfo::~TRegProcessInfo()
{
    if (__hprocThis != kCIDLib::hprocInvalid)
    {
        TKrnlProcRegistry::CloseHandle(__hprocThis);
        __hprocThis = kCIDLib::hprocInvalid;
    }
}


// ----------------------------------------------------------------------------
//  TRegProcessInfo: Public operators
// ----------------------------------------------------------------------------

tCIDLib::TBoolean
TRegProcessInfo::operator==(const TRegProcessInfo& rpiToTest) const
{
    if (this == &rpiToTest)
        return kCIDLib::True;

    //
    //  Note that the handle is not tested! If the process id is the same,
    //  the handles refer to the same process even though they are not the
    //  same value!
    //

    if (__c4InfoCard != rpiToTest.__c4InfoCard)
        return kCIDLib::False;

    if (__pidThis != rpiToTest.__pidThis)
        return kCIDLib::False;

    if (__strInfoLine != rpiToTest.__strInfoLine)
        return kCIDLib::False;

    if (__strName != rpiToTest.__strName)
        return kCIDLib::False;

    return kCIDLib::True;
}

TRegProcessInfo& TRegProcessInfo::operator=(const TRegProcessInfo& rpiToAssign)
{
    if (this == &rpiToAssign)
        return *this;

    // If there is a handle, then close it
    if (__hprocThis == kCIDLib::hprocInvalid)
    {
        TKrnlProcRegistry::CloseHandle(__hprocThis);
        __hprocThis = kCIDLib::hprocInvalid;
    }

    // Copy over the fields
    __c4InfoCard    = rpiToAssign.__c4InfoCard;
    __pidThis       = rpiToAssign.__pidThis;
    __strInfoLine   = rpiToAssign.__strInfoLine;
    __strName       = rpiToAssign.__strName;

    // And create a new process handle for ourselves
    __hprocThis = TKrnlProcRegistry::hprocFromId(__pidThis);

    return *this;
}


// ----------------------------------------------------------------------------
//  TRegProcessInfo: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::EExitCodes
TRegProcessInfo::eWaitForDeath(const tCIDLib::TCard4 c4MilliSecs) const
{
    tCIDLib::EExitCodes eRet;

    try
    {
        eRet = TKrnlProcRegistry::eWaitForDeath(__hprocThis, c4MilliSecs);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPReg_WaitForDeath
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
    return eRet;
}


// ----------------------------------------------------------------------------
//  TRegProcessInfo: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TRegProcessInfo::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << facCIDLib.strMsg(kCIDMsgs::midGen_Id) << __pidThis
                    << facCIDLib.strMsg(kCIDMsgs::midGen_Handle)
                    << TOSHandle(__hprocThis)
                    << facCIDLib.strMsg(kCIDMsgs::midGen_Name) << __strName;
}
