//
// NAME: CIDLib_PolyStreamer.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_PolyStreamer.Cpp module. This class
//  supports smart polymorphic streaming, in a way that fully minimizes the
//  storage requirements of such storage.
//
//  In order to polymorphically store objects, the type must be stored with
//  the object. In many cases this overhead is trivial and not an issue. But,
//  for large numbers of small objects, this overhead is greater than the
//  overhead of the objects themselves.
//
//  This smart streamer builds a map of type names, associating an identifier
//  with each one. So, it only has to stream out this identifier with each
//  object. The id is 16 bits (to minimize storage), so only a maximum of
//  64K different types can be streamed into a single file using this class.
//
//  The class type information is stored intermixed with the actual data.
//  When a new type is seen, its written out along with its id. When the file
//  is streamed back in, its known that any object's class will be seen before
//  any object of that type is seen. During the streamout, a hashed set
//  collection of classes written is maintained so that it can quickly see
//  whether a class has been written yet or not. During the stream in, it is
//  built up and used to map ids back to class types.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/07/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class TClassInfoKeyOps;
template <class TElem> class THashSet;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TPolyStreamerBase
// PREFIX: pstmr
// -----------------------------------------------------------------------------
class CIDLIBEXP TPolyStreamerBase : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and destructors
        // ---------------------------------------------------------------------
        TPolyStreamerBase
        (
                    TBinaryStream* const    pstrmToUse
        );

        ~TPolyStreamerBase();


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid Reset();


    protected   :
        // ---------------------------------------------------------------------
        //  Declare our friends
        // ---------------------------------------------------------------------
        friend class TClassInfoKeyOps;

        // ---------------------------------------------------------------------
        //  Protected, virtual methods
        // ---------------------------------------------------------------------
        virtual TObject* _pobjStreamIn
        (
            const   TClass&                 clsType
            ,       TBinaryStream&          strmToUse
        ) = 0;

        virtual tCIDLib::TVoid _StreamOut
        (
            const   TObject&                objToStream
            ,       TBinaryStream&          strmToUse
        ) = 0;


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        TObject* _pobjStreamFrom();

        tCIDLib::TVoid _StreamTo
        (
            const   TObject&                objToStream
        );


    private :
        // ---------------------------------------------------------------------
        //  Nested types
        //
        //  TClassInfo
        //      This is a simple class that allows us to store a TClass object
        //      and an indentifier number that maps to that class.
        // ---------------------------------------------------------------------
        class TClassInfo : public TObject
        {
            public :
                // -------------------------------------------------------------
                //  Constructors and Destructors
                // -------------------------------------------------------------
                TClassInfo() :

                    __c2Id(0)
                {
                }

                TClassInfo(const TClass& clsToStore, const tCIDLib::TCard2 c2Id) :

                    __c2Id(c2Id)
                    , __clsThis(clsToStore)
                {
                }

                TClassInfo(const TClassInfo& clsiToCopy) :

                    __c2Id(clsiToCopy.__c2Id)
                    , __clsThis(clsiToCopy.__clsThis)
                {
                }

                ~TClassInfo()
                {
                }

                // -------------------------------------------------------------
                //  Public operators
                // -------------------------------------------------------------
                TClassInfo& operator=(const TClassInfo& clsiToAssign)
                {
                    if (this == &clsiToAssign)
                        return *this;

                    __c2Id = clsiToAssign.__c2Id;
                    __clsThis = clsiToAssign.__clsThis;
                    return *this;
                }

            protected  :
                // -------------------------------------------------------------
                //  Declare our friends
                // -------------------------------------------------------------
                friend class TPolyStreamerBase;
                friend class TClassInfoKeyOps;


            private :
                // -------------------------------------------------------------
                //  Private data members
                //
                //  __c2Id
                //      The id that is stored in the file for this class
                //
                //  __clsThis
                //      This is the class that this object represents
                // -------------------------------------------------------------
                tCIDLib::TCard2 __c2Id;
                TClass          __clsThis;

                // -------------------------------------------------------------
                //  Magic macros
                // -------------------------------------------------------------
                RTTIMacros(TClassInfo, TObject)
        };

        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TPolyStreamerBase();

        TPolyStreamerBase(const TPolyStreamerBase&);

        tCIDLib::TVoid operator=(const TPolyStreamerBase&);


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean __bCheckOrAdd
        (
            const   TClass&                 clsNewType
            ,       tCIDLib::TCard2&        c2NewId
        );

        tCIDLib::TVoid __FindClassById
        (
            const   tCIDLib::TCard2         c2NextId
            ,       TClass&                 clsNewType
        );


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __c2CurId
        //      This is the current id. Its bumped up for each new class. It
        //      starts at 1 so 0 is an invalid value.
        //
        //  __pcolClassSet
        //      This is the hash set of TClassInfo objects that is used to
        //      track what types have been streamed in our out. This is a
        //      nested class that contains a class and an id.
        //
        //  __pstrmToUse
        //      This is a pointer to the stream to use. We do not adopt it
        //      just reference it.
        // ---------------------------------------------------------------------
        tCIDLib::TCard2         __c2CurId;
        THashSet<TClassInfo>*   __pcolClassSet;
        TBinaryStream*          __pstrmToUse;


        // ---------------------------------------------------------------------
        //  Magic macros
        // ---------------------------------------------------------------------
        RTTIMacros(TPolyStreamerBase, TObject)
};



// -----------------------------------------------------------------------------
//  CLASS: TPolyStreamer
// PREFIX: pstmr
// -----------------------------------------------------------------------------
template <class TElem> class TPolyStreamer : public TPolyStreamerBase
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TPolyStreamer(TBinaryStream* const pstrmToUse) :

            TPolyStreamerBase(pstrmToUse)
        {
        }

        ~TPolyStreamer()
        {
        }


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        TElem* pobjStreamIn()
        {
            return reinterpret_cast<TElem*>(_pobjStreamFrom());
        }

        tCIDLib::TVoid StreamOut(const TElem& objToStream)
        {
            _StreamTo(objToStream);
        }


    protected :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        TObject* _pobjStreamIn(const TClass& clsType, TBinaryStream& strmToUse)
        {
            TObject* const      pobjNew = clsType.pobjMakeNew();
            TJanitor<TObject>   janNewObj(pobjNew);

            //
            //  Make sure that the new type has a valid relationship with the
            //  type of the pointer we have to fill in. The class type we read
            //  in must be a class of or derived from the instantiation type.
            //
            TBinaryStream::CheckRelationship(pobjNew, TElem::clsThis);

            // Its ok so we can cast our pointer safely
            TElem* pobjTyped = reinterpret_cast<TElem*>(pobjNew);

            //
            //  And now let it stream itself in. If this class does not support
            //  the streaming API, it will be caught here at compile time.
            //
            strmToUse >> *pobjTyped;

            // Orphan the new object and give back the pointer
            janNewObj.Orphan();

            return pobjNew;
        }

        tCIDLib::TVoid
        _StreamOut(const TObject& objToStream, TBinaryStream& strmToUse)
        {
            const TElem* pobjTmp = reinterpret_cast<const TElem*>(&objToStream);
            strmToUse << *pobjTmp;
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TPolyStreamer();

        TPolyStreamer(const TPolyStreamer<TElem>&);

        tCIDLib::TVoid operator=(const TPolyStreamer<TElem>&);


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;

        // ---------------------------------------------------------------------
        //  Magic macros
        // ---------------------------------------------------------------------
        RTTIMacros(TPolyStreamer<TElem>, TPolyStreamerBase)
};

template <class TElem> tCIDLib::TBoolean TPolyStreamer<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TPolyStreamer<TElem>::clsThis;

#pragma pack(pop)
