//
// NAME: CIDLib_PolyStreamer.Cpp
//
// DESCRIPTION:
//
//  This module implements the TPolyStreamerBase class. This class supports smart
//  polymorphic streaming of collections of objects.
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 09/07/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  We use the internal hash that the TClass objects have already calculated
//      for themselves. So our TClassKeyOps object does not have to do anything
//      but return the existing hash. But that means we MUST construt the
//      hash set with the same hash modulus!!
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(TPolyStreamerBase,TObject)
RTTIData(TPolyStreamerBase::TClassInfo, TObject)


// -----------------------------------------------------------------------------
//  Local constants
//
//  __c1ClassRecord
//  __c1ObjRecord
//      These are the values written out to indicate whether the next record
//      will be a class or object record.
// -----------------------------------------------------------------------------
static const tCIDLib::TCard1    __c1ClassRecord = 0xAC;
static const tCIDLib::TCard1    __c1ObjRecord   = 0xEA;


// -----------------------------------------------------------------------------
//  A key ops object for a TClass object
// -----------------------------------------------------------------------------
class TClassInfoKeyOps : public TKeyOps<TPolyStreamerBase::TClassInfo>
{
    public :
        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        TClassInfoKeyOps()
        {
        }

        ~TClassInfoKeyOps()
        {
        }

        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean
        bCompKeys(  const   TPolyStreamerBase::TClassInfo&  clsi1
                    , const TPolyStreamerBase::TClassInfo&  clsi2) const
        {
            return (clsi1.__clsThis == clsi2.__clsThis);
        }

        tCIDLib::THashVal
        hshKey( const   TPolyStreamerBase::TClassInfo&  clsiToHash
                , const tCIDLib::TCard4) const
        {
            return clsiToHash.__clsThis.hshInternal();
        }

        // ---------------------------------------------------------------------
        //  Magic macros
        // ---------------------------------------------------------------------
        RTTIMacros(TClassInfoKeyOps, TKeyOps<TPolyStreamerBase::TClassInfo>)
        DefPolyDup(TClassInfoKeyOps)
};

RTTIData(TClassInfoKeyOps, TKeyOps<TPolyStreamerBase::TClassInfo>)


// -----------------------------------------------------------------------------
//  CLASS: TPolyStreamerBase
// PREFIX: pstmr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TPolyStreamerBase: Constructors and destructors
// -----------------------------------------------------------------------------

TPolyStreamerBase::TPolyStreamerBase(TBinaryStream* const pstrmToUse) :

    __c2CurId(1)
    , __pcolClassSet(0)
    , __pstrmToUse(pstrmToUse)
{
    __pcolClassSet = new THashSet<TPolyStreamerBase::TClassInfo>
    (
        kCIDLib::c4ClassModulus
        , new TClassInfoKeyOps
    );
}

TPolyStreamerBase::~TPolyStreamerBase()
{
    delete __pcolClassSet;
}


// -----------------------------------------------------------------------------
//  TPolyStreamerBase: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TPolyStreamerBase::Reset()
{
    __pstrmToUse->Reset();
}


// -----------------------------------------------------------------------------
//  TPolyStreamerBase: Protected, non-virtual methods
// -----------------------------------------------------------------------------

TObject* TPolyStreamerBase::_pobjStreamFrom()
{
    //
    //  Stream in the next record type. There could be a class record before
    //  the next object type.
    //
    tCIDLib::TCard1 c1RecordType;
    *__pstrmToUse >> c1RecordType;

    tCIDLib::TBoolean   bNewClass = kCIDLib::False;
    TClass              clsNewType;
    tCIDLib::TCard2     c2NewId;
    if (c1RecordType == __c1ClassRecord)
    {
        //
        //  Stream in the new record object. First we pull in the new id
        //  and then the class info. We use these to create a new class
        //  info object to add to the collection of classes.
        //
        *__pstrmToUse >> c2NewId;
        clsNewType = __pstrmToUse->clsReadClassInfo();

        // Add this one to the set if its not already
        bNewClass = __bCheckOrAdd(clsNewType, c2NewId);

        // Now get the next record type
        *__pstrmToUse >> c1RecordType;
    }

    // Make sure its a legal record type
    if (c1RecordType != __c1ObjRecord)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_BadRecordType
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_StreamFormat
            , TCardinal(c1RecordType)
        );
    }

    //
    //  Read in the next id. This will be the id of the next object. We
    //  have to map it to a class.
    //
    tCIDLib::TCard2 c2NextId;
    *__pstrmToUse >> c2NextId;

    //
    //  If we just loaded a new class, then it has to be that this object
    //  is of that type. So make sure.
    //
    if (bNewClass && (c2NextId != c2NewId))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NotOfLastId
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_StreamFormat
            , TCardinal(c2NextId)
            , TCardinal(c2NewId)
            , clsNewType
        );
    }

    // If we did not just load a class, then we need to look up this id
    if (!bNewClass)
        __FindClassById(c2NextId, clsNewType);

    return _pobjStreamIn(clsNewType, *__pstrmToUse);
}

tCIDLib::TVoid TPolyStreamerBase::_StreamTo(const TObject& objToStream)
{
    //
    //  See if this object's type has been streamed out yet. If its a new
    //  class, then we stream out the class record first.
    //
    tCIDLib::TCard2 c2ThisId;
    const TClass&   clsThisObj = objToStream.clsIsA();
    
    if (__bCheckOrAdd(clsThisObj, c2ThisId))
    {
        // Stream out the class record id
        *__pstrmToUse << __c1ClassRecord;

        // Now stream out the id for this class and the class itself
        *__pstrmToUse << c2ThisId;
        __pstrmToUse->WriteClassInfo(clsThisObj);
    }

    // Now write out the record id
    *__pstrmToUse << __c1ObjRecord;

    // And stream out the id of this type
    *__pstrmToUse << c2ThisId;

    //
    //  Ask the derived class to write out the object. We cannot do it because
    //  we don't know exactly what the type is.
    //
    _StreamOut(objToStream, *__pstrmToUse);
}


// -----------------------------------------------------------------------------
//  TPolyStreamerBase: Private, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean
TPolyStreamerBase::__bCheckOrAdd(   const   TClass&             clsNewType
                                    ,       tCIDLib::TCard2&    c2NewId)
{
    // Iterate the collection and see if this class is present
    if (__pcolClassSet->bResetIter())
    {
        do
        {
            if (clsNewType == __pcolClassSet->objCur().__clsThis)
            {
                c2NewId = __pcolClassSet->objCur().__c2Id;
                return kCIDLib::False;
            }
        }   while (__pcolClassSet->bNext());
    }

    //
    //  It was not already present, so add it. We bump up the current id
    //  after using it.
    //
    c2NewId = __c2CurId++;
    __pcolClassSet->Add(TClassInfo(clsNewType, c2NewId));

    // Return true to indicate we added a new one
    return kCIDLib::True;
}

tCIDLib::TVoid
TPolyStreamerBase::__FindClassById( const   tCIDLib::TCard2 c2NextId
                                    ,       TClass&         clsNewType)
{
    // Iterate the collection and see if this class is present
    if (__pcolClassSet->bResetIter())
    {
        do
        {
            if (c2NextId == __pcolClassSet->objCur().__c2Id)
            {
                clsNewType = __pcolClassSet->objCur().__clsThis;
                return;
            }
        }   while (__pcolClassSet->bNext());
    }

    // This should never happen
    facCIDLib.LogErr
    (
        __FILE__
        , __LINE__
        , kCIDErrs::errcStrm_TypeIdNotInList
        , tCIDLib::ESev_APIFailed
        , tCIDLib::EClass_StreamFormat
        , TCardinal(c2NextId)
    );
}
