//
// NAME: CIDLib_Point.Cpp
//
// DESCRIPTION:
//
//	This module implements the TPoint class, which is a 2D point with signed,
//  integral coordinates.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TPoint,TObject)


// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------
static tCIDLib::TVoid
__CheckOffset(const tCIDLib::TInt4 i4Value, const tCIDLib::TInt4 i4Ofs)
{
    tCIDLib::TInt8 i8Test = i4Value;
    i8Test += i4Ofs;

    if ((i8Test < kCIDLib::i4MinInt) || (i8Test > kCIDLib::i4MaxInt))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPoint_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }
}



// -----------------------------------------------------------------------------
//   CLASS: TPoint
//  PREFIX: pnt
//
//  This class defines a 2 dimensional point.
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TPoint: Constructors and Destructors
// -----------------------------------------------------------------------------

TPoint::TPoint(const TPoint& pntSrc) :

    __i4X(pntSrc.__i4X)
    , __i4Y(pntSrc.__i4Y)
{
}

TPoint::TPoint() :

    __i4X(0)
    , __i4Y(0)
{
}

TPoint::TPoint(const tCIDLib::THostPoint& ptSrc) :

    __i4X(ptSrc.i4X)
    , __i4Y(ptSrc.i4Y)
{
}

TPoint::TPoint(const tCIDLib::TInt4 i4X, const tCIDLib::TInt4 i4Y) :

    __i4X(i4X)
    , __i4Y(i4Y)
{
}

TPoint::TPoint(const tCIDLib::TInt4 i4Both) :

    __i4X(i4Both)
    , __i4Y(i4Both)
{
}


// -----------------------------------------------------------------------------
//  TPoint: Public operators
// -----------------------------------------------------------------------------

TPoint::operator const tCIDLib::THostPoint*() const
{
    return (const tCIDLib::THostPoint*)&__i4X;
}

TPoint::operator tCIDLib::THostPoint*()
{
    return (tCIDLib::THostPoint*)&__i4X;
}


tCIDLib::TBoolean TPoint::operator==(const TPoint& pntToTest) const
{
    return  (__i4X == pntToTest.__i4X) && (__i4Y == pntToTest.__i4Y);
}


tCIDLib::TVoid TPoint::operator=(const TPoint& pntToAssign)
{
    if (&pntToAssign == this)
        return;

    __i4X = pntToAssign.__i4X;
    __i4Y = pntToAssign.__i4Y;
    return;
}

tCIDLib::TVoid TPoint::operator+=(const TPoint& pntToAdd)
{
    #if CID_DEBUG_ON
    __CheckOffset(__i4X, pntToAdd.__i4X);
    __CheckOffset(__i4Y, pntToAdd.__i4Y);
    #endif

    __i4X += pntToAdd.__i4X;
    __i4Y += pntToAdd.__i4Y;
}

tCIDLib::TVoid TPoint::operator-=(const TPoint& pntToSub)
{
    #if CID_DEBUG_ON
    __CheckOffset(__i4X, -pntToSub.__i4X);
    __CheckOffset(__i4Y, -pntToSub.__i4Y);
    #endif

    __i4X -= pntToSub.__i4X;
    __i4Y -= pntToSub.__i4Y;
}

TPoint operator+(const TPoint& pnt1, const TPoint& pnt2)
{
    #if CID_DEBUG_ON
    __CheckOffset(pnt1.__i4X, pnt2.__i4X);
    __CheckOffset(pnt1.__i4Y, pnt2.__i4Y);
    #endif

    return TPoint(pnt1.__i4X + pnt2.__i4X, pnt1.__i4Y + pnt2.__i4Y);
}

TPoint operator-(const TPoint& pnt1, const TPoint& pnt2)
{
    #if CID_DEBUG_ON
    __CheckOffset(pnt1.__i4X, -pnt2.__i4X);
    __CheckOffset(pnt1.__i4Y, -pnt2.__i4Y);
    #endif

    return TPoint(pnt1.__i4X - pnt2.__i4X, pnt1.__i4Y - pnt2.__i4Y);
}



// -----------------------------------------------------------------------------
//  TPoint: Public, non-virtual methods
// -----------------------------------------------------------------------------

inline tCIDLib::TVoid
TPoint::Adjust(const tCIDLib::TInt4 i4XOfs, const tCIDLib::TInt4 i4YOfs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__i4X, i4XOfs);
    __CheckOffset(__i4Y, i4YOfs);
    #endif

    __i4X += i4XOfs;
    __i4Y += i4YOfs;
}

inline tCIDLib::TVoid TPoint::AdjustX(const tCIDLib::TInt4 i4Ofs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__i4X, i4Ofs);
    #endif

    __i4X += i4Ofs;
}

inline tCIDLib::TVoid TPoint::AdjustY(const tCIDLib::TInt4 i4Ofs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__i4Y, i4Ofs);
    #endif

    __i4Y += i4Ofs;
}

tCIDLib::TBoolean TPoint::bInArea(const TArea& areaTarget) const
{
    if ((__i4X < areaTarget.__i4X)
    ||  (__i4Y < areaTarget.__i4Y)
    ||  (__i4X > areaTarget.i4Right())
    ||  (__i4Y > areaTarget.i4Bottom()))
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}


inline tCIDLib::TCard4 TPoint::c4Packed() const
{
    if ((__i4X > kCIDLib::i2MaxInt) || (__i4Y > kCIDLib::i2MaxInt))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPoint_CantPack
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return TRawBits::c4From16(tCIDLib::TCard2(__i4X), tCIDLib::TCard2(__i4Y));
}


tCIDLib::TVoid TPoint::FromHostPt(const tCIDLib::THostPoint& ptSrc)
{
    __i4X = ptSrc.i4X;
    __i4Y = ptSrc.i4Y;
}


tCIDLib::TVoid TPoint::ToHostPt(tCIDLib::THostPoint& ptTarget) const
{
    ptTarget.i4X = __i4X;
    ptTarget.i4Y = __i4Y;
}

tCIDLib::TVoid
TPoint::Scale(const tCIDLib::TFloat4 f4XScale, const tCIDLib::TFloat4 f4YScale)
{
    // Insure the scaling values are legal
    if ((f4XScale < 0.0) || (f4XScale > 1.0)
    ||  (f4YScale < 0.0) || (f4YScale > 1.0))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPoint_BadScaling
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
        );
    }

    tCIDLib::TFloat8 f4Tmp = __i4X;
    f4Tmp *= f4XScale;
    if ((f4Tmp < kCIDLib::i4MinInt) || (f4Tmp > kCIDLib::i4MaxInt))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPoint_Range
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TFloat(f4Tmp, 0)
        );
    }
    __i4X = tCIDLib::TInt4(f4Tmp);

    f4Tmp = __i4Y;
    f4Tmp *= f4YScale;
    if ((f4Tmp < kCIDLib::i4MinInt) || (f4Tmp > kCIDLib::i4MaxInt))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPoint_Range
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TFloat(f4Tmp, 0)
        );
    }
    __i4Y = tCIDLib::TInt4(f4Tmp);
}


// -----------------------------------------------------------------------------
//  TPoint: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TPoint::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo << __i4X << L"," << __i4Y;
}


tCIDLib::TVoid TPoint::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __i4X;
    strmToReadFrom >> __i4Y;
}

tCIDLib::TVoid TPoint::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __i4X;
    strmToWriteTo << __i4Y;
}
