//
// NAME: CIDLib_PathString.Cpp
//
// DESCRIPTION:
//
//  This module implements a path string class. It is basically a string
//  with added methods to manipulate the path components.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 07/27/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TPathStr,TString)


// ----------------------------------------------------------------------------
//   CLASS: TPathStr
//  PREFIX: path
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TPathStr: Constructors and Destructors
// ----------------------------------------------------------------------------

TPathStr::TPathStr() :

    TString(kCIDLib::pszEmptyZStr, kCIDLib::c4MaxPathLen, kCIDLib::c4MaxPathLen)
{
}

TPathStr::TPathStr(const TString& strInitVal) :

    TString
    (
        strInitVal.pszData()
        , kCIDLib::c4MaxPathLen
        , kCIDLib::c4MaxPathLen
    )
{
}

TPathStr::TPathStr(const TPathStr& pathToCopy) :

    TString(pathToCopy)
{
}

TPathStr::TPathStr(const tCIDLib::Tch* const pszInitValue) :

    TString(pszInitValue, kCIDLib::c4MaxPathLen, kCIDLib::c4MaxPathLen)
{
}

TPathStr::~TPathStr()
{
}


// ----------------------------------------------------------------------------
//  TPathStr: Public, non-virtual
// ----------------------------------------------------------------------------

tCIDLib::TVoid TPathStr::AddLevel(const TString& strNewLevel)
{
    // If there is no text in the new level, then just return
    if (!strNewLevel.c4Length())
        return;

    // Get a local copy of the path separator for efficiency
    tCIDLib::Tch chSeparator = kCIDLib::chPathSeparator;

    //
    //  If anything in this string object, then see if we or the new level
    //  has the separator. If not add one. If both have it, get rid one of.
    //
    tCIDLib::TCard4  c4CurLen = c4Length();
    if (c4CurLen)
    {
        if ((chAt(c4CurLen-1) == chSeparator)
        &&  (strNewLevel.chAt(0) == chSeparator))
        {
            (*this)[c4CurLen-1] = 0;
        }
         else if ((chAt(c4CurLen-1) != chSeparator)
              &&  (strNewLevel.chAt(0) != chSeparator))
        {
            if (c4CurLen)
                Append(chSeparator);
        }
    }
    Append(strNewLevel);
}


tCIDLib::TVoid TPathStr::AppendExtension(const TString& strExt)
{
    if (strExt.bEmpty())
        return;

    tCIDLib::TBoolean bPerEnd   = (chLast() == kCIDLib::chPeriod);
    tCIDLib::TBoolean bExtStart = (strExt.chFirst() == kCIDLib::chPeriod);

    if (bPerEnd && bExtStart)
    {
        DeleteLast();
        bPerEnd = kCIDLib::False;
    }

    if (!bPerEnd && !bExtStart)
        Append(kCIDLib::chPeriod);

    Append(strExt);
}


tCIDLib::TBoolean TPathStr::bIsSpecialDir() const
{
    TPathStr pathTmp;

    if (!bQueryName(pathTmp))
        return kCIDLib::False;

    return TKrnlFileSys::bIsSpecialDir(pathTmp.pszData());
}


tCIDLib::TBoolean TPathStr::bHasExt() const
{
    TKrnlFileSys    kfsysInfo;

    // Look for the last period
    tCIDLib::TCard4 c4LastPeriod;
    if (!bLastOccurrence(L'.', c4LastPeriod))
        return kCIDLib::False;

    // If it is the last character, then no extention
    if (c4LastPeriod+1 == c4Length())
        return kCIDLib::False;

    // If the next char is a \, then a relative path, no extension
    if (chAt(c4LastPeriod+1) == kCIDLib::chPathSeparator)
        return kCIDLib::False;

    return kCIDLib::True;
}


tCIDLib::TBoolean
TPathStr::bParseComponents(TPathStr::TPathComp& PathInfo) const
{
    // Create a state flag for the state machine
    tCIDLib::EPathParts eState  = tCIDLib::EPathPart_Drive;

    // Init the current and last indexes to the start of the string
    tCIDLib::TBoolean   bLoop   = kCIDLib::True;
    tCIDLib::TBoolean   bOk;
    tCIDLib::TCard4     c4Cur, c4Last;
    tCIDLib::Tch        ch0, ch1;

    //
    //  Preload a local with the path separator because its referenced
    //  so many times here.
    //
    tCIDLib::Tch chSeparator = kCIDLib::chPathSeparator;

    // Init the return offsets
    PathInfo.chDrive    = 0;
    PathInfo.szVolume[0]= 0;
    PathInfo.c4PathOfs  = kCIDLib::c4MaxCard;
    PathInfo.c4PathEnd  = kCIDLib::c4MaxCard;
    PathInfo.c4NameOfs  = kCIDLib::c4MaxCard;
    PathInfo.c4NameEnd  = kCIDLib::c4MaxCard;
    PathInfo.c4ExtOfs   = kCIDLib::c4MaxCard;
    PathInfo.c4ExtEnd   = kCIDLib::c4MaxCard;
    PathInfo.c4NodeOfs  = kCIDLib::c4MaxCard;
    PathInfo.c4NodeEnd  = kCIDLib::c4MaxCard;

    // If this string is empty, then just return
    tCIDLib::TCard4 c4Chars = c4Length();
    if (!c4Chars)
        return kCIDLib::False;

    //
    //  Do a quick scan for invalid characters in the path and abort if so.
    //  The last parm tells him that we want to match any single char in the
    //  substring, not the whole substring, which is the default.
    //
    if (bFirstOccurrence(kCIDLib::szBadPathChars, c4Cur, kCIDLib::True))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_PathFormat
            , *this
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_BadParms
        );
        return kCIDLib::False;
    }

    //
    //  See if there is a ':' in the string. If so, must be the second char
    //  and the first char must be a valid drive
    //
    if (bFirstOccurrence(kCIDLib::chColon, c4Cur))
    {
        tCIDLib::Tch chFirstCh = chFirst();
        if ((c4Cur != 1) || !TRawStr::bIsAlpha(chFirstCh))
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcFile_PathFormat
                , *this
                , tCIDLib::ESev_Warning
                , tCIDLib::EClass_BadParms
            );
            return kCIDLib::False;
        }
    }

    //
    //  Check for \\, which is a UNC name with no node name. Or \\\, which is
    //  just a screwed up path. These are just to simplify the state machine
    //  below.
    //
    if ((*this == L"\\\\") || (*this == L"\\\\\\"))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_PathFormat
            , *this
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_BadParms
        );
        return kCIDLib::False;
    }

    //
    //  Check for the special case of \., which is the current directory of
    //  the root
    //
    if (*this == L"\\.")
    {
        PathInfo.c4PathOfs = 0;
        PathInfo.c4PathEnd = 0;
        PathInfo.c4NameOfs = 1;
        PathInfo.c4NameEnd = 1;
        return kCIDLib::True;
    }

    //
    //  Check for the special case of \.., which is the parent directory of
    //  the root
    //
    if (*this == L"\\..")
    {
        PathInfo.c4PathOfs = 0;
        PathInfo.c4PathEnd = 0;
        PathInfo.c4NameOfs = 1;
        PathInfo.c4NameEnd = 2;
        return kCIDLib::True;
    }

    // Check for the two special directories by themselves
    if (*this == L".")
    {
        PathInfo.c4NameOfs = 0;
        PathInfo.c4NameEnd = 0;
        return kCIDLib::True;
    }

    if (*this == L"..")
    {
        PathInfo.c4NameOfs = 0;
        PathInfo.c4NameEnd = 1;
        return kCIDLib::True;
    }

    // Loop until we are done with the string
    c4Cur = 0;
    while (bLoop)
    {
        switch(eState)
        {
            case tCIDLib::EPathPart_Drive :
                //
                //  If the length is less than 2 or the second char is not ':'
                //  there can't be a drive.
                //
                if ((c4Chars < 2) || (chAt(1) != L':'))
                {
                    eState = tCIDLib::EPathPart_Node;
                    break;
                }

                //
                //  Get a temp copy of the first char and convert it to upper
                //  case. We confirmed above that the drive letter is valid.
                //
                PathInfo.chDrive = TRawStr::chUpper(chFirst());

                // And fill in the volume name, the prefered way to do drives
                PathInfo.szVolume[0] = PathInfo.chDrive;
                PathInfo.szVolume[1] = kCIDLib::chColon;
                PathInfo.szVolume[2] = kCIDLib::chPathSeparator;
                PathInfo.szVolume[3] = kCIDLib::chNull;

                //
                //  Bump up the current index, set the state to body, and
                //  break out.
                //
                c4Cur   = 2;
                eState  = tCIDLib::EPathPart_Path;
                break;


            case tCIDLib::EPathPart_Node :
                //
                //  It must be at least 3 chars. If not, then go on to the body
                //
                if (c4Chars < 3)
                {
                    eState = tCIDLib::EPathPart_Path;
                    break;
                }

                //
                //  The string must start with a separator
                //
                ch0 = chFirst();
                ch1 = chAt(1);
                if ((ch0 != chSeparator) || (ch1 != chSeparator))
                {
                    eState = tCIDLib::EPathPart_Path;
                    break;
                }

                //
                //  If there is not another sep, then everything from c4Cur to
                //  the end is the node name. If it is longer than 8 chars,
                //  then not legal. We checked for the pathological case of
                //  \\ above, so don't have to check for that. Set cLast to
                //  1, so that the search will start at the next char.
                //
                c4Last = 1;
                if (!bNextOccurrence(chSeparator, c4Last))
                {
                    PathInfo.c4NodeOfs = c4Cur;
                    PathInfo.c4NodeEnd = c4Chars-1;

                    if (PathInfo.c4NodeEnd-PathInfo.c4NodeOfs > 8)
                    {
                        facCIDLib.LogErr
                        (
                            __FILE__
                            , __LINE__
                            , kCIDErrs::errcFile_PathFormat
                            , *this
                            , tCIDLib::ESev_Warning
                            , tCIDLib::EClass_BadParms
                        );
                        return kCIDLib::False;
                    }

                    // And we are done
                    bLoop = kCIDLib::False;
                    break;
                }

                // So the node is everything from 2 to cLast-1
                PathInfo.c4NodeOfs = 2;
                PathInfo.c4NodeEnd = c4Last-1;

                // Set c4Cur to c4Last, which is where the search will continue
                c4Cur  = c4Last;
                eState = tCIDLib::EPathPart_Path;
                break;


            case tCIDLib::EPathPart_Path :
                //
                //  If there is not a sep in the remaining text, then there
                //  is is either a \ path or no path at all. So look for the
                //  last sep that is further along than cCur.
                //
                if (bLastOccurrence(chSeparator, c4Last))
                {
                    // If it was before c4Cur, then just set it to c4Cur
                    if (c4Last < c4Cur)
                        c4Last = c4Cur;
                }
                 else
                {
                    c4Last = c4Cur;
                }

                // If no more separators, either a single \ or no path at all
                if (c4Last == c4Cur)
                {
                    if (chAt(c4Cur) == chSeparator)
                    {
                        PathInfo.c4PathOfs = c4Cur;
                        PathInfo.c4PathEnd = c4Cur;
                        c4Cur++;
                    }

                    eState = tCIDLib::EPathPart_Name;
                    break;
                }

                //
                //  If sep is c4Cur+1 and c4Cur is a '.' then a relative path
                //
                if ((c4Last == c4Cur+1) && (chAt(c4Cur) == L'.'))
                {
                    PathInfo.c4PathOfs = c4Cur;
                    PathInfo.c4PathEnd = c4Cur+1;

                    c4Cur += 2;
                }
                 else
                {
                    //
                    //  It is the regular case. Everything from c4Cur to
                    //  c4Last is the path
                    //
                    PathInfo.c4PathOfs = c4Cur;
                    PathInfo.c4PathEnd = c4Last;

                    c4Cur = c4Last + 1;
                }

                // Set the current index and move to the next state
                eState  = tCIDLib::EPathPart_Name;
                break;


            case tCIDLib::EPathPart_Name :
                //
                //  Check for the special directories . and ..
                //
                if ((c4Chars-c4Cur == 1) && (chAt(c4Cur) == L'.'))
                {
                    PathInfo.c4NameOfs  = c4Cur;
                    PathInfo.c4NameEnd  = c4Cur;
                    bLoop = kCIDLib::False;
                    break;
                }

                if ((c4Chars-c4Cur == 2)
                    && (chAt(c4Cur) == L'.')
                    && (chAt(c4Cur+1) == L'.'))
                {
                    PathInfo.c4NameOfs  = c4Cur;
                    PathInfo.c4NameEnd  = c4Cur+1;
                    bLoop = kCIDLib::False;
                    break;
                }

                //
                //  The current character now can't be a period or sep
                //
                if ((chAt(c4Cur) == L'.') || (chAt(c4Cur) == chSeparator))
                {
                    facCIDLib.LogErr
                    (
                        __FILE__
                        , __LINE__
                        , kCIDErrs::errcFile_PathFormat
                        , *this
                        , tCIDLib::ESev_Warning
                        , tCIDLib::EClass_BadParms
                    );
                    return kCIDLib::False;
                }

                //
                //  Look for the last period in the remaining text. If there
                //  is not one, then everthing else is the name; else, copy
                //  from  c4Cur up to (but not including) the period.
                //
                bOk = bLastOccurrence(L'.', c4Last);

                if (!bOk || (c4Last <= c4Cur))
                {
                    PathInfo.c4NameOfs = c4Cur;
                    PathInfo.c4NameEnd = c4Chars-1;
                    bLoop = kCIDLib::False;
                    break;
                }

                PathInfo.c4NameOfs = c4Cur;
                PathInfo.c4NameEnd = c4Last-1;

                eState = tCIDLib::EPathPart_Extension;
                c4Cur = c4Last;
                break;


            case tCIDLib::EPathPart_Extension :
                //
                //  Everything else is the extension
                //
                PathInfo.c4ExtOfs = c4Cur;
                PathInfo.c4ExtEnd = c4Chars-1;
                bLoop = kCIDLib::False;
                break;

            default :
                break;
        }

        // If we hit the end of the string, then break out
        if (c4Cur >= c4Chars)
            break;
    }

    // It all went ok
    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bQueryExt(TString& strBuf) const
{
    tCIDLib::TCard4  c4LastPeriod;

    // Init the return string
    strBuf = kCIDLib::pszEmptyZStr;

    // Look for the last period
    if (!bLastOccurrence(L'.', c4LastPeriod))
        return kCIDLib::False;

    // If it is the last character, then no extention
    if (c4LastPeriod+1 == c4Length())
        return kCIDLib::False;

    // If the next char is a \, then a relative path, no extension
    if (chAt(c4LastPeriod+1) == kCIDLib::chPathSeparator)
        return kCIDLib::False;

    // Copy the text from the next character to the end
    strBuf.CopyInSubStr(*this, c4LastPeriod);

    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bQueryName(TString& strBuf) const
{
    TPathStr::TPathComp PathInfo;

    // Init the return string
    strBuf.Clear();

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    //
    //  Append the the name substring onto the current string, which is empty
    //  because we cleared it above
    //
    if (PathInfo.c4NameOfs != kCIDLib::c4MaxCard)
    {
        strBuf.AppendSubStr
        (
            *this
            , PathInfo.c4NameOfs
            , (PathInfo.c4NameEnd-PathInfo.c4NameOfs)+1
        );
    }

    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bQueryNameExt(TString& strBuf) const
{
    TPathStr::TPathComp PathInfo;

    // Init the return string
    strBuf.Clear();

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    //
    //  Append the two substrings onto the current string, which is empty
    //  because we cleared it above
    //
    if (PathInfo.c4NameOfs != kCIDLib::c4MaxCard)
    {
        strBuf.AppendSubStr
        (
            *this
            , PathInfo.c4NameOfs
            , (PathInfo.c4NameEnd-PathInfo.c4NameOfs)+1
        );
    }

    if (PathInfo.c4ExtOfs != kCIDLib::c4MaxCard)
    {
        strBuf.AppendSubStr
        (
            *this
            , PathInfo.c4ExtOfs
            , (PathInfo.c4ExtEnd-PathInfo.c4ExtOfs)+1
        );
    }

    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bQueryPath(TString& strBuf) const
{
    TPathStr::TPathComp PathInfo;

    // Init the return string
    strBuf.Clear();

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    // If the there is no path component
    if (PathInfo.c4PathOfs == kCIDLib::c4MaxCard)
        return kCIDLib::False;

    //
    //  Append the substring onto the current string, which is empty
    //  because we cleared it above
    //
    strBuf.AppendSubStr
    (
        *this
        , PathInfo.c4PathOfs
        , (PathInfo.c4PathEnd-PathInfo.c4PathOfs)+1
    );
    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bQueryVolume(TString& strBuf) const
{
    TPathStr::TPathComp PathInfo;

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    // If the there is no drive, then return false
    if (PathInfo.chDrive == kCIDLib::chNull)
    {
        strBuf.Clear();
        return kCIDLib::False;
    }

    strBuf = PathInfo.szVolume;
    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bRemoveExt()
{
    TPathStr::TPathComp  PathInfo;

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    // If there is an extension, put a null there
    if (PathInfo.c4ExtOfs != kCIDLib::c4MaxCard)
        (*this)[PathInfo.c4ExtOfs] = 0;

    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bRemoveName()
{
    TPathStr::TPathComp  PathInfo;

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    // If there is a name, do a substring cut
    if (PathInfo.c4NameOfs != kCIDLib::c4MaxCard)
        Cut(PathInfo.c4NameOfs, (PathInfo.c4NameEnd-PathInfo.c4NameOfs)+1);

    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bRemoveNameExt()
{
    TPathStr::TPathComp  PathInfo;

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    // If there is an extension, put a null there
    if (PathInfo.c4NameOfs != kCIDLib::c4MaxCard)
        (*this)[PathInfo.c4NameOfs] = 0;

    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bRemovePath()
{
    TPathStr::TPathComp  PathInfo;

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    //
    //  If there is a name component, cut from 0 to its offset. Otherwise,
    //  just zero out the string because there is nothing but path.
    //
    if (PathInfo.c4NameOfs != kCIDLib::c4MaxCard)
        Cut(0, PathInfo.c4NameOfs);
     else
        Clear();

    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bRemoveLevel()
{
    TPathStr::TPathComp  PathInfo;

    // Parse the components
    if (!bParseComponents(PathInfo))
        return kCIDLib::False;

    // If there is no path component
    if (PathInfo.c4PathOfs == kCIDLib::c4MaxCard)
        return kCIDLib::False;

    //
    //  If the path is only 1 character long and it is a sep, then there
    //  is nothing to do.
    //
    if (PathInfo.c4PathEnd-PathInfo.c4PathOfs == 1)
    {
        if (chAt(PathInfo.c4PathOfs) == kCIDLib::chPathSeparator)
            return kCIDLib::False;
    }

    //
    //  Scarf a layer off of the path. We look for another sep behind the
    //  last character. If there is not one, then the whole thing is cleared
    //  out. If there is one, clear out back to it, but leave the sep
    //  itself.
    //
    tCIDLib::TCard4  c4Cut = c4Length()-1;
    if (bPrevOccurrence(kCIDLib::chPathSeparator, c4Cut))
    {
        // Cut from the char after the sep
        Cut(c4Cut+1);
    }
     else
    {
        // No separators, so the whole thing goes
        Clear();
    }
    return kCIDLib::True;
}


tCIDLib::TBoolean TPathStr::bRemoveTrailingSeparator()
{
    tCIDLib::TCard4 c4Len = c4Length();

    if (!c4Len)
        return kCIDLib::False;

    if (chAt(c4Len-1) != kCIDLib::chPathSeparator)
        return kCIDLib::False;

    (*this)[c4Len-1] = 0;

    return kCIDLib::True;
}


tCIDLib::Tch TPathStr::chDrive() const
{
    // If there is not a drive, then return the default drive
    if (chAt(1) != L':')
        return tCIDLib::Tch(0);

    tCIDLib::Tch  chDrv = chFirst();

    if (!TRawStr::bIsAlpha(chDrv))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_PathFormat
            , *this
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }

    // Upper case it
    chDrv = TRawStr::chUpper(chDrv);

    // Cast to an eDrives and return
    return tCIDLib::Tch(chDrv-'A' + tCIDLib::Tch(1));
}
