//
// NAME: CIDLib_ObjectArray.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_ObjArray.Cpp module. This module
//  implements the TObjArray class, which is a templatize array of objects.
//  TObjArray is not a derivative of TCollection because it does not fit well
//  into the standard collection scheme. However, its collection class is a
//  standard collection cursor derivative, so it can be used in many cases
//  where a collection would be used.
//
//  Also implemented here is the TArrayCursor class, which is a class for
//  readonly iteration of the elements of an object array.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 10/21/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//  
//  1)  Since TObjArray does not derive from TCollection, it must maintain
//      its own serial number for invalidating cursors. And it must implement
//      the lockable interface itself.
//


// ----------------------------------------------------------------------------
//  Forward reference the array cursor class
// ----------------------------------------------------------------------------
template <class TElem> class TArrayCursor;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TObjArray
//  PREFIX: obja
// ----------------------------------------------------------------------------
template <class TElem> class TObjArray :

    public TObject, public MDuplicable, public MLockable
{
    public  :
        // --------------------------------------------------------------------
        //  A nested typedef for cursor type used by an array.
        // --------------------------------------------------------------------
        typedef class TArrayCursor<TElem> TCursor;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TObjArray(  const   tCIDLib::TCard4     c4MaxElemCount
                    , const tCIDLib::EMTStates eMTSafe = tCIDLib::EMTState_Unsafe) :

            __c4MaxElemCount(c4MaxElemCount)
            , __c4SerialNum(0)
            , __paobjList(0)
            , __pmtxLock(0)
        {
            __CheckFirstOfType();

            // Allocate the object array
            __paobjList = new TElem[c4MaxElemCount];

            if (eMTSafe)
                __pmtxLock = new TMutex;
        }

        TObjArray(const TObjArray<TElem>& objaToCopy) :

            __c4MaxElemCount(objaToCopy.__c4MaxElemCount)
            , __c4SerialNum(0)
            , __paobjList(0)
            , __pmtxLock(0)
        {
            __CheckFirstOfType();

            // Lock the source
            TLocker lockSource(&objaToCopy);

            // Allocate the node array
            __paobjList = new TElem[__c4MaxElemCount];

            for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxElemCount; c4Ind++)
                __paobjList[c4Ind] = objaToCopy.__paobjList[c4Ind];

            // Create a mutex if the source has one
            if (objaToCopy.__pmtxLock)
                __pmtxLock = new TMutex;
        }

        ~TObjArray()
        {
            // Now delete the array
            delete [] __paobjList;
            __paobjList = 0;

            delete __pmtxLock;
            __pmtxLock = 0;
        }


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        const TElem& operator[](const tCIDLib::TCard4 c4Index) const
        {
            // Lock this object
            TLocker lockThis(this);

            _VerifyIndex(c4Index, __LINE__);
            return __paobjList[c4Index];
        }

        TElem& operator[](const tCIDLib::TCard4 c4Index)
        {
            // Lock this object
            TLocker lockThis(this);

            _VerifyIndex(c4Index, __LINE__);
            return __paobjList[c4Index];
        }

        tCIDLib::TBoolean operator==(const TObjArray<TElem>& objaToTest) const
        {
            if (this == &objaToTest)
                return kCIDLib::True;

            // Lock this object and the source
            TLocker lockThis(this);
            TLocker lockTest(&objaToTest);

            if (objaToTest.__c4MaxElemCount != __c4MaxElemCount)
                return kCIDLib::False;

            // Compare all the objects
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4MaxElemCount; c4Index++)
            {
                if (__paobjList[c4Index] != objaToTest.__paobjList[c4Index])
                    return kCIDLib::False;
            }
            return kCIDLib::True;
        }

        tCIDLib::TBoolean operator!=(const TObjArray<TElem>& objaToTest) const
        {
            return !operator==(objaToTest);
        }

        TObjArray<TElem>& operator=(const TObjArray<TElem>& objaToAssign)
        {
            if (this == &objaToAssign)
                return *this;

            // Lock the source
            TLocker lockTest(&objaToAssign);
            
            // Delete and reallocate
            delete [] __paobjList;
            __paobjList = new TElem[objaToAssign.__c4MaxElemCount];

            // Copy over the members
            __c4MaxElemCount = objaToAssign.__c4MaxElemCount;
            for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxElemCount; c4Ind++)
                __paobjList[c4Ind] = objaToAssign.__paobjList[c4Ind];

            // Bump the serial number
            __c4SerialNum++;

            //
            //  If the source is not MT safe, then make this one not. If
            //  the source is, then make sure this one becomes so.
            //
            if (!objaToAssign.__pmtxLock)
            {
                delete __pmtxLock;
                __pmtxLock = 0;
            }
             else if (!__pmtxLock)
            {
                __pmtxLock = new TMutex;
            }

            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid
        Lock(const tCIDLib::TCard4 c4Timeout = kCIDLib::c4MaxWait) const
        {
            // If this one is lockable, then do the lock
            if (__pmtxLock)
                __pmtxLock->Lock(c4Timeout);
        }

        tCIDLib::TVoid Unlock() const
        {
            // If this one is lockable, then do the unlock
            if (__pmtxLock)
                __pmtxLock->Unlock();
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsMTSafe() const
        {
            return (__pmtxLock != 0);
        }

        tCIDLib::EMTStates eMTState() const
        {
            if (__pmtxLock)
                return tCIDLib::EMTState_Safe;
            return tCIDLib::EMTState_Unsafe;
        }

        tCIDLib::TCard4 c4MaxElemCount() const
        {
            // Lock this object
            tCIDLib::TCard4 c4Ret;
            {
                TLocker lockThis(this);
                c4Ret = __c4MaxElemCount;
            }
            return c4Ret;
        }

        tCIDLib::TCard4 c4SerialNum() const
        {
            // Lock this object
            tCIDLib::TCard4 c4Ret;
            {
                TLocker lockThis(this);
                c4Ret = __c4SerialNum;
            }
            return c4Ret;
        }

        tCIDLib::TVoid ExchangeElems(   tCIDLib::TCard4     c4First
                                        , tCIDLib::TCard4   c4Second)
        {
            // Lock this object
            TLocker lockThis(this);

            // Sanity check both element indexes
            _VerifyIndex(c4First, __LINE__);
            _VerifyIndex(c4Second, __LINE__);

            // If they are the same, then nothing to do
            if (c4First == c4Second)
                return;

            // Ok, we can switch them puppies
            TElem objTmp(__paobjList[c4First]);

            __paobjList[c4First] = __paobjList[c4Second];
            __paobjList[c4Second] = objTmp;

            // Bump the serial number
            __c4SerialNum++;
        }

        const TElem& objAt(const tCIDLib::TCard4 c4Index) const
        {
            // Lock this object and get the reference out
            TElem* pobjRet = 0;
            {
                TLocker lockThis(this);
                _VerifyIndex(c4Index, __LINE__);
                pobjRet = &__paobjList[c4Index];
            }
            return *pobjRet;
        }

        TElem& objAt(const tCIDLib::TCard4 c4Index)
        {
            // Lock this object and get the reference out
            TElem* pobjRet = 0;
            {
                TLocker lockThis(this);
                _VerifyIndex(c4Index, __LINE__);
                pobjRet = &__paobjList[c4Index];
            }
            return *pobjRet;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid
        _VerifyIndex(   const   tCIDLib::TCard4 c4Index
                        , const tCIDLib::TCard4 c4Line) const
        {
            if (c4Index >= __c4MaxElemCount)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , c4Line
                    , kCIDErrs::errcBAry_IndexError
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , clsIsA()
                    , TCardinal(c4Index)
                    , TCardinal(__c4MaxElemCount-1)
                );
            }
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TObjArray();


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp(L"TObjArrayOf");
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4MaxElemCount
        //      The maximum count of elements that they array can expand
        //      up to.
        //
        //  __c4SerialNum
        //      This is the array serial number. It is bumped up each time
        //      that something happens that would invalid a cursor of this
        //      array. For this class, that only happens when the array is
        //      reallocated.
        //
        //  __paobjList
        //      This is a pointer to the the array of objects. It is
        //      allocated in the constructor to be as big as the user wants.
        //
        //  __pmtxLock
        //      This is the optional mutex that allows this array to
        //      be lockable. The derived class indicates to our constructor
        //      whether this collection should be mutlti-thread safe.
        // --------------------------------------------------------------------
        tCIDLib::TCard4     __c4MaxElemCount;
        tCIDLib::TCard4     __c4SerialNum;
        TElem*              __paobjList;
        TMutex*             __pmtxLock;


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TObjArray<TElem>,TObject)
        DefPolyDup(TObjArray<TElem>)
};

template <class TElem> tCIDLib::TBoolean TObjArray<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TObjArray<TElem>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TArrayCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class TArrayCursor : public TElemCursor<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TArrayCursor(const TObjArray<TElem>* const pobjaToCursor) :

            TElemCursor<TElem>(0)
            , __c4CurIndex(0)
            , __c4SerialNum(pobjaToCursor->c4SerialNum())
            , __pobjaCursoring(pobjaToCursor)
        {
            // Set the class name on the first creation of a bag of this type
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp("TArrayCursorOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        TArrayCursor(const TArrayCursor<TElem>& cursToCopy) :

            TElemCursor<TElem>(cursToCopy)
            , __c4CurIndex(cursToCopy.__c4CurIndex)
            , __pobjaCursoring(cursToCopy.__pobjaCursoring)
        {
        }

        ~TArrayCursor() {}



        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TArrayCursor& operator=(const TArrayCursor<TElem>& cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            TParent::operator=(cursToAssign);
            __c4CurIndex = cursToAssign.__c4CurIndex;
            __pobjaCursoring = cursToAssign.__pobjaCursoring;

            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsEmpty() const
        {
            // Arrays are never empty
            return kCIDLib::False;
        }

        tCIDLib::TBoolean bIsValid() const
        {
            // If we are out of sync, then its invalid
            if (c4SerialNum() != __pobjaCursoring->c4SerialNum())
                return kCIDLib::False;

            // If we are at the end, then its invalid
            if (__c4CurIndex >= __pobjaCursoring->c4MaxElemCount())
                return kCIDLib::False;

            return kCIDLib::True;
        }

        tCIDLib::TBoolean bNext()
        {
            _CheckSerialNum(__pobjaCursoring->c4SerialNum());
            __c4CurIndex++;
            if (__c4CurIndex >= __pobjaCursoring->c4MaxElemCount())
                return kCIDLib::False;
            return kCIDLib::True;
        }

        tCIDLib::TBoolean bReset()
        {
            // Get ourself back in sync with object array
            _SetSerialNum(__pobjaCursoring->c4SerialNum());
            __c4CurIndex = 0;
            return kCIDLib::True;
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            // A by value array always has the full number of elements
            return __pobjaCursoring->c4MaxElemCount();
        }

        const TElem& objCur() const
        {
            _CheckSerialNum(__pobjaCursoring->c4SerialNum());

            if (__c4CurIndex >= __pobjaCursoring->c4MaxElemCount())
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pobjaCursoring->objAt(__c4CurIndex);
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TObjArray<TElem>;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TArrayCursor();


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4CurIndex
        //      This is the current index that we are on in our iteration.
        //
        //  __pobjaCursoring
        //      This is the array that we are cursoring. It is provided in
        //      the constructor.
        // --------------------------------------------------------------------
        tCIDLib::TCard4             __c4CurIndex;
        const TObjArray<TElem>*     __pobjaCursoring;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a bag cursor of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TArrayCursor<TElem>,TElemCursor<TElem>)
};

template <class TElem> tCIDLib::TBoolean TArrayCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TArrayCursor<TElem>::clsThis;

#pragma pack(pop)
