//
// NAME: CIDLib_Numeric.Cpp
//
// DESCRIPTION:
//
//  This module provides classes for the basic data types integer, cardinal,
//  and float. They are just to allow numeric values to be treated as
//  objects when needed. They all have their own headers, but all of the
//  implementation is here.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 04/15/92
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TCardinal,TObject)
RTTIData2(TLocCardinal,TCardinal)
RTTIData2(TFloat,TObject)
RTTIData2(TLocFloat,TFloat)
RTTIData2(TInteger,TObject)
RTTIData2(TLocInteger,TInteger)


// ----------------------------------------------------------------------------
//   CLASS: TCardinal
//  PREFIX: c
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TCardinal: Constructors and Destructors
// ----------------------------------------------------------------------------

TCardinal::TCardinal(   const   tCIDLib::TCard4     c4Val
                        , const tCIDLib::ERadices   eRadix) :
    __c4Val(c4Val)
    , __eRadix(eRadix)
{
}

TCardinal::TCardinal() :

    __c4Val(0)
    , __eRadix(tCIDLib::ERadix_Dec)
{
}

TCardinal::TCardinal(const TCardinal& cToCopy) :

    __c4Val(cToCopy.__c4Val)
    , __eRadix(cToCopy.__eRadix)
{
}


// ----------------------------------------------------------------------------
//  TCardinal: Public operators
// ----------------------------------------------------------------------------

TCardinal& TCardinal::operator=(const TCardinal& cNew)
{
    if (this == &cNew)
        return *this;

    // Copy our our members
    __c4Val         = cNew.__c4Val;
    __eRadix        = cNew.__eRadix;

    return *this;
}

tCIDLib::TBoolean TCardinal::operator==(const TCardinal& cToTest) const
{
    if ((__eRadix != cToTest.__eRadix) || (__c4Val != cToTest.__c4Val))
        return kCIDLib::False;
    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TCardinal: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TCardinal::_FormatTo(TTextStream& strmToWriteTo) const
{
    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(__c4Val, szTmp, 64, __eRadix);
    strmToWriteTo << szTmp;
}


tCIDLib::TVoid TCardinal::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __c4Val;
    strmToReadFrom >> __eRadix;
}

tCIDLib::TVoid TCardinal::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __c4Val;
    strmToWriteTo << __eRadix;
}



// ----------------------------------------------------------------------------
//   CLASS: TLocCardinal
//  PREFIX: c
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TLocCardinal: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TLocCardinal::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Get a quick ref to the value
    const tCIDLib::TCard4 c4Fmt = c4Val();

    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(c4Fmt, szTmp, 64, tCIDLib::ERadix_Dec);

    //
    //  If the value was 0, then there's not relevant to locale formatting
    //  so don't bother doing anything.
    //
    if (c4Fmt)
    {
        tCIDLib::TCard4     c4GroupSz = TLocale::c4GroupSize();
        tCIDLib::TCard4     c4Len = TRawStr::c4StrLen(szTmp);
        tCIDLib::TCard4     c4Pos = c4Len;
        tCIDLib::TZStr64    szLocFmt;
        tCIDLib::Tch*       pszCur = szLocFmt;
        for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Len; c4Index++, c4Pos--)
        {
            //
            //  If this is not the left or right most digit, and the position
            //  is divisible by the group size, then we need to put in a
            //  divider.
            //  
            if (c4Pos && (c4Pos != c4Len) && !(c4Pos % c4GroupSz))
                *pszCur++ = TLocale::chThousandsSep();
            *pszCur++ = szTmp[c4Index];
        }

        // Cap it off
        *pszCur = kCIDLib::chNull;

        strmToWriteTo << szLocFmt;
        return;
    }
    strmToWriteTo << szTmp;
}



// ----------------------------------------------------------------------------
//   CLASS: TFloat
//  PREFIX: f
// ----------------------------------------------------------------------------

TFloat::TFloat() :

    __c4Precision(2)
    , __eTrailingStyle(tCIDLib::ETrail_Zeroes)
    , __f8Val(0)
{
}

TFloat::TFloat(const TFloat& fToCopy) :

    __c4Precision(fToCopy.__c4Precision)
    , __eTrailingStyle(fToCopy.__eTrailingStyle)
    , __f8Val(fToCopy.__f8Val)
{
}

TFloat::TFloat( const   tCIDLib::TFloat8        f8Val
                , const tCIDLib::TCard1         c1Precision
                , const tCIDLib::ETrailFormats  eStyle) :

    __c4Precision(c1Precision)
    , __eTrailingStyle(eStyle)
    , __f8Val(f8Val)
{
}

TFloat::TFloat( const   tCIDLib::TInt8          i8Val
                , const tCIDLib::TCard1         c1Precision
                , const tCIDLib::ETrailFormats  eStyle) :

    __c4Precision(c1Precision)
    , __eTrailingStyle(eStyle)
    , __f8Val(tCIDLib::TFloat8(i8Val))
{
}

TFloat::TFloat( const   tCIDLib::TCard4         c4Val
                , const tCIDLib::TCard1         c1Precision
                , const tCIDLib::ETrailFormats  eStyle) :

    __c4Precision(c1Precision)
    , __eTrailingStyle(eStyle)
    , __f8Val(tCIDLib::TFloat8(c4Val))
{
}

TFloat::TFloat( const   tCIDLib::TInt4          i4Val
                , const tCIDLib::TCard1         c1Precision
                , const tCIDLib::ETrailFormats  eStyle) :

    __c4Precision(c1Precision)
    , __eTrailingStyle(eStyle)
    , __f8Val(tCIDLib::TFloat8(i4Val))
{
}



// ----------------------------------------------------------------------------
//  TFloat: Public operators
// ----------------------------------------------------------------------------

TFloat& TFloat::operator=(const TFloat& fNew)
{
    if (this == &fNew)
        return *this;

    // Copy over our values
    __c4Precision       = fNew.__c4Precision;
    __eTrailingStyle    = fNew.__eTrailingStyle;
    __f8Val             = fNew.__f8Val;

    return *this;
}

tCIDLib::TBoolean TFloat::operator==(const TFloat& fToTest) const
{
    // Compare the values of the objects
    if ((fToTest.__f8Val != __f8Val)
    ||  (fToTest.__c4Precision != __c4Precision)
    ||  (fToTest.__eTrailingStyle != __eTrailingStyle))
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TFloat: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TFloat::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format to a temp string, then dump it
    tCIDLib::TZStr128   szTmp;
    TRawStr::FormatVal(__f8Val, szTmp, __c4Precision, 128, __eTrailingStyle);

    strmToWriteTo << szTmp;
}


tCIDLib::TVoid TFloat::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __f8Val;
    strmToReadFrom >> __c4Precision;
    strmToReadFrom >> __eTrailingStyle;
}

tCIDLib::TVoid TFloat::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __f8Val;
    strmToWriteTo << __c4Precision;
    strmToWriteTo << __eTrailingStyle;
}



// ----------------------------------------------------------------------------
//   CLASS: TLocFloat
//  PREFIX: f
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TLocFloat: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TLocFloat::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Get a quick ref to the value
    const tCIDLib::TFloat8 f8Fmt = f8Val();

    // Format to a temp string, then dump it
    tCIDLib::TZStr128   szTmp;
    TRawStr::FormatVal(f8Fmt, szTmp, c1Precision(), 128, tCIDLib::ETrail_Zeroes);

    //
    //  If the value was 0, then there's not relevant to locale formatting
    //  so don't bother doing anything.
    //
    if (f8Fmt != 0.0)
    {
        tCIDLib::TCard4     c4GroupSz = TLocale::c4GroupSize();
        tCIDLib::TCard4     c4Index = 0;
        tCIDLib::TCard4     c4Len = TRawStr::c4StrLen(szTmp);
        tCIDLib::TCard4     c4Pos = c4Len;
        tCIDLib::TZStr64    szLocFmt;
        tCIDLib::Tch*       pszCur = szLocFmt;

        //
        //  Handle the sign separately in order to make the below code
        //  simpler.
        //
        if (f8Fmt < 0.0)
        {
            *pszCur++ = TLocale::chNegSign();

            //
            //  If the formatted value has a sign, then skip over it because
            //  we provide one that is locale specific.
            //
            if ((szTmp[c4Index] == TLocale::chNegSign())
            ||  (szTmp[c4Index] == TLocale::chPosSign()))
            {
                c4Index++;
                c4Pos--;
            }
        }

        for (; c4Index < c4Len; c4Index++, c4Pos--)
        {
            // If the current char is the decimal, then fall out
            if (szTmp[c4Index] == kCIDLib::chPeriod)
                break;

            //
            //  If this is not the left or right most digit, and the position
            //  is divisible by the group size, then we need to put in a
            //  divider.
            //  
            if (c4Pos && (c4Pos != c4Len) && !(c4Pos % c4GroupSz))
                *pszCur++ = TLocale::chThousandsSep();
            *pszCur++ = szTmp[c4Index];
        }

        // Copy over the decimal digits as is
        for (; c4Index < c4Len; c4Index++)
            *pszCur++ = szTmp[c4Index];

        // Cap it off
        *pszCur = kCIDLib::chNull;

        strmToWriteTo << szLocFmt;
        return;
    }
    strmToWriteTo << szTmp;
}



// ----------------------------------------------------------------------------
//   CLASS: TInteger
//  PREFIX: i
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TInteger: Constructors and Destructors
// ----------------------------------------------------------------------------

TInteger::TInteger() :

    __i4Val(0)
    , __eRadix(tCIDLib::ERadix_Dec)
{
}

TInteger::TInteger( const   tCIDLib::TInt4      i4Val
                    , const tCIDLib::ERadices   eRadix) :

    __i4Val(i4Val)
    , __eRadix(eRadix)
{
}

TInteger::TInteger(const TInteger& iToCopy) :

    __i4Val(iToCopy.__i4Val)
    , __eRadix(iToCopy.__eRadix)
{
}

// ----------------------------------------------------------------------------
//  TInteger: Public operators
// ----------------------------------------------------------------------------

TInteger& TInteger::operator=(const TInteger& iNew)
{
    if (this == &iNew)
        return *this;

    // Copy our members
    __i4Val  = iNew.__i4Val;
    __eRadix = iNew.__eRadix;

    return *this;
}

tCIDLib::TBoolean TInteger::operator==(const TInteger& iToTest) const
{
    if ((__eRadix != iToTest.__eRadix) || (__i4Val != iToTest.__i4Val))
        return kCIDLib::False;
    return kCIDLib::True;

}

// ----------------------------------------------------------------------------
//  TInteger: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TInteger::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Do the conversion into a temporary string
    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(__i4Val, szTmp, 64, __eRadix);

    // And append the formatted string to the stream
    strmToWriteTo << szTmp;
}


tCIDLib::TVoid TInteger::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __i4Val;
    strmToReadFrom >> __eRadix;
}

tCIDLib::TVoid TInteger::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __i4Val;
    strmToWriteTo << __eRadix;
}



// ----------------------------------------------------------------------------
//   CLASS: TLocInteger
//  PREFIX: i
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TLocInteger: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TLocInteger::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Get a quick ref to the value
    const tCIDLib::TInt4 i4Fmt = i4Val();

    // Do the conversion into a temporary string
    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(i4Fmt, szTmp, 64, tCIDLib::ERadix_Dec);

    //
    //  If the value was 0, then there's not relevant to locale formatting
    //  so don't bother doing anything.
    //
    if (i4Fmt != 0)
    {
        tCIDLib::TCard4     c4GroupSz = TLocale::c4GroupSize();
        tCIDLib::TCard4     c4Index = 0;
        tCIDLib::TCard4     c4Len = TRawStr::c4StrLen(szTmp);
        tCIDLib::TCard4     c4Pos = c4Len;
        tCIDLib::TZStr64    szLocFmt;
        tCIDLib::Tch*       pszCur = szLocFmt;

        //
        //  Handle the sign separately in order to make the below code
        //  simpler.
        //
        if (i4Fmt < 0.0)
        {
            *pszCur++ = TLocale::chNegSign();

            //
            //  If the formatted value has a sign, then skip over it because
            //  we provide one that is locale specific.
            //
            if ((szTmp[c4Index] == TLocale::chNegSign())
            ||  (szTmp[c4Index] == TLocale::chPosSign()))
            {
                c4Index++;
                c4Pos--;
            }
        }

        for (; c4Index < c4Len; c4Index++, c4Pos--)
        {
            //
            //  If this is not the left or right most digit, and the position
            //  is divisible by the group size, then we need to put in a
            //  divider.
            //  
            if (c4Pos && (c4Pos != c4Len) && !(c4Pos % c4GroupSz))
                *pszCur++ = TLocale::chThousandsSep();
            *pszCur++ = szTmp[c4Index];
        }

        // Cap it off
        *pszCur = kCIDLib::chNull;

        strmToWriteTo << szLocFmt;
        return;
    }
    strmToWriteTo << szTmp;
}
