//
// NAME: CIDLib_Mutex.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_Mutex.Cpp module. It implements the
//  TMutex class, which is a mutual exclusion semaphore class.
//
//  This class implements the MLockable interface, therefore it can be
//  locked using the standard TLocker lock janitor class.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/23/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  The locked/unlocked state is a mutable attribute (actually its not
//      a member at all) so the lock/unlock methods are const (which is
//      required by the MLockable interface anyway.)
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//  CLASS: TMutex
// PREFIX: mtx
// ----------------------------------------------------------------------------
class CIDLIBEXP TMutex : public TObject, public MFormattable, public MLockable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors.
        // --------------------------------------------------------------------
        TMutex();

        TMutex
        (
            const   tCIDLib::ELockStates    eInitState
            , const tCIDLib::EShareStates   eShareState = tCIDLib::EShareState_Unshared
        );

        TMutex
        (
            const   TResourceName&          rsnToUse
            , const tCIDLib::ELockStates    eInitState = tCIDLib::ELockState_Unlocked
            , const tCIDLib::ECreateActions eCreateAct = tCIDLib::ECreateAct_OpenOrCreate
        );

        ~TMutex();


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bNamed() const;

        tCIDLib::EShareStates eShareState() const;

        const TResourceName& rsnName() const;


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Lock
        (
            const   tCIDLib::TCard4         c4Timeout = kCIDLib::c4MaxWait
        )   const;

        tCIDLib::TVoid Unlock() const;


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TMutex(const TMutex&);

        tCIDLib::TVoid operator=(const TMutex&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __bNamed
        //      Indicates whether this is a named mutex or not.
        //
        //  __kmtxImpl
        //      This is the kernel mutex object that provides the real
        //      implementation and platform abstraction.
        //
        //  __rsnThis
        //      The name of the mutex, in a resource name object.
        // --------------------------------------------------------------------
        tCIDLib::TBoolean       __bNamed;
        TKrnlMutex              __kmtxImpl;
        TResourceName           __rsnThis;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TMutex,TObject)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Public, inherited methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TMutex::bNamed() const
{
    return __bNamed;
}

inline tCIDLib::EShareStates TMutex::eShareState() const
{
    return __kmtxImpl.eShareState();
}

inline const TResourceName& TMutex::rsnName() const
{
    return __rsnThis;
}
