//
// NAME: CIDLib_Mutex.Cpp
//
// DESCRIPTION:
//
//  This module implements the TMutex class, which is a mutual exclusion
//  semaphore. Only one thread at a time can own it. It can be 'janitorized'
//  by way of the TLocker template class because it supports the needed
//  Lock/Unlock syntax.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/20/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TMutex,TObject)




// ----------------------------------------------------------------------------
//   CLASS: TMutex
//  PREFIX: mtx
//
//  This class provides a mutual exclusion semaphore class
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TMutex: Constructors and Destructors
// ----------------------------------------------------------------------------

TMutex::TMutex() :

    __bNamed(kCIDLib::False)
    , __kmtxImpl(tCIDLib::EShareState_Unshared)
{
    try
    {
        __kmtxImpl.Create(tCIDLib::ELockState_Unlocked);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMtx_Create
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
        );
    }
}

TMutex::TMutex( const   tCIDLib::ELockStates    eInitState
                , const tCIDLib::EShareStates   eShareState) :

    __bNamed(kCIDLib::False)
    , __kmtxImpl(eShareState)
{
    try
    {
        __kmtxImpl.Create(eInitState);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMtx_Create
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
        );
    }
}

TMutex::TMutex( const   TResourceName&          rsnToUse
                , const tCIDLib::ELockStates    eInitState
                , const tCIDLib::ECreateActions eAction) :

    __bNamed(kCIDLib::True)
    , __kmtxImpl
      (
        rsnToUse.strFullName(tCIDLib::ENamedRsc_Mutex).pszData()
      )
    , __rsnThis(rsnToUse)
{
    #if CID_DEBUG_ON
    //
    //  Check for illegal create actions for this type of operation.
    //  
    if ((eAction == tCIDLib::ECreateAct_ReplaceIfExists)
    ||  (eAction == tCIDLib::ECreateAct_TruncateIfExists))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMtx_CreateAction
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TString(L"'Replace/truncate if exists'")
        );
    }
    #endif

    //
    //  First try to open it, then see what that means vis a vis the
    //  open actions. If it fails, then try the create if we are
    //  asked to.
    //
    try
    {
        __kmtxImpl.Open();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        // If we were asked to open an existing only, then we failed
        if (eAction == tCIDLib::ECreateAct_OpenIfExists)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMtx_Open
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
            );
        }

        // If we were asked to create then try that
        if ((eAction == tCIDLib::ECreateAct_CreateIfNew)
        ||  (eAction == tCIDLib::ECreateAct_OpenOrCreate))
        {
            try
            {
                __kmtxImpl.Create(eInitState);
            }

            catch(const TKrnlError& kerrToCatch)
            {
                facCIDLib.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcMtx_Create
                    , kerrToCatch
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                    , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
                );
            }
            return;
        }
    }

    // If we should fail if exists, then close it and fail
    if (eAction == tCIDLib::ECreateAct_FailIfExists)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMtx_AlreadyExists
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
        );
    }
}

TMutex::~TMutex()
{
    try
    {
        __kmtxImpl.Close();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        //
        //  If we get an error, log it but don't throw out of the destructor
        //  so we use a warning message.
        //
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMtx_Close
            , kerrToCatch
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_Internal
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
        );
    }
}


tCIDLib::TVoid TMutex::Lock(const tCIDLib::TCard4 c4Timeout) const
{
    try
    {
        __kmtxImpl.Lock(c4Timeout);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if (kerrToCatch.errcId() == kKrnlErrors::errcTimeout)
        {
            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMtx_Timeout
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMtx_LockError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
            );
        }
    }
}


tCIDLib::TVoid TMutex::Unlock() const
{
    try
    {
        __kmtxImpl.Unlock();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMtx_Unlock
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex)
        );
    }
}


// ----------------------------------------------------------------------------
//  TMutex: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TMutex::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo << L"Name="
                  << __rsnThis.strFullName(tCIDLib::ENamedRsc_Mutex);
}
