//
// NAME: CIDLib_Money.Cpp
//
// DESCRIPTION:
//
//  This module provides a formatting class for monetary values. Such values
//  have special formatting needs, which need to conform to locale specific
//  rules.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 09/15/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TMoney,TObject)


// ----------------------------------------------------------------------------
//   CLASS: TMoney
//  PREFIX: mon
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TMoney: Constructors and Destructors
// ----------------------------------------------------------------------------

TMoney::TMoney() :

    __f8Val(0)
{
}

TMoney::TMoney(const TMoney& fToCopy) :

    __f8Val(fToCopy.__f8Val)
{
}

TMoney::TMoney(const tCIDLib::TFloat8 f8Val) :

    __f8Val(f8Val)
{
}


// ----------------------------------------------------------------------------
//  TMoney: Public operators
// ----------------------------------------------------------------------------

TMoney& TMoney::operator=(const TMoney& monNew)
{
    if (this == &monNew)
        return *this;

    // Copy over our values
    __f8Val = monNew.__f8Val;

    return *this;
}

tCIDLib::TBoolean TMoney::operator==(const TMoney& monToTest) const
{
    // Compare the values of the objects
    return (monToTest.__f8Val == __f8Val);
}


// ----------------------------------------------------------------------------
//  TMoney: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TMoney::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Get a quick ref to the value
    const tCIDLib::TFloat8 f8Fmt = f8Val();

    // Format to a temp string, then dump it
    tCIDLib::TZStr128   szTmp;
    TRawStr::FormatVal
    (
        f8Fmt
        , szTmp
        , TLocale::c4MonetaryDecDigits()
        , 128
        , tCIDLib::ETrail_Zeroes
    );

    //
    //  If the value was 0, then there's not relevant to locale formatting
    //  so don't bother doing anything.
    //
    if (f8Fmt != 0.0)
    {
        tCIDLib::TBoolean   bNegative = (f8Fmt < 0.0);
        tCIDLib::TCard4     c4GroupSz = TLocale::c4MonetaryGrpSize();
        tCIDLib::TCard4     c4Index = 0;
        tCIDLib::TCard4     c4Len = TRawStr::c4StrLen(szTmp);
        tCIDLib::TCard4     c4Pos = c4Len;
        tCIDLib::TZStr64    szLocFmt;
        tCIDLib::Tch*       pszCur = szLocFmt;

        //
        //  Handle the negative sign, if its present, separately in order
        //  to simplify the code below. We ignore it in the string by just
        //  starting the index up one and position down one.
        //
        if (szTmp[c4Index] == TLocale::chNegSign())
        {
            c4Index++;
            c4Pos--;
        }

        //
        //  Get the sign information. We use a different one according
        //  to whether the value is negative or positive.
        //
        tCIDLib::TBoolean       bSymPrecedes;
        tCIDLib::TBoolean       bSpacedSym;
        tCIDLib::Tch            chSign;
        tCIDLib::ESignPositions eSignPos;
        if (bNegative)
        {
            bSymPrecedes = TLocale::bNegSymPrecedes();
            bSpacedSym = TLocale::bNegSpacedSym();
            chSign = TLocale::chNegSign();
            eSignPos = TLocale::eNegSignPosition();
        }
        else
        {
            bSymPrecedes = TLocale::bPosSymPrecedes();
            bSpacedSym = TLocale::bPosSpacedSym();
            chSign = TLocale::chPosSign();
            eSignPos = TLocale::ePosSignPosition();
        }

        //
        //  First see if the sign position is before everything else. If
        //  so, then place it in now if not 0.
        //
        if (eSignPos == tCIDLib::ESignPos_SignFirst)
        {
            if (chSign)
                *pszCur++ = chSign;
        }
         else if (eSignPos == tCIDLib::ESignPos_Parens)
        {
            *pszCur++ = kCIDLib::chOpenParen;
        }

        //
        //  If the symbol procedes the value, then lets add it now. If
        //  its spaced, then add a space after it. If the sign immediately
        //  procedes or suceeds the currency symbol then do it.
        //
        if (bSymPrecedes)
        {
            if (eSignPos == tCIDLib::ESignPos_BeforeCurrency)
            {
                if (chSign)
                    *pszCur++ = chSign;
            }

            if (TLocale::chCurrencySymbol())
                *pszCur++ = TLocale::chCurrencySymbol();

            // If the symbol is spaced, then output a space
            if (bSpacedSym)
                *pszCur++ = kCIDLib::chSpace;

            if (eSignPos == tCIDLib::ESignPos_AfterCurrency)
            {
                if (chSign)
                    *pszCur++ = chSign;
            }
        }

        for (; c4Index < c4Len; c4Index++, c4Pos--)
        {
            //
            //  If the current char is the decimal, then replace it with
            //  the monetary decimal for this locale and then break out.
            //
            if (szTmp[c4Index] == kCIDLib::chPeriod)
            {
                if (TLocale::chMonetaryDecimalSym())
                    *pszCur++ = TLocale::chMonetaryDecimalSym();

                c4Index++;
                break;
            }

            //
            //  If this is not the left or right most digit, and the position
            //  is divisible by the group size, then we need to put in a
            //  divider.
            //  
            if (c4Pos && (c4Pos != c4Len) && !(c4Pos % c4GroupSz))
            {
                if (TLocale::chMonetaryGrpSep())
                    *pszCur++ = TLocale::chMonetaryGrpSep();
            }
            *pszCur++ = szTmp[c4Index];
        }

        // Copy over the decimal digits as is
        for (; c4Index < c4Len; c4Index++)
            *pszCur++ = szTmp[c4Index];

        //
        //  If the symbol succeeds the value, then lets add it now. If
        //  its spaced, then add a space before it. If the sign immediately
        //  procedes or suceeds the currency symbol then do it.
        //
        if (!bSymPrecedes)
        {
            // If the symbol is spaced, then output a space
            if (bSpacedSym)
                *pszCur++ = kCIDLib::chSpace;

            if (eSignPos == tCIDLib::ESignPos_BeforeCurrency)
            {
                if (chSign)
                    *pszCur++ = chSign;
            }

            if (TLocale::chCurrencySymbol())
                *pszCur++ = TLocale::chCurrencySymbol();

            if (eSignPos == tCIDLib::ESignPos_AfterCurrency)
            {
                if (chSign)
                    *pszCur++ = chSign;
            }
        }

        //
        //  See if the sign position is after everything else. If so, then
        //  place it in now if not 0.
        //
        if (eSignPos == tCIDLib::ESignPos_SignLast)
        {
            if (chSign)
                *pszCur++ = chSign;
        }
         else if (eSignPos == tCIDLib::ESignPos_Parens)
        {
            *pszCur++ = kCIDLib::chCloseParen;
        }

        // Cap it off
        *pszCur = kCIDLib::chNull;

        strmToWriteTo << szLocFmt;
        return;
    }
    strmToWriteTo << szTmp;
}

tCIDLib::TVoid TMoney::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __f8Val;
}

tCIDLib::TVoid TMoney::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __f8Val;
}
