//
// NAME: CIDLib_ModuleInit_.Hpp
//
// DESCRIPTION:
//
//  This header provides a generic module init function that all DLLs can
//  call to handle their module init/term work. They just provide a list
//  of init/term functions, which this function will call in the appropriate
//  series of steps.
//
//  Because of the fact that _DllMainCRTStartup calls the constructors for
//  DLL in which it is called (i.e. it ignores the module handle), this
//  has to be done as a template, so that it actually gets generated into
//  each DLL's entry point. Otherwise it would just cause CIDLib.Dll's
//  globals to construct over and over again.
//
//  We just make the template type be the type of the record that is used
//  to build each entry of the module init function array. It doesn't really
//  matter, because we won't every instantiate it for any other type, but
//  it wants us to use the template instantiation type somewhere in order
//  to be a valid template. The caller is unaware of this either way and
//  just calls the function.
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 05/11/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  Because of the special nature of this header it is not provided
//      via the main facility header. Each DLL's main module will import
//      it themselves.
//


// ----------------------------------------------------------------------------
//  Extern prototypes
//
//  _DllMainCRTStartup
//      This is the RTL method to kick off global constructors.
// ----------------------------------------------------------------------------
extern "C" tCIDLib::TCard4 DLLINITAPI _DllMainCRTStartup
(
    tCIDLib::TVoid*
    , tCIDLib::TUInt
    , tCIDLib::TVoid*
);

//
// FUNCTION/METHOD NAME: bCIDLib_CallModuleInitTerm
//
// DESCRIPTION:
//
//  This is a helper function for the main module init function, below. This
//  guy actually does all of the grunt work. The main function below just
//  calls it twice, once for each globals state.
// ---------------------------------------
//   INPUT: eGlobals indicates whether this is for before or after globals
//              are constructed or destructed.
//          eInitTerm is the reason we are getting called, which will be
//              init or term.
//          modInit is the module object, which is passed to any init
//              functions that need it.
//          c4InitCount is the count of init functions in the list.
//          afnInitTerm is the list of init/term functions.
//          pszTitle1, pszTitle2 are the two titles used in error popups
//              if an error occurs during init.
//
//  OUTPUT: None
//
//  RETURN: True if successful, else False.
//
template <class T> tCIDLib::TBoolean
bCIDLib_CallModuleInitTerm( const   tCIDLib::EGlobalStates  eGlobals
                            , const tCIDLib::EInitTerm      eInitTerm
                            , const TModule&                modInit
                            , const tCIDLib::TCard4         c4InitCount
                            , const T*                      afnInitTerm
                            , const tCIDLib::Tch* const     pszTitle1
                            , const tCIDLib::Tch* const     pszTitle2)
{
    tCIDLib::TBoolean       bSuccess = kCIDLib::False;
    const tCIDLib::TCard4   c4FailBufMax = 1024;
    const tCIDLib::TCard4   c4MsgBufMax = 1024;
    tCIDLib::TOSErrCode     errcHostId = 0;
    tCIDLib::TErrCode       errcId = 0;
    tCIDLib::TErrCode       errcKrnlId = 0;
    tCIDLib::Tch            szFailBuffer[c4FailBufMax+1];
    tCIDLib::Tch            szMsgBuf[c4MsgBufMax+1];
    const tCIDLib::Tch*     pszPhase = L"Unknown";

    TRawStr::CopyStr(szFailBuffer, L"Uknown", c4FailBufMax);
    try
    {
        if (eInitTerm == tCIDLib::EInitTerm_Initialize)
        {
            // We are initing so call each init function
            for (tCIDLib::TCard4 c4Index = 0; c4Index < c4InitCount; c4Index++)
            {
                pszPhase = afnInitTerm[c4Index].pszDescription;
                afnInitTerm[c4Index].pfnInitTermFunc
                (
                    eInitTerm
                    , eGlobals
                    , modInit
                    , c4FailBufMax
                    , szFailBuffer
                );
            }
        }
         else
        {
            // We are terminating, so call them backwards this time
            tCIDLib::TInt4 i4Index = c4InitCount-1;
            while (i4Index > 0)
            {
                pszPhase = afnInitTerm[i4Index].pszDescription;
                afnInitTerm[i4Index].pfnInitTermFunc
                (
                    eInitTerm
                    , eGlobals
                    , modInit
                    , c4FailBufMax
                    , szFailBuffer
                );
                i4Index--;
            }
        }

        //
        //  If we are in initialize phase and before globals, then
        //  do the constructors. If we are in term phase and before
        //  globals, then do the destructors.
        //
        if (((eGlobals == tCIDLib::EGlobalState_Before)
        &&   (eInitTerm == tCIDLib::EInitTerm_Initialize))
        ||  ((eGlobals == tCIDLib::EGlobalState_After)
        &&   (eInitTerm == tCIDLib::EInitTerm_Terminate)))
        {
            if (eInitTerm == tCIDLib::EInitTerm_Initialize)
                pszPhase = L"Global/Static Constructor";
            else
                pszPhase = L"Global/Static Destructor";

            _DllMainCRTStartup(modInit.kmodThis().hmodThis(), eInitTerm, 0);
        }
        bSuccess = kCIDLib::True;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        TRawStr::CopyCatStr
        (
            szMsgBuf
            , c4MsgBufMax
            , L"A kernel error occured during the "
            , pszPhase
            , L" phase of init/term. Reason="
            , szFailBuffer
        );
        errcKrnlId = kerrToCatch.errcId();
        errcHostId = kerrToCatch.errcHostId();
    }

    catch(const TError& errToCatch)
    {
        TRawStr::CopyCatStr
        (
            szMsgBuf
            , c4MsgBufMax
            , L"An error occured during the "
            , pszPhase
            , L" phase of init/term. Reason="
            , szFailBuffer
        );
        errcId = errToCatch.errcId();
        errcKrnlId = errToCatch.errcKrnlId();
        errcHostId = errToCatch.errcHostId();
    }

    catch(...)
    {
        TRawStr::CopyCatStr
        (
            szMsgBuf
            , c4MsgBufMax
            , L"An unknown exception occured during the "
            , pszPhase
            , L" phase of init/term."
        );
    }

    if (!bSuccess)
    {
        // Do a runtime error dump <TBD>

        TFacCIDLib::PopUpErr
        (
            __FILE__
            , __LINE__
            , pszTitle1
            , pszTitle2
            , errcId
            , errcKrnlId
            , errcHostId
            , szMsgBuf
            , kCIDLib::pszEmptyZStr
        );
    }
    return bSuccess;
}


// -----------------------------------------------------------------------------
//  Global functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: bCIDLib_DoModuleInitTerm
//
// DESCRIPTION:
//
//  This is the standard module init handler. Each DLL provides its own
//  entry point, which just calls this method, passing along the init/term
//  information and the list of init/term methods.
// ---------------------------------------
//   INPUT: eGlobals indicates whether this is for before or after globals
//              are constructed or destructed.
//          eInitTerm is the reason we are getting called, which will be
//              init or term.
//          hmodThis is the module handle, which is passed to any init
//              functions that need it.
//          eModType is the type of module being inited.
//          c4InitCount is the count of init functions in the list.
//          afnInitTerm is the list of init/term functions.
//          pszTitle1, pszTitle2 are the two titles used in error popups
//              if an error occurs during init.
//
//  OUTPUT: None
//
//  RETURN: True if successful, else False.
//
template <class T> tCIDLib::TBoolean
bCIDLib_DoModuleInitTerm(  const   tCIDLib::EInitTerm   eInitTerm
                            , const tCIDLib::TModHandle hmodThis
                            , const tCIDLib::EModTypes  eModType
                            , const tCIDLib::TCard4     c4InitCount
                            , const T*                  afnInitTerm
                            , const tCIDLib::Tch* const pszTitle1
                            , const tCIDLib::Tch* const pszTitle2)
{
    //
    //  If this is not one of our known init/term values, or the caller
    //  has no init/term functions, we just pass it on to the host RTL
    //  function and return its return.
    //
    if (((eInitTerm != tCIDLib::EInitTerm_Initialize)
    &&   (eInitTerm != tCIDLib::EInitTerm_Terminate))
    ||  !c4InitCount)
    {
        if (_DllMainCRTStartup(hmodThis, eInitTerm, 0))
            return kCIDLib::True;
        return kCIDLib::False;
    }

    //
    //  Create a TModule object for use during module init. This can
    //  be used by any init code that needs to load strings or anything like
    //  that.
    //
    TModule modInit(hmodThis, eModType);

    // Call once for pre global constructor or destructor
    if (!bCIDLib_CallModuleInitTerm
    (
        tCIDLib::EGlobalState_Before
        , eInitTerm
        , modInit
        , c4InitCount
        , afnInitTerm
        , pszTitle1
        , pszTitle2))
    {
        return kCIDLib::False;
    }

    // And once for post global constructor or destructor
    if (!bCIDLib_CallModuleInitTerm
    (
        tCIDLib::EGlobalState_After
        , eInitTerm
        , modInit
        , c4InitCount
        , afnInitTerm
        , pszTitle1
        , pszTitle2))
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}
