//
// NAME: CIDLib_Module.Cpp
//
// DESCRIPTION:
//
//  TModule is the basic public class that represents a loadable module,
//  a DLL or Exe module.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/05/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TModule,TObject)


// ----------------------------------------------------------------------------
//  Local static data
//
//  __pszStrLoadFailure
//      This is the default string that is used if a string load fails. It
//      is loaded during init. The default value is provided here, in case
//      its needed before our module init, or if the load of our default
//      failure string fails itself.
// ----------------------------------------------------------------------------
static const tCIDLib::Tch*  __pszStrLoadFailure = L"Could not load system message. Id=%(1), Err=%(2)";



// ----------------------------------------------------------------------------
//  Functions for intrafacility use
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermMod
//
// DESCRIPTION:
//
//  This method is called from the process attach DLL init event.
// -------------------------------------
//   INPUT: eInitTerm is the init/term phase that we are in.
//          eGlobals indicates whether this is before/after constructors or
//              before/after destructors. It depends upon the eInitTerm
//              parm.
//          modInit is the module object that represents this module.
//          c4MaxChars is the max chars the failure reason buffer can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for any failure
//
//  RETURN: None
//
tCIDLib::TVoid _InitTermMod(const   tCIDLib::EInitTerm      eInitTerm
                            , const tCIDLib::EGlobalStates  eGlobals
                            , const TModule&                modInit
                            , const tCIDLib::TCard4         c4MaxChars
                            ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = L"Uknown";

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            const tCIDLib::TCard4   c4BufMax = 512;

            TModule::__pmtxLogSync = 0;
            TModule::__plgrTarget = 0;

            tCIDLib::TBoolean       bGotSem = kCIDLib::False;
            tCIDLib::Tch            szFileName[c4BufMax+1];
            tCIDLib::Tch            szMutexName[c4BufMax+1];
            tCIDLib::Tch*           pszMutexName = 0;

            //
            //  Load up the default string that is used if there is a
            //  failure to load a string. If it loads, then override the
            //  default string.
            //
            const tCIDLib::Tch* pszDefStr = modInit.pszLoadCIDMsg
            (
                kCIDMsgs::midMod_DefStrLoad
            );

            if (pszDefStr)
                __pszStrLoadFailure = pszDefStr;

            //
            //  We need to check for a path to a local log file and a sync
            //  mutex. If the env variable CIDLOCALLOG= is found, then the
            //  log file is created there. If CIDLOGSEM= is found, then
            //  a sync semaphore will be used.
            //
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMod_LogSemLookup);
            if (TKrnlEnvironment::bFind(L"CIDLOGSEM", szMutexName, c4BufMax))
                pszMutexName = szMutexName;

            if (TKrnlEnvironment::bFind(L"CIDLOCALLOG", szFileName, c4BufMax))
            {
                //
                //  There is a log file name, so lets create a logger that
                //  logs to that file. Create a resource name object for the
                //  mutex if its to be named.
                //
                if (pszMutexName)
                {
                    TModule::__plgrTarget = new TTextFileLogger
                    (
                       szFileName
                       , tCIDLib::ETextFmt_UNICode
                       , TResourceName(L"CIDCorp", L"CIDLib", szMutexName)
                    );
                }
                 else
                {
                    TModule::__plgrTarget = new TTextFileLogger(szFileName);
                }
            }

            //
            //  If we did not get a logger set yet, then we will create a
            //  default one that logs to redirectable console.
            //
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMod_DefLoggerInit);
            if (!TModule::__plgrTarget)
                TModule::__plgrTarget = new TStdOutLogger();

            //
            //  Set the name of the internal log sync mutex object and then
            //  create it. This one is unnamed and is just for within this
            //  process.
            //
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMod_LogSyncMtxCreate);
            TModule::__pmtxLogSync = new TMutex(tCIDLib::ELockState_Unlocked);
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            // Close down the logging stuff
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMod_LoggerDelete);
            delete TModule::__plgrTarget;
            TModule::__plgrTarget = 0;

            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMod_LogSyncMtxDelete);
            delete TModule::__pmtxLogSync;
            TModule::__pmtxLogSync = 0;
        }
    }

    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}


// ----------------------------------------------------------------------------
//   CLASS: TModule
//  PREFIX: mod
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TModule: Static data members
// ----------------------------------------------------------------------------
tCIDLib::ESeverities    TModule::__eSevThreshold = tCIDLib::ESev_APIFailed;
TLogger*                TModule::__plgrTarget = 0;
TMutex*                 TModule::__pmtxLogSync = 0;


// ----------------------------------------------------------------------------
//  TModule: Constructors and destructors
// ----------------------------------------------------------------------------

TModule::TModule(   const   TString&                strName
                    , const tCIDLib::EModTypes      eModType
                    , const tCIDLib::TBoolean       bLoad) :

    __eModType(eModType)
    , __kmodThis(strName.pszData(), eModType, bLoad)
    , __strName(strName)
{
    //
    //  Init the module handle data member. We could have done this in
    //  the __kmodThis contructor, but we would not have been able to
    //  trap the exception and translate it to a TError.
    //
    try
    {
        __kmodThis.QueryFromName(strName.pszData(), __eModType);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMod_BadName
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Unknown
            , __strName
        );
    }
}

TModule::TModule(   const   tCIDLib::TModHandle     hmodToUse
                    , const tCIDLib::EModTypes      eModType
                    , const tCIDLib::EAdoptOpts     eAdopt
                    , const tCIDLib::TBoolean       bViaLoad) :

    __eModType(eModType)
    , __kmodThis(hmodToUse, eAdopt, bViaLoad)
{
    //
    //  Set up or implementation object to adopt the handle. We could
    //  have done this in the constructor of __kmodThis, but then we
    //  would hot be able to catch the exception and translate it.
    //
    try
    {
        // Get the name of module
        tCIDLib::Tch szTmp[kCIDLib::c4MaxPathLen+1];
        __kmodThis.RawQueryModName(hmodToUse, szTmp, kCIDLib::c4MaxPathLen);
        __strName = szTmp;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMod_QueryName
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}

TModule::~TModule()
{
}


// ----------------------------------------------------------------------------
//  TModule: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean
TModule::bLoadOSMsg(const tCIDLib::TOSErrCode errcId, TString& strMsgBuf)
{
    tCIDLib::TCard4 c4Len = 4096;
    tCIDLib::Tch    szTmp[4096+1];

    try
    {
        TKrnlSysInfo::LoadOSMsg(errcId, szTmp, c4Len);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        strMsgBuf = __pszStrLoadFailure;
        strMsgBuf.ReplaceToken(TCardinal(errcId), L'1');
        strMsgBuf.ReplaceToken(TCardinal(kerrToCatch.errcId()), L'2');
        return kCIDLib::False;
    }

    // Copy the text back to the string
    strMsgBuf = szTmp;

    return kCIDLib::True;
}


tCIDLib::ESeverities 
TModule::eSevThreshold(const tCIDLib::ESeverities eSevToSet)
{
    TMtxLock mtxlLog(TModule::__pmtxLogSync);

    __eSevThreshold = eSevToSet;
    
    if (eSevToSet == tCIDLib::ESev_Status)
        __eSevThreshold = tCIDLib::ESev_Information;

    return __eSevThreshold;
}


tCIDLib::TVoid TModule::InstallLogger(TLogger* const plgrToAdopt)
{
    TMtxLock mtxlLog(TModule::__pmtxLogSync);

    // Clean up any existing stuff
    delete __plgrTarget;
    __plgrTarget = 0;

    // And store the new one
    __plgrTarget = plgrToAdopt;
}




// ----------------------------------------------------------------------------
//  TModule: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TModule::LoadCIDMsg(const   tCIDLib::TMsgId midToLoad
                    ,       TString&        strMsgBuf) const
{
    const tCIDLib::Tch* pszMsg = __kmodThis.pszLoadCIDMsg(midToLoad);
    strMsgBuf = pszMsg;
}

tCIDLib::TVoid
TModule::LoadCIDMsg(const   tCIDLib::TMsgId     midToLoad
                    ,       TString&            strMsgBuf
                    , const MFormattable&       fmtblToken1
                    , const MFormattable&       fmtblToken2
                    , const MFormattable&       fmtblToken3
                    , const MFormattable&       fmtblToken4) const
{
    // Call the other version to load up the string
    LoadCIDMsg(midToLoad, strMsgBuf);

    // And do the token replacements
    if (&fmtblToken1)
        strMsgBuf.ReplaceToken(fmtblToken1, L'1');
    if (&fmtblToken2)
        strMsgBuf.ReplaceToken(fmtblToken2, L'2');
    if (&fmtblToken3)
        strMsgBuf.ReplaceToken(fmtblToken3, L'3');
    if (&fmtblToken4)
        strMsgBuf.ReplaceToken(fmtblToken4, L'4');
}


tCIDLib::TVoid
TModule::LoadResString( const   tCIDLib::TResId ridToLoad
                        ,       TString&        strMsgBuf) const
{
    //
    //  Create a temp buffer to load the message into. We ask how big the
    //  string and resize the string up if needed.
    //
    tCIDLib::TCard4 c4Size;
    QueryResourceSize(tCIDLib::ERscType_String, ridToLoad, c4Size);

    // See if we need to reallocate the string
    if (c4Size > strMsgBuf.c4BufChars())
    {
        //
        //  If this string reports overflows as errors, then just take our
        //  chances and reallocate to the string size. Otherwise take the
        //  smaller of the string size or the max string realloc size.
        //
        if (strMsgBuf.bErrOnOverflow())
            strMsgBuf.ReallocateTo(c4Size);
        else
            strMsgBuf.ReallocateTo(MinVal(c4Size, strMsgBuf.c4MaxChars()));
    }
    
    // Now we can load the message. This guy does not throw exceptions!
    __kmodThis.bLoadResString
    (
        ridToLoad
        , strMsgBuf.pszData()
        , strMsgBuf.c4BufChars()
    );
}

tCIDLib::TVoid
TModule::LoadResString( const   tCIDLib::TResId ridToLoad
                        ,       TString&        strMsgBuf
                        , const MFormattable&   fmtblToken1
                        , const MFormattable&   fmtblToken2
                        , const MFormattable&   fmtblToken3
                        , const MFormattable&   fmtblToken4) const
{
    // Call the other version to load up the string
    LoadResString(ridToLoad, strMsgBuf);

    // And do the token replacements
    if (&fmtblToken1)
        strMsgBuf.ReplaceToken(fmtblToken1, L'1');
    if (&fmtblToken2)
        strMsgBuf.ReplaceToken(fmtblToken2, L'2');
    if (&fmtblToken3)
        strMsgBuf.ReplaceToken(fmtblToken3, L'3');
    if (&fmtblToken4)
        strMsgBuf.ReplaceToken(fmtblToken4, L'4');
}


tCIDLib::TVoid
TModule::LogErr(const   TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const tCIDLib::TErrCode       errcToLog
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass) const
{
    TString  strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg(errcToLog, strTmp);

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToLog
        , strTmp
        , kCIDLib::pszEmptyZStr
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}

tCIDLib::TVoid
TModule::LogErr(const   TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const tCIDLib::TErrCode       errcToLog
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass
                , const MFormattable&           fmtblToken1
                , const MFormattable&           fmtblToken2
                , const MFormattable&           fmtblToken3
                , const MFormattable&           fmtblToken4) const
{
    TString strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg
    (
        errcToLog
        , strTmp
        , fmtblToken1
        , fmtblToken2
        , fmtblToken3
        , fmtblToken4
    );

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToLog
        , strTmp
        , kCIDLib::pszEmptyZStr
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}

tCIDLib::TVoid
TModule::LogErr(const   TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const tCIDLib::TErrCode       errcToLog
                , const TString&                strAuxText
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass) const
{
    TString strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg(errcToLog, strTmp);

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToLog
        , strTmp
        , strAuxText
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}

tCIDLib::TVoid
TModule::LogErr(const   TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const tCIDLib::TErrCode       errcToLog
                , const TString&                strAuxText
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass
                , const MFormattable&           fmtblToken1
                , const MFormattable&           fmtblToken2
                , const MFormattable&           fmtblToken3
                , const MFormattable&           fmtblToken4) const
{
    TString strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg
    (
        errcToLog
        , strTmp
        , fmtblToken1
        , fmtblToken2
        , fmtblToken3
        , fmtblToken4
    );

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToLog
        , strTmp
        , strAuxText
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}


tCIDLib::TVoid TModule::LogErrObj(const TError& errToLog) const
{
    //
    //  We use a static flag to catch nested errors and freak out because
    //  this is likely to get us into a stack overflow.
    //
    static TKrnlSafeCounter scntNested(-1);

    // Check for a nested error scenario that's just looping
    if (scntNested.i4Value() > 16)
    {
        _MSGPOPUP_
        (
            L"TModule::LogErrObj: A circular error logging sitation has occured"
        )
    }

    // Get the severity out since its used a lot below
    tCIDLib::ESeverities eSev = errToLog.eSeverity();

    try
    {
        // Bump up the count of warning messages if we get one
        if (eSev == tCIDLib::ESev_Warning)
            _pmtrgCore->IncCounter(tCIDLib_::eCoreMetric_WarningMsgs);

        if (eSev == tCIDLib::ESev_APIFailed)
            _pmtrgCore->IncCounter(tCIDLib_::eCoreMetric_FailureMsgs);

        if (eSev == tCIDLib::ESev_ProcessFatal)
            _pmtrgCore->IncCounter(tCIDLib_::eCoreMetric_FatalMsgs);

        //
        //  If it is above the logging threshold, then we need to log it.
        //
        if (eSev >= __eSevThreshold)
        {
            // Get control of the sync semaphore before we do the output
            TMtxLock mtxlLog(TModule::__pmtxLogSync);
            __plgrTarget->LogMsg(errToLog);
        }

        //
        //  If a status message, then we are done. Otherwise, this would be
        //  interpreted as a fatal error below and be thrown.
        //
        if (eSev == tCIDLib::ESev_Status)
            return;

        //
        //  Dump out the info as a runtime error if it is a fatal error.
        //
        if (eSev >= tCIDLib::ESev_ProcessFatal)
        {
            const TThread* const pthrTmp = TThread::pthrCaller();
            TKrnlThread::DumpRuntimeError
            (
                pthrTmp->kthrThis()
                , pthrTmp->strName().pszData()
                , errToLog.strModName().pszData()
                , errToLog.strErrText().pszData()
                , errToLog.strAuxText().pszData()
                , errToLog.errcId()
                , errToLog.errcKrnlId()
                , errToLog.errcHostId()
                , errToLog.strFileName().pszData()
                , errToLog.c4LineNum()
            );
        }
    }

    //
    //  Do a freak out error popup, but don't rethrow. Just fall through
    //  and continue on.
    //
    catch(...)
    {
        _MSGPOPUP_
        (
            L"TModule::LogErrObj: An exception occured while logging an error"
        )
    }

    // Bump the counter back down now
    scntNested.bDec();

    //
    //  If its at or above the process fatal level, then we need to exit the
    //  program now. We use the standard runtime error code.
    //
    //  If we are in testing mode, then we don't exit since the host process
    //  is just probing to insure that errors are caught and wants to continue
    //  processing.
    //
    if ((eSev >= tCIDLib::ESev_ProcessFatal) && !TSysInfo::bTestMode())
        TFacCIDLib::ExitProcess(tCIDLib::EExit_RuntimeError);

    // Throw it if it is greater than or equal to API_Failed.
    if (eSev >= tCIDLib::ESev_APIFailed)
        throw errToLog;
}


tCIDLib::TVoid
TModule::LogMsg(const   TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const TString&                strMsg
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass) const
{
    TError  errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , strMsg
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}

tCIDLib::TVoid
TModule::LogMsg(const   TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const TString&                strMsg
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass
                , const MFormattable&           fmtblToken1
                , const MFormattable&           fmtblToken2
                , const MFormattable&           fmtblToken3
                , const MFormattable&           fmtblToken4) const
{
    TString strTmp(strMsg.pszData(), 256);

    // Do parameter replacement
    if (&fmtblToken1)
        strTmp.ReplaceToken(fmtblToken1, L'1');
    if (&fmtblToken2)
        strTmp.ReplaceToken(fmtblToken2, L'2');
    if (&fmtblToken3)
        strTmp.ReplaceToken(fmtblToken3, L'3');
    if (&fmtblToken4)
        strTmp.ReplaceToken(fmtblToken4, L'4');

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , strTmp
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}


tCIDLib::TVoid
TModule::ThrowMsg(  const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const TString&                strMsg
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass) const
{
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , strMsg
        , eSeverity
        , eClass
    );
    throw errToThrow;
}

tCIDLib::TVoid
TModule::ThrowMsg(  const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const TString&                strMsg
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass
                    , const MFormattable&           fmtblToken1
                    , const MFormattable&           fmtblToken2
                    , const MFormattable&           fmtblToken3
                    , const MFormattable&           fmtblToken4) const
{
    TString  strTmp(strMsg.pszData(), 128);

    // Do parameter replacement
    if (&fmtblToken1)
        strTmp.ReplaceToken(fmtblToken1, L'1');
    if (&fmtblToken2)
        strTmp.ReplaceToken(fmtblToken2, L'2');
    if (&fmtblToken3)
        strTmp.ReplaceToken(fmtblToken3, L'3');
    if (&fmtblToken4)
        strTmp.ReplaceToken(fmtblToken4, L'4');

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , strTmp
        , eSeverity
        , eClass
    );
    throw errToThrow;
}


tCIDLib::TVoid
TModule::LogKrnlErr(const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const tCIDLib::TErrCode       errcToLog
                    , const TKrnlError&             kerrToLog
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass) const
{
    TString strKrnlMsg(kCIDLib::pszEmptyZStr, 128);
    TString strMsg(kCIDLib::pszEmptyZStr, 128);

    //
    //  Load up the nost OS message, if this error was caused by an underlying
    //  host OS error. Note that it sets the string to a default message if
    //  this fails, so we don't do anything if so.
    //
    if (kerrToLog.errcHostId())
        bLoadOSMsg(kerrToLog.errcHostId(), strKrnlMsg);

    // Load up the CIDLib message
    LoadCIDMsg(errcToLog, strMsg);

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToLog
        , kerrToLog
        , strMsg
        , strKrnlMsg
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}

tCIDLib::TVoid
TModule::LogKrnlErr(const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const tCIDLib::TErrCode       errcToLog
                    , const TKrnlError&             kerrToLog
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass
                    , const MFormattable&           fmtblToken1
                    , const MFormattable&           fmtblToken2
                    , const MFormattable&           fmtblToken3
                    , const MFormattable&           fmtblToken4) const
{
    TString strKrnlMsg(kCIDLib::pszEmptyZStr, 128);
    TString strText(kCIDLib::pszEmptyZStr, 256);

    //
    //  Load up the host OS message, if this was caused by an underlying
    //  host error. Note that it sets the string to a default message if
    //  this fails, so we don't do anything if so.
    //
    if (kerrToLog.errcHostId())
        bLoadOSMsg(kerrToLog.errcHostId(), strKrnlMsg);

    // Load the CIDLib message and do token replacment
    LoadCIDMsg
    (
        errcToLog
        , strText
        , fmtblToken1
        , fmtblToken2
        , fmtblToken3
        , fmtblToken4
    );

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToLog
        , kerrToLog
        , strText
        , strKrnlMsg
        , eSeverity
        , eClass
    );

    // Now log that bad boy
    LogErrObj(errToThrow);
}


const tCIDLib::Tch*
TModule::pszLoadCIDMsg( const   tCIDLib::TMsgId     midToLoad
                        ,       tCIDLib::TBoolean&  bLoaded) const
{
    // Note that this guy never throws an exception!
    return __kmodThis.pszLoadCIDMsg(midToLoad, bLoaded);
}


const tCIDLib::Tch*
TModule::pszLoadResString(  const   tCIDLib::TResId     ridToLoad
                            ,       tCIDLib::TBoolean&  bLoaded) const
{
    // Note that this guy never throws an exception!
    return __kmodThis.pszLoadResString(ridToLoad, bLoaded);
}


tCIDLib::TVoid
TModule::QueryResourceSize( const   tCIDLib::ERscTypes  eRscType
                            , const tCIDLib::TResId     ridToQuery
                            ,       tCIDLib::TCard4&    c4Size) const
{
    try
    {
        c4Size = __kmodThis.c4QueryResourceSize(eRscType, ridToQuery);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMod_ResQuerySize
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TInteger(eRscType)
            , TCardinal(ridToQuery)
            , __strName
        );
    }
}


TString TModule::strMsg(const tCIDLib::TMsgId midToLoad)
{
    TString strRet(kCIDLib::pszEmptyZStr, 1024);
    LoadCIDMsg(midToLoad, strRet);
    return strRet;
}

TString TModule::strMsg(const   tCIDLib::TMsgId midToLoad
                        , const MFormattable&   fmtblToken1
                        , const MFormattable&   fmtblToken2
                        , const MFormattable&   fmtblToken3
                        , const MFormattable&   fmtblToken4)
{
    TString strRet(kCIDLib::pszEmptyZStr, 1024);
    LoadCIDMsg
    (
        midToLoad
        , strRet
        , fmtblToken1
        , fmtblToken2
        , fmtblToken3
        , fmtblToken4
    );
    return strRet;
}


tCIDLib::TVoid
TModule::ThrowErr(const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const tCIDLib::TErrCode       errcToThrow
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass) const
{
    TString  strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg(errcToThrow, strTmp);

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToThrow
        , strTmp
        , kCIDLib::pszEmptyZStr
        , eSeverity
        , eClass
    );
    throw errToThrow;
}

tCIDLib::TVoid
TModule::ThrowErr(const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const tCIDLib::TErrCode       errcToThrow
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass
                    , const MFormattable&           fmtblToken1
                    , const MFormattable&           fmtblToken2
                    , const MFormattable&           fmtblToken3
                    , const MFormattable&           fmtblToken4) const
{
    TString  strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg
    (
        errcToThrow
        , strTmp
        , fmtblToken1
        , fmtblToken2
        , fmtblToken3
        , fmtblToken4
    );

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToThrow
        , strTmp
        , kCIDLib::pszEmptyZStr
        , eSeverity
        , eClass
    );
    throw errToThrow;
}

tCIDLib::TVoid
TModule::ThrowErr(const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const tCIDLib::TErrCode       errcToThrow
                    , const TString&                strAuxText
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass) const
{
    TString  strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg(errcToThrow, strTmp);

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToThrow
        , strTmp
        , strAuxText
        , eSeverity
        , eClass
    );
    throw errToThrow;
}

tCIDLib::TVoid
TModule::ThrowErr(const   TString&                strFileName
                    , const tCIDLib::TCard4         c4LineNum
                    , const tCIDLib::TErrCode       errcToThrow
                    , const TString&                strAuxText
                    , const tCIDLib::ESeverities    eSeverity
                    , const tCIDLib::EErrClasses    eClass
                    , const MFormattable&           fmtblToken1
                    , const MFormattable&           fmtblToken2
                    , const MFormattable&           fmtblToken3
                    , const MFormattable&           fmtblToken4) const
{
    TString  strTmp(kCIDLib::pszEmptyZStr, 128);

    // Load up the message
    LoadCIDMsg
    (
        errcToThrow
        , strTmp
        , fmtblToken1
        , fmtblToken2
        , fmtblToken3
        , fmtblToken4
    );

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToThrow
        , strTmp
        , strAuxText
        , eSeverity
        , eClass
    );
    throw errToThrow;
}


tCIDLib::TVoid
TModule::ThrowKrnlErr(  const   TString&                strFileName
                        , const tCIDLib::TCard4         c4LineNum
                        , const tCIDLib::TErrCode       errcToThrow
                        , const TKrnlError&             kerrToThrow
                        , const tCIDLib::ESeverities    eSeverity
                        , const tCIDLib::EErrClasses    eClass) const
{
    TString strKrnlMsg(kCIDLib::pszEmptyZStr, 128);
    TString strMsg(kCIDLib::pszEmptyZStr, 128);

    //
    //  Load up the host OS message, if this was caused by an underlying
    //  host error. Note that it sets the string to a default message if
    //  this fails, so we don't do anything if so.
    //
    if (kerrToThrow.errcHostId())
        bLoadOSMsg(kerrToThrow.errcHostId(), strKrnlMsg);

    // Load the CIDLib error message
    LoadCIDMsg(errcToThrow, strMsg);

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToThrow
        , kerrToThrow
        , strMsg
        , strKrnlMsg
        , eSeverity
        , eClass
    );

    throw errToThrow;
}

tCIDLib::TVoid
TModule::ThrowKrnlErr(  const   TString&                strFileName
                        , const tCIDLib::TCard4         c4LineNum
                        , const tCIDLib::TErrCode       errcToThrow
                        , const TKrnlError&             kerrToThrow
                        , const tCIDLib::ESeverities    eSeverity
                        , const tCIDLib::EErrClasses    eClass
                        , const MFormattable&           fmtblToken1
                        , const MFormattable&           fmtblToken2
                        , const MFormattable&           fmtblToken3
                        , const MFormattable&           fmtblToken4) const
{
    TString strKrnlMsg(kCIDLib::pszEmptyZStr, 128);
    TString strText(kCIDLib::pszEmptyZStr, 256);

    //
    //  Load up the host OS message, if this was caused by an underlying
    //  host error. Note that it sets the string to a default message if
    //  this fails, so we don't do anything if so.
    //
    if (kerrToThrow.errcHostId())
        bLoadOSMsg(kerrToThrow.errcHostId(), strKrnlMsg);

    // Get a copy of the err text and do the token replacement
    LoadCIDMsg
    (
        errcToThrow
        , strText
        , fmtblToken1
        , fmtblToken2
        , fmtblToken3
        , fmtblToken4
    );

    // Create the error object
    TError errToThrow
    (
        __strName
        , strFileName
        , c4LineNum
        , errcToThrow
        , kerrToThrow
        , strText
        , strKrnlMsg
        , eSeverity
        , eClass
    );
    throw errToThrow;
}
