//
// NAME: CIDLib_Metrics.Cpp
// 
// DESCRIPTION: 
// 
//  This module contains the metrics support for the client program and for
//  the CIDLib system itself. This system is just an interface to the
//  metrics implementation in the CIDKernel facility.
//
// 
// AUTHOR: Dean Roddey
// 
// CREATE DATE: 01/11/94
// 
// COPYRIGHT: 1992..1997, 'CIDCorp
// 
// CAVEATS/GOTCHAS: 
// 

// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TMetricGroup,TObject)
RTTIData(TMetricSystem,TObject)


// ----------------------------------------------------------------------------
//  Intrafacility data
//
//  _pmtrgCore
//      This is the core metric group that is used to track metrics for the
//      CIDLib facility.
// ----------------------------------------------------------------------------
TMetricGroup* _pmtrgCore;


// ----------------------------------------------------------------------------
//  Local static data
//
//  __pmtxMetricDir
//      This is our own mutex to use to lock the metrics directory. The
//      kernel has a TKrnlMutex which it uses, but we need to use a TMutex
//      for exception reasons. Since its named, we can just open our own
//      copy.
// ----------------------------------------------------------------------------
static TMutex*      __pmtxMetricDir;


// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermMetrics
//
// DESCRIPTION:
//
//  This is called from the DLL init and it sets 
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modInit is a temp module object for this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid
_InitTermMetrics(   const   tCIDLib::EInitTerm      eInitTerm
                    , const tCIDLib::EGlobalStates  eGlobals
                    , const TModule&                modInit
                    , const tCIDLib::TCard4         c4MaxChars
                    ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            // Create our own copy of the metrics directory mutex
            TResourceName rsnMetrics
            (
                L"CIDCorp"
                , L"Metrics"
                , L"Directory"
                , TKrnlSysInfo::pidThis()
            );

            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_MutexCreate);
            __pmtxMetricDir = new TMutex(rsnMetrics);

            //
            //  Create the core metrics group. It is declared and exported to all
            //  other facility modules that have metrics include, in the
            //  CIDLib_.Hpp header.
            //
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_CoreGroupInit);
            TMetricSystem::TMetricInfo amtriCoreInfo[tCIDLib_::eCoreMetric_Count];

            // Loop through the metrics and fill them in
            tCIDLib::TCard4 c4Index;
            for (c4Index = 0; c4Index < tCIDLib_::eCoreMetric_Count; c4Index++)
            {
                TMetricSystem::TMetricInfo* pCur = &amtriCoreInfo[c4Index];

                if (c4Index == tCIDLib_::eCoreMetric_Verbose)
                {
                    pCur->strMetricName = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_Verbose);
                    pCur->strMetricHelp = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_VerboseHelp);
                    pCur->eType = tCIDLib::EMetric_Toggle;
                }
                 else if (c4Index == tCIDLib_::eCoreMetric_ThreadCnt)
                {
                    pCur->strMetricName = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_ThreadCount);
                    pCur->strMetricHelp = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_ThreadCountHelp);
                    pCur->eType = tCIDLib::EMetric_Cardinal;
                }
                 else if (c4Index == tCIDLib_::eCoreMetric_WarningMsgs)
                {
                    pCur->strMetricName = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_WarningMsgs);
                    pCur->strMetricHelp = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_WarningMsgsHelp);
                    pCur->eType = tCIDLib::EMetric_TickCounter;
                }
                 else if (c4Index == tCIDLib_::eCoreMetric_FailureMsgs)
                {
                    pCur->strMetricName = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_FailMsgs);
                    pCur->strMetricHelp = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_FailMsgsHelp);
                    pCur->eType = tCIDLib::EMetric_TickCounter;
                }
                 else if (c4Index == tCIDLib_::eCoreMetric_FatalMsgs)
                {
                    pCur->strMetricName = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_FatalMsgs);
                    pCur->strMetricHelp = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_FatalMsgsHelp);
                    pCur->eType = tCIDLib::EMetric_TickCounter;
                }
                 else if (c4Index == tCIDLib_::eCoreMetric_ObjCount)
                {
                    pCur->strMetricName = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_ObjCount);
                    pCur->strMetricHelp = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_ObjCountHelp);
                    pCur->eType = tCIDLib::EMetric_Cardinal;
                }
                 else if (c4Index == tCIDLib_::eCoreMetric_OpenFiles)
                {
                    pCur->strMetricName = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_OpenedFiles);
                    pCur->strMetricHelp = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_OpenedFilesHelp);
                    pCur->eType = tCIDLib::EMetric_Cardinal;
                }
                pCur++;
            }

            //
            //  Register the new group and save away the pointer for internal
            //  use.
            //
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_CoreGroupRegister);
            _pmtrgCore = TMetricSystem::pmtrgRegisterGroup
            (
                modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_CoreGroup)
                , modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_CoreGroupHelp)
                , amtriCoreInfo
                , tCIDLib_::eCoreMetric_Count
            );
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midMtr_CoreGroupDelete);
            delete _pmtrgCore;
            _pmtrgCore = 0;
        }
    }

    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TMetricGroup
//  PREFIX: mtrg
//
//  This class represents a group of metrics.
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TMetricGroup: Public, non-virtual methods
// ----------------------------------------------------------------------------

#if CID_DEBUG_ON
#define CHECKMETRICIND(mind)   \
if (mind >= __pmtrgData->c4MetricCount) \
{ \
    facCIDLib.LogErr \
    ( \
        __FILE__ \
        , __LINE__ \
        , kCIDErrs::errcMetric_BadMetricId \
        , tCIDLib::ESev_ProcessFatal \
        , tCIDLib::EClass_BadParms \
        , TString(__pmtrgData->szGroupName)  \
        , TCardinal(__pmtrgData->c4MetricCount)\
        , TCardinal(mind) \
    ); \
}
#else
#define CHECKMETRICIND(mind)
#endif

#if     CID_DEBUG_ON
#define CHECKMETRICTYPE(pmetr,type) \
if ((pmetr)->eType != type) \
{ \
    TStringStream  strmType(64); \
    TStringStream  strmTargetType(64); \
    strmType << (pmetr)->eType; \
    strmTargetType << tCIDLib::EMetricTypes(type); \
\
    facCIDLib.LogErr \
    ( \
        __FILE__ \
        , __LINE__ \
        , kCIDErrs::errcMetric_WrongAccess \
        , tCIDLib::ESev_ProcessFatal \
        , tCIDLib::EClass_BadParms \
        , TString(__pmtrgData->szGroupName) \
        , TString((pmetr)->szMetricName) \
        , strmType.strData() \
        , strmTargetType.strData()); \
}
#else
#define CHECKMETRICTYPE(pmetr,type)
#endif


tCIDLib::TVoid
TMetricGroup::AddToAccum(   const   tCIDLib::TCard4 c4MetricInd
                            , const tCIDLib::TCard4 c4Add)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    // Get a pointer to the metric for speed and textual brevity
    tCIDLib::TRawMetric* const pMetric = &__pmtrgData->amtrMetrics[c4MetricInd];

    //
    //  Make sure this is an accumulator. If not, then unlock and abort.
    //
    CHECKMETRICTYPE(pMetric, tCIDLib::EMetric_Accum)

    //  Update the metric
    pMetric->f4FValue += 1.0;
}


tCIDLib::TVoid
TMetricGroup::FlipToggle(const tCIDLib::TCard4 c4MetricInd)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    // Get a pointer to the metric for speed and textual brevity
    tCIDLib::TRawMetric* const pMetric = &__pmtrgData->amtrMetrics[c4MetricInd];

    //
    //  Make sure this is a toggle metric. If not, then unlock and abort.
    //
    CHECKMETRICTYPE(pMetric, tCIDLib::EMetric_Toggle)

    if (pMetric->bToggle)
        pMetric->bToggle = kCIDLib::False;
     else
        pMetric->bToggle = kCIDLib::True;
}


tCIDLib::TVoid
TMetricGroup::IncCounter(const tCIDLib::TCard4 c4MetricInd)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    // Get a pointer to the metric for speed and textual brevity
    tCIDLib::TRawMetric* const pMetric = &__pmtrgData->amtrMetrics[c4MetricInd];

    //
    //  Make sure this is a tick counter metric. If not, then unlock and
    //  abort.
    //
    CHECKMETRICTYPE(pMetric, tCIDLib::EMetric_TickCounter)

    pMetric->f4FValue += 1.0;
}


tCIDLib::TVoid
TMetricGroup::OffsetCard(   const   tCIDLib::TCard4 c4MetricInd
                            , const tCIDLib::TInt4  i4Offset)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    //
    //  Make sure this is a card value metric. If not, then unlock and
    //  abort.
    //
    CHECKMETRICTYPE
    (
        &__pmtrgData->amtrMetrics[c4MetricInd]
        , tCIDLib::EMetric_Cardinal
    )

    __pmtrgData->amtrMetrics[c4MetricInd].c4CValue += i4Offset;
}


tCIDLib::TVoid
TMetricGroup::OffsetInt(const   tCIDLib::TCard4 c4MetricInd
                        , const tCIDLib::TInt4  i4Offset)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    //
    //  Make sure this is a int value metric. If not, then unlock and
    //  abort.
    //
    CHECKMETRICTYPE
    (
        &__pmtrgData->amtrMetrics[c4MetricInd]
        , tCIDLib::EMetric_Integer
    )

    __pmtrgData->amtrMetrics[c4MetricInd].i4IValue += i4Offset;
}


tCIDLib::TVoid
TMetricGroup::SetCard(  const   tCIDLib::TCard4 c4MetricInd
                        , const tCIDLib::TCard4 c4NewValue)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    //
    //  Make sure this is a card value metric. If not, then unlock and
    //  abort.
    //
    CHECKMETRICTYPE
    (
        &__pmtrgData->amtrMetrics[c4MetricInd]
        , tCIDLib::EMetric_Cardinal
    )

    __pmtrgData->amtrMetrics[c4MetricInd].c4CValue = c4NewValue;
}


tCIDLib::TVoid
TMetricGroup::SetFlag(  const   tCIDLib::TCard4     c4MetricInd
                        , const tCIDLib::TCard4     c4BitNum
                        , const tCIDLib::TBoolean   bNewState)
{
    // Check the bit number of we are debugging
    #if CID_DEBUG_ON
    if (c4BitNum > 31)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMetric_BadBitIndex
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4BitNum)
        );
    }
    #endif

    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    // Get a pointer to the metric for speed and textual brevity
    tCIDLib::TRawMetric* const pMetric = &__pmtrgData->amtrMetrics[c4MetricInd];

    //
    //  Make sure this is a flag set value metric. If not, then unlock and
    //  abort.
    //
    CHECKMETRICTYPE(pMetric, tCIDLib::EMetric_FlagSet)

    if (bNewState)
        pMetric->c4CValue |= (0x1L << c4BitNum);
     else
        pMetric->c4CValue &= ~(0x1L << c4BitNum);
}


tCIDLib::TVoid
TMetricGroup::SetFlags( const   tCIDLib::TCard4 c4MetricInd
                        , const tCIDLib::TCard4 c4TargetMask
                        , const tCIDLib::TCard4 c4ValueMask)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    //
    //  Make sure this is a flag set value metric. If not, then unlock and
    //  abort.
    //
    CHECKMETRICTYPE
    (
        &__pmtrgData->amtrMetrics[c4MetricInd]
        , tCIDLib::EMetric_FlagSet
    )

    // Turn off all of the target bits
    __pmtrgData->amtrMetrics[c4MetricInd].c4CValue &= ~c4TargetMask;

    // And turn on the ones that are on in the value mask
    __pmtrgData->amtrMetrics[c4MetricInd].c4CValue |= c4ValueMask;
}


tCIDLib::TVoid
TMetricGroup::SetInt(   const   tCIDLib::TCard4 c4MetricInd
                        , const tCIDLib::TInt4  i4NewValue)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    //
    //  Make sure this is a signed value metric. If not, then unlock and
    //  abort.
    //
    CHECKMETRICTYPE
    (
        &__pmtrgData->amtrMetrics[c4MetricInd]
        , tCIDLib::EMetric_Integer
    )

    __pmtrgData->amtrMetrics[c4MetricInd].i4IValue = i4NewValue;
}


tCIDLib::TVoid
TMetricGroup::SetPeak(  const   tCIDLib::TCard4 c4MetricInd
                        , const tCIDLib::TCard4 c4NewValue)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    // Make sure this is a toggle metric. If not, then unlock and abort.
    CHECKMETRICTYPE
    (
        &__pmtrgData->amtrMetrics[c4MetricInd]
        , tCIDLib::EMetric_PeakMeter
    )

    if (c4NewValue > __pmtrgData->amtrMetrics[c4MetricInd].c4CValue)
        __pmtrgData->amtrMetrics[c4MetricInd].c4CValue = c4NewValue;
}


tCIDLib::TVoid
TMetricGroup::SetToggle(const   tCIDLib::TCard4     c4MetricInd
                        , const tCIDLib::TBoolean   bNewValue)
{
    TMtxLock mtxlGroup(__pmtxMetricDir, 5 Seconds);

    //
    //  If we are debugging, then make sure that the passed metrics index
    //  is within the valid range.
    //
    CHECKMETRICIND(c4MetricInd)

    // Make sure this is a toggle metric. If not, then unlock and abort.
    CHECKMETRICTYPE
    (
        &__pmtrgData->amtrMetrics[c4MetricInd]
        , tCIDLib::EMetric_Toggle
    )

    __pmtrgData->amtrMetrics[c4MetricInd].bToggle = bNewValue;
}


// ----------------------------------------------------------------------------
//  TMetricGroup: Hidden constructors and operators
// ----------------------------------------------------------------------------

TMetricGroup::TMetricGroup( const tCIDLib::TCard4                 c4Count
                            ,     tCIDLib::TRawMetricGroup* const pkmtrgData) :
    __c4MetricCount(c4Count)
    , __pmtrgData(pkmtrgData)
{
}



// ----------------------------------------------------------------------------
//   CLASS: TMetricSystem
//  PREFIX: mtrs
// ----------------------------------------------------------------------------

TMetricGroup*
TMetricSystem::pmtrgRegisterGroup(  const   TString&        strGroupName
                                    , const TString&        strGroupHelp
                                    , const TMetricInfo*    amtriList
                                    , const tCIDLib::TCard4 c4Count)
{
    //
    //  We need to convert the passed metric info constructor into a
    //  the format needed by the kernel.
    //
    tCIDLib::TRawMetric kmtrgTmp[kCIDLib::c4MaxMetricsPerGrp];

    // Fill the whole thing with zeros
    TRawMem::SetMemBuf(kmtrgTmp, tCIDLib::TCard1(0), sizeof(kmtrgTmp));

    // Copy over the individual metrics
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        // Copy over this guy
        TRawStr::CopyStr
        (
            kmtrgTmp[c4Index].szMetricName
            , amtriList[c4Index].strMetricName.pszData()
            , sizeof(kmtrgTmp[c4Index].szMetricName)
        );

        TRawStr::CopyStr
        (
            kmtrgTmp[c4Index].szMetricHelp
            , amtriList[c4Index].strMetricHelp.pszData()
            , sizeof(kmtrgTmp[c4Index].szMetricHelp)
        );
        kmtrgTmp[c4Index].eType = amtriList[c4Index].eType;
    }

    //
    //  Now register the new group with the kernel and get back the
    //  raw kernel group object. We then use this to construct the
    //  CIDLib level version of a metric group and return that.
    //
    tCIDLib::TRawMetricGroup* pmtrgNew = 0;
    try
    {
        pmtrgNew = TKrnlMetricSystem::pmtrgRegisterGroup
        (
            strGroupName.pszData()
            , strGroupHelp.pszData()
            , kmtrgTmp
            , c4Count
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMetric_Register
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strGroupName
        );
    }

    return new TMetricGroup(c4Count, pmtrgNew);
}
