//
// NAME: CIDLib_MemoryStreamImpl.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_MemoryStreamImpl.Cpp module. This module
//  implements stream implementation classes that support binary and text mode
//  streaming to a memory buffer. These stream implementation classes are used
//  binary and text streams as their data sinks/sources.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 10/29/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: MMemStreamImpl
// -----------------------------------------------------------------------------
class CIDLIBEXP MMemStreamImpl
{
    public  :
        // ---------------------------------------------------------------------
        //  Public types
        // ---------------------------------------------------------------------
        enum ECheck
        {
            ECheck_ForRead
            , ECheck_ForWrite
        };

        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        MMemStreamImpl
        (
                    TMemBuf* const          pmbufToStream
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
        );

        MMemStreamImpl
        (
                    TMemBuf* const          pmbufToStream
        );

        ~MMemStreamImpl();


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        virtual tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid Reset();

        tCIDLib::TVoid TruncateAtZero();


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid AdoptBuffer();

        tCIDLib::TBoolean bBufferIsAdopted() const;

        tCIDLib::TVoid OrphanBuffer();

        TMemBuf& mbufStream();

        const TMemBuf& mbufStream() const;


    protected  :
        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _CheckCount
        (
            const   tCIDLib::TCard4         c4Line
            , const tCIDLib::TCard4         c4ToWrite
            , const tCIDLib::TCard4         c4Actual
        );

        tCIDLib::TVoid _CheckAvail
        (
            const   tCIDLib::TCard4         c4Line
            , const tCIDLib::TCard4         c4ToWrite
            , const ECheck                  eCheck
        );


        // ---------------------------------------------------------------------
        //  Protected data members
        //
        //  _c4EndIndex
        //      This is the current end of data index. We have to emulate
        //      the scenario where we pretend like only as much data is
        //      available as has been written out (even though the buffer
        //      has actually more allocated bytes.)
        //
        //  _c4Index
        //      The current index into the memory buffer, i.e. the read/write
        //      position. It can go beyond _c4EndIndex but only for a write
        //      which will 'extend' the data available.
        //
        //  _eAdopted
        //      Indicates whether we were told to adopt this buffer object
        //      and clean it up.
        //
        //  _pmbufData
        //      This is the memory buffer object to stream.
        // ---------------------------------------------------------------------
        tCIDLib::TCard4         _c4EndIndex;
        tCIDLib::TCard4         _c4Index;
        tCIDLib::EAdoptOpts     _eAdopted;
        TMemBuf*                _pmbufData;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        MMemStreamImpl(const MMemStreamImpl&);

        tCIDLib::TVoid operator=(const MMemStreamImpl&);
};



// -----------------------------------------------------------------------------
//   CLASS: TMemBinStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------
class CIDLIBEXP TMemBinStreamImpl :

    public TBinStreamImpl, public MMemStreamImpl
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TMemBinStreamImpl
        (
                    TMemBuf* const          pmbufToStream
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
         );

        ~TMemBinStreamImpl();


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        virtual tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid ReadBytes
        (
                    tCIDLib::TVoid* const   pBufToFill
            , const tCIDLib::TCard4         c4BytesToRead
            ,       tCIDLib::TCard4&        c4Actual = NUL_TCard4
        );

        tCIDLib::TVoid Reset();

        tCIDLib::TVoid TruncateAtZero();

        tCIDLib::TVoid WriteBytes
        (
            const   tCIDLib::TVoid* const   pBuffer
            , const tCIDLib::TCard4         c4BytesToWrite
            ,       tCIDLib::TCard4&        c4Actual = NUL_TCard4
        );


    protected   :
        // ---------------------------------------------------------------------
        //  Hidden constructors and operators
        // ---------------------------------------------------------------------
        TMemBinStreamImpl();

        TMemBinStreamImpl(const TMemBinStreamImpl&);

        tCIDLib::TVoid operator=(const TMemBinStreamImpl&);

    private :
        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TMemBinStreamImpl,TBinStreamImpl)
};



// -----------------------------------------------------------------------------
//   CLASS: TMemTextStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------
class CIDLIBEXP TMemTextStreamImpl :

    public TTextStreamImpl, public MMemStreamImpl
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TMemTextStreamImpl
        (
                    TMemBuf* const          pmbufToStream
            , const tCIDLib::ETextFormats   eTextFmt = tCIDLib::ETextFmt_UNICode
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
         );

        ~TMemTextStreamImpl();


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TVoid Reset();

        tCIDLib::TCard4 c4GetLine
        (
                    tCIDLib::Tch* const     pszBufToFill
            , const tCIDLib::TCard4         c4BufferSize
            , const tCIDLib::TBoolean       bStripWhitespace = kCIDLib::True
        );

        tCIDLib::Tch chGet();

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        virtual tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tch            chToWrite
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tsch           schToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tch* const     pszBufToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tsch* const    pszBufToWrite
        );

        tCIDLib::Tsch schGet();

        tCIDLib::TVoid TruncateAtZero();


    protected   :
        // ---------------------------------------------------------------------
        //  Hidden constructors and operators
        // ---------------------------------------------------------------------
        TMemTextStreamImpl();

        TMemTextStreamImpl(const TMemTextStreamImpl&);

        tCIDLib::TVoid operator=(const TMemTextStreamImpl&);


    private :
        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TMemTextStreamImpl,TTextStreamImpl)
};



#pragma pack(pop)


// -----------------------------------------------------------------------------
//  MMemStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean MMemStreamImpl::bEndOfStream() const
{
    if (_c4Index >= _c4EndIndex)
        return kCIDLib::True;
    else
        return kCIDLib::False;
}

inline tCIDLib::TFilePos MMemStreamImpl::fposCurPos() const
{
    return tCIDLib::TFilePos(_c4Index);
}

inline tCIDLib::TFilePos MMemStreamImpl::fposLogicalEnd() const
{
    return tCIDLib::TFilePos(_c4EndIndex);
}

inline tCIDLib::TFilePos MMemStreamImpl::fposPhysicalEnd() const
{
    return tCIDLib::TFilePos(_pmbufData->c4MaxSize());
}

inline tCIDLib::TFilePos MMemStreamImpl::fposSeekToEnd()
{
    _c4Index = _c4EndIndex;
    return tCIDLib::TFilePos(_c4Index);
}

inline tCIDLib::TVoid MMemStreamImpl::Reset()
{
    _c4Index = 0;
}

inline tCIDLib::TVoid MMemStreamImpl::TruncateAtZero()
{
    // Basically just set both indexes back to 0
    _c4Index = 0;
    _c4EndIndex = 0;
}



// -----------------------------------------------------------------------------
//  MMemStreamImpl: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid MMemStreamImpl::AdoptBuffer()
{
    _eAdopted = tCIDLib::EAdoptOpt_Adopt;
}

inline tCIDLib::TBoolean MMemStreamImpl::bBufferIsAdopted() const
{
    if (_eAdopted)
        return kCIDLib::True;
    else
        return kCIDLib::False;
}

inline tCIDLib::TVoid MMemStreamImpl::OrphanBuffer()
{
    _eAdopted = tCIDLib::EAdoptOpt_NoAdopt;
}

inline TMemBuf& MMemStreamImpl::mbufStream()
{
    return *_pmbufData;
}

inline const TMemBuf& MMemStreamImpl::mbufStream() const
{
    return *_pmbufData;
}


// -----------------------------------------------------------------------------
//  TMemBinStreamImpl: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TMemBinStreamImpl::bEndOfStream() const
{
    return MMemStreamImpl::bEndOfStream();
}

inline tCIDLib::TFilePos TMemBinStreamImpl::fposCurPos() const
{
    return MMemStreamImpl::fposCurPos();
}

inline tCIDLib::TFilePos TMemBinStreamImpl::fposLogicalEnd() const
{
    return MMemStreamImpl::fposLogicalEnd();
}

inline tCIDLib::TFilePos TMemBinStreamImpl::fposPhysicalEnd() const
{
    return MMemStreamImpl::fposPhysicalEnd();
}

inline tCIDLib::TFilePos TMemBinStreamImpl::fposSeekToEnd()
{
    return MMemStreamImpl::fposSeekToEnd();
}

inline tCIDLib::TFilePos
TMemBinStreamImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    return MMemStreamImpl::fposSkipForwardBy(c4SkipBy);
}

inline tCIDLib::TVoid TMemBinStreamImpl::Reset()
{
    MMemStreamImpl::Reset();
}

inline tCIDLib::TVoid TMemBinStreamImpl::TruncateAtZero()
{
    MMemStreamImpl::TruncateAtZero();
}



// -----------------------------------------------------------------------------
//  TMemTextStreamImpl: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TMemTextStreamImpl::bEndOfStream() const
{
    return MMemStreamImpl::bEndOfStream();
}

inline tCIDLib::TFilePos TMemTextStreamImpl::fposCurPos() const
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MMemStreamImpl::fposCurPos();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;

}

inline tCIDLib::TFilePos TMemTextStreamImpl::fposLogicalEnd() const
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MMemStreamImpl::fposLogicalEnd();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}

inline tCIDLib::TFilePos TMemTextStreamImpl::fposPhysicalEnd() const
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MMemStreamImpl::fposPhysicalEnd();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}

inline tCIDLib::TFilePos TMemTextStreamImpl::fposSeekToEnd()
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MMemStreamImpl::fposSeekToEnd();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}

inline tCIDLib::TFilePos
TMemTextStreamImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    // We have to convert if UNICode
    const tCIDLib::TBoolean bUNIFmt = eTextFormat() == tCIDLib::ETextFmt_UNICode;

    // If the format is UNICode we need to convert skip count to bytes
    tCIDLib::TCard4 c4Skip = c4SkipBy;
    if (bUNIFmt)
        c4Skip *= kCIDLib::c4CharBytes;

    // Get the raw new file position
    tCIDLib::TFilePos fposRet = MMemStreamImpl::fposSkipForwardBy(c4SkipBy);

    // If the data is UNICode, then turn into a character offset
    if (bUNIFmt)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}

inline tCIDLib::TVoid TMemTextStreamImpl::Reset()
{
    MMemStreamImpl::Reset();
}

inline tCIDLib::TVoid TMemTextStreamImpl::TruncateAtZero()
{
    MMemStreamImpl::TruncateAtZero();
}
