//
// NAME: CIDLib_MemoryStreamImpl.Cpp
//
// DESCRIPTION: 
//
//  This module implements a stream data source/sink that wraps around a
//  memory buffer object. One of these is given to a stream which will then
//  use it as the backing storing for the data read and written on the
//  stream. There is one for binary streams and one for text streams.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/29/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TMemBinStreamImpl,TBinStreamImpl)
RTTIData(TMemTextStreamImpl,TTextStreamImpl)



// -----------------------------------------------------------------------------
//   CLASS: MMemStreamImpl
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  MMemStreamImpl: Constructors and Destructors
// -----------------------------------------------------------------------------

MMemStreamImpl::MMemStreamImpl
(
    TMemBuf* const                  pmbufToStream
    , const tCIDLib::EAdoptOpts     eAdopt) :

    _c4EndIndex(0)
    , _c4Index(0)
    , _eAdopted(eAdopt)
    , _pmbufData(pmbufToStream)
{
}

MMemStreamImpl::MMemStreamImpl(TMemBuf* const pmbufData) :

    _c4EndIndex(0)
    , _c4Index(0)
    , _eAdopted(tCIDLib::EAdoptOpt_NoAdopt)
    , _pmbufData(pmbufData)
{
}

MMemStreamImpl::~MMemStreamImpl()
{
    if (_eAdopted)
    {
        delete _pmbufData;
        _pmbufData = 0;
    }
}

// -----------------------------------------------------------------------------
//  MMemStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TFilePos
MMemStreamImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    if ((c4SkipBy > kCIDLib::c4MaxCard)
    ||  (_c4Index + c4SkipBy > _pmbufData->c4MaxSize()))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InvalidRelSeek
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(c4SkipBy)
        );
    }

    //
    //  Add the skip to our current index and return it. This might move
    //  it beyond _c4EndIndex, but that's cool. It just means that any
    //  reads will fail, but writes can work ok and will extend the available
    //  data up to the write position.
    //
    _c4Index += c4SkipBy;
    return _c4Index;
}

// -----------------------------------------------------------------------------
//  MMemStreamImpl: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
MMemStreamImpl::_CheckCount(const   tCIDLib::TCard4 c4Line
                            , const tCIDLib::TCard4 c4ToWrite
                            , const tCIDLib::TCard4 c4Actual)
{
    if (c4Actual != c4ToWrite)
    {
        tCIDLib::TErrCode errcToLog;
        if (!c4Actual)
            errcToLog = kCIDErrs::errcStrm_EndOfStream;
        else
            errcToLog = kCIDErrs::errcStrm_NotAllData;

        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , errcToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_OutResource
        );
    }
}

tCIDLib::TVoid
MMemStreamImpl::_CheckAvail(const   tCIDLib::TCard4         c4Line
                            , const tCIDLib::TCard4         c4BytesNeeded
                            , const MMemStreamImpl::ECheck  eCheck)
{
    tCIDLib::TCard4 c4Avail;
    if (eCheck == ECheck_ForWrite)
    {
        // Safety measure to insure that we don't do something horrible
        #if CID_DEBUG_ON
        if (_c4Index > _pmbufData->c4MaxSize())
        {
            facCIDLib.LogMsg
            (
                __FILE__
                , __LINE__
                , L"Index is beyond buffer max size"
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
            );
        }
        #endif

        // For writing, we can go to the end of the buffer
        c4Avail = _pmbufData->c4MaxSize() - _c4Index;
    }
     else
    {
        // For reading, we can only read what's been written
        if (_c4EndIndex < _c4Index)
            c4Avail = 0;
        else
            c4Avail = _c4EndIndex - _c4Index;
    }

    if (c4Avail < c4BytesNeeded)
    {
        tCIDLib::TErrCode errcToLog;
        if (!c4Avail)
            errcToLog = kCIDErrs::errcStrm_EndOfStream;
        else
            errcToLog = kCIDErrs::errcStrm_NotAllData;

        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , errcToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_OutResource
        );
    }
}


// -----------------------------------------------------------------------------
//   CLASS: TMemBinStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TMemBinStreamImpl: Constructors and Destructors
// -----------------------------------------------------------------------------

TMemBinStreamImpl::TMemBinStreamImpl
(
    TMemBuf* const                  pmbufToStream
    , const tCIDLib::EAdoptOpts     eAdopt) :

    MMemStreamImpl(pmbufToStream, eAdopt)
{
}

TMemBinStreamImpl::~TMemBinStreamImpl()
{
}


// -----------------------------------------------------------------------------
//  TMemBinStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TMemBinStreamImpl::ReadBytes(           tCIDLib::TVoid* const   pBufToFill
                                , const tCIDLib::TCard4         c4Count
                                ,       tCIDLib::TCard4&        c4Actual)
{
    //
    //  If the current index is beyond the end of data index, then set
    //  the actual bytes to read to zero. Else see if the current index
    //  plus the bytes read would go beyond the end of data, and clip the
    //  actual bytes to read back if so.
    //
    tCIDLib::TCard4 c4TmpActual = c4Count;
    if (_c4Index >= _c4EndIndex)
        c4TmpActual = 0;
    else if (_c4Index + c4Count > _c4EndIndex)
        c4TmpActual = _c4EndIndex - _c4Index;

    // If we have any actual bytes to read, then do it
    if (c4TmpActual)
    {
        //
        //  This will extend the allocation if required. However, this
        //  should never happen since we only allow reads up to _c4EndIndex
        //  and it represents how much data has been written. Such writes
        //  will have expanded the buffer to anything that we would ever
        //  read here (or that's the theory.)
        //
        _pmbufData->CopyOut(pBufToFill, c4TmpActual, _c4Index);

        // Add the actual bytes to the current position
        _c4Index += c4TmpActual;
    }

    if (&c4Actual)
        c4Actual = c4TmpActual;
     else
        _CheckCount(__LINE__, c4Count, c4TmpActual);
}


tCIDLib::TVoid
TMemBinStreamImpl::WriteBytes(  const   tCIDLib::TVoid* const   pBufToUse
                                , const tCIDLib::TCard4         c4Count
                                ,       tCIDLib::TCard4&        c4Actual)
{
    tCIDLib::TCard4 c4TmpActual = c4Count;
    tCIDLib::TCard4 c4CurPos    = _c4Index;

    if (c4CurPos + c4Count > _pmbufData->c4MaxSize())
        c4TmpActual = _pmbufData->c4MaxSize() - c4CurPos;

    // This will extend the current allocation if needed
    _pmbufData->CopyIn(pBufToUse, c4TmpActual, c4CurPos);

    // Add the actual bytes to the current position
    _c4Index += c4TmpActual;

    //
    //  A write extends the readable data, so a successful write means
    //  we need to move our end position index up to the current position.
    //
    _c4EndIndex = _c4Index;

    if (&c4Actual)
        c4Actual = c4TmpActual;
     else
        _CheckCount(__LINE__, c4Count, c4TmpActual);
}




// -----------------------------------------------------------------------------
//   CLASS: TMemTextStreamImpl
//  PREFIX: strm
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TMemTextStreamImpl: Constructors and Destructors
// -----------------------------------------------------------------------------

TMemTextStreamImpl::TMemTextStreamImpl
(
    TMemBuf* const                  pmbufToStream
    , const tCIDLib::ETextFormats   eTextFmt
    , const tCIDLib::EAdoptOpts     eAdopt) :

    MMemStreamImpl(pmbufToStream, eAdopt)
    , TTextStreamImpl(eTextFmt)
{
}

TMemTextStreamImpl::~TMemTextStreamImpl()
{
}

// -----------------------------------------------------------------------------
//  TMemTextStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4
TMemTextStreamImpl::c4GetLine(          tCIDLib::Tch* const pszBufToFill
                                , const tCIDLib::TCard4     c4BufChars
                                , const tCIDLib::TBoolean   bStripWhitespace)
{
    // If we are already at the end of stream, thats an error
    if (bEndOfStream())
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_EndOfStream
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_OutResource
        );
    }

    // Get the bytes we need for each char
    tCIDLib::TCard4 c4PerChar = 1;
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        c4PerChar = kCIDLib::c4CharBytes;

    tCIDLib::Tch    chCur;
    tCIDLib::TCard4 c4Count = 0;
    while ((_c4Index + c4PerChar) <= _c4EndIndex)
    {
        //
        //  Get the next char and bump the index. According to the source
        //  format, we may have to convert to internal UNICode format. And
        //  this affects the amount we bump up the index.
        //
        if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        {
            chCur = _pmbufData->chAt(_c4Index);
            _c4Index += kCIDLib::c4CharBytes;
        }
         else
        {
            chCur = TRawStr::chConvert
            (
                _pmbufData->schAt(_c4Index++)
            );
        }

        // If a newline, then cap the string and return
        if (chCur == kCIDLib::chLF)
        {
            pszBufToFill[c4Count] = 0;
            break;
        }

        // Put it in the buffer and bump up the count
        pszBufToFill[c4Count] = chCur;
        c4Count++;

        //
        //  Don't overflow the buffer. And leave room for the terminating
        //  null.
        //
        if (c4Count >= c4BufChars-1)
            break;
    }

    // Cap it off now
    pszBufToFill[c4Count] = 0;

    //
    //  If the last character was a carraige return, then do away with
    //  it, since its just part of a CR/LF at the line end.
    //
    if (c4Count)
    {
        if (pszBufToFill[c4Count-1] == kCIDLib::chCR)
        {
            pszBufToFill[c4Count-1] = kCIDLib::chNull;
            c4Count--;
        }
    }

    // Strip white space off it requested, and adjust the count
    if (bStripWhitespace)
    {
        TRawStr::StripStr
        (
            pszBufToFill
            , kCIDLib::szWhitespace
            , tCIDLib::EStripMode_LeadTrail
        );
        c4Count = TRawStr::c4StrLen(pszBufToFill);
    }

    return c4Count;
}


tCIDLib::Tch TMemTextStreamImpl::chGet()
{
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        _CheckAvail(__LINE__, sizeof(tCIDLib::Tch), ECheck_ForRead);
        tCIDLib::Tch chRet = _pmbufData->chAt(_c4Index);
        _c4Index += sizeof(chRet);
        return chRet;
    }
     else
    {
        _CheckAvail(__LINE__, sizeof(tCIDLib::Tsch), ECheck_ForRead);
        tCIDLib::Tsch schRet = _pmbufData->schAt(_c4Index);
        _c4Index += sizeof(schRet);
        return TRawStr::chConvert(schRet);
    }
}


tCIDLib::TVoid TMemTextStreamImpl::PutCh(const tCIDLib::Tch chToWrite)
{
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        _CheckAvail(__LINE__, sizeof(chToWrite), ECheck_ForWrite);
        _pmbufData->PutCh(chToWrite, _c4Index);
        _c4Index += sizeof(chToWrite);
    }
     else
    {
        tCIDLib::Tsch schTmp = TRawStr::schConvert(chToWrite);
        _pmbufData->PutCh(schTmp, _c4Index);
        _c4Index += sizeof(schTmp);
    }

    // See if we need to bump up the logical end
    if (_c4Index > _c4EndIndex)
        _c4EndIndex = _c4Index;
}

tCIDLib::TVoid TMemTextStreamImpl::PutCh(const tCIDLib::Tsch schToWrite)
{
    if (eTextFormat() == tCIDLib::ETextFmt_ASCII)
    {
        _CheckAvail(__LINE__, sizeof(schToWrite), ECheck_ForWrite);
        _pmbufData->PutCh(schToWrite, _c4Index);
        _c4Index += sizeof(schToWrite);
    }
     else
    {
        tCIDLib::Tch chTmp = TRawStr::chConvert(schToWrite);
        _pmbufData->PutCh(chTmp, _c4Index);
        _c4Index += sizeof(chTmp);
    }

    // See if we need to bump up the logical end
    if (_c4Index > _c4EndIndex)
        _c4EndIndex = _c4Index;
}


tCIDLib::TVoid
TMemTextStreamImpl::PutLine(const tCIDLib::Tch* const pszBufToWrite)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszBufToWrite);
    if (!c4Len)
        return;

    tCIDLib::TCard4 c4ToWrite = c4Len;
    const tCIDLib::TVoid* pToWrite = pszBufToWrite;
    tCIDLib::Tsch*  pszTmp = 0;
    try
    {
        if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        {
            c4ToWrite *= kCIDLib::c4CharBytes;
        }
         else
        {
            tCIDLib::Tsch* pszTmp = TRawStr::pszConvert(pszBufToWrite);
            pToWrite = pszTmp;
        }

        _CheckAvail(__LINE__, c4ToWrite, ECheck_ForWrite);

        _pmbufData->CopyIn(pToWrite, c4ToWrite, _c4Index);

        // Add the actual bytes to the current position
        _c4Index += c4ToWrite;

        // See if we need to bump up the logical end
        if (_c4Index > _c4EndIndex)
            _c4EndIndex = _c4Index;
    }

    catch(...)
    {
        delete pszTmp;
        throw;
    }

    delete pszTmp;
}

tCIDLib::TVoid
TMemTextStreamImpl::PutLine(const tCIDLib::Tsch* const pszBufToWrite)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszBufToWrite);
    if (!c4Len)
        return;

    tCIDLib::TCard4 c4ToWrite = c4Len;
    const tCIDLib::TVoid* pToWrite = pszBufToWrite;
    tCIDLib::Tsch*  pszTmp = 0;
    try
    {
        if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        {
            c4ToWrite *= kCIDLib::c4CharBytes;
            tCIDLib::Tch* pszTmp = TRawStr::pszConvert(pszBufToWrite);
            pToWrite = pszTmp;
        }

        _CheckAvail(__LINE__, c4ToWrite, ECheck_ForWrite);

        _pmbufData->CopyIn(pToWrite, c4ToWrite, _c4Index);

        // Add the actual bytes to the current position
        _c4Index += c4ToWrite;

        // See if we need to bump up the logical end
        if (_c4Index > _c4EndIndex)
            _c4EndIndex = _c4Index;
    }

    catch(...)
    {
        delete pszTmp;
        throw;
    }

    delete pszTmp;
}


tCIDLib::Tsch TMemTextStreamImpl::schGet()
{
    if (eTextFormat() == tCIDLib::ETextFmt_ASCII)
    {
        _CheckAvail(__LINE__, sizeof(tCIDLib::Tsch), ECheck_ForRead);
        tCIDLib::Tsch schRet = _pmbufData->schAt(_c4Index);
        _c4Index += sizeof(schRet);
        return schRet;
    }
     else
    {
        _CheckAvail(__LINE__, sizeof(tCIDLib::Tch), ECheck_ForRead);
        tCIDLib::Tch chRet = _pmbufData->chAt(_c4Index);
        _c4Index += sizeof(chRet);
        return TRawStr::schConvert(chRet);
    }
}
