//
// NAME: CIDLib_Memory.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_Memory.Cpp module. This module provides
//  the TMemBuf class, and two derivatives thereof THeapBuf and TSysBuf. TMemBuf
//  is the base memory class, which knows how to manage a memory buffer but does
//  not know how to create or destroy them. Derived classes handle creating and
//  destroying the buffers.
//
//  THeapBuf is a derivative for heap based buffers. TSysBuf is a derivaitve for
//  unshared system memory buffers.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/18/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TMemBuf
// PREFIX: mbuf
// -----------------------------------------------------------------------------
class CIDLIBEXP TMemBuf :

    public TObject, public MDuplicable, public MStreamable
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        ~TMemBuf();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        tCIDLib::TCard1& operator[]
        (
            const   tCIDLib::TCard4         c4Ind
        );

        const tCIDLib::TCard1 operator[]
        (
            const   tCIDLib::TCard4         c4Ind
        )   const;

        tCIDLib::TBoolean operator==
        (
            const   TMemBuf&                mbufToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TMemBuf&                mbufToTest
        )   const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bBuffersEqual
        (
            const   TMemBuf&                mbufToCompare
        );

        tCIDLib::TCard1 c1At
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::TCard2 c2At
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::TCard4 c4At
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::Tch chAt
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::TCard1 c1Fill() const;

        tCIDLib::TCard1 c1Fill
        (
            const   tCIDLib::TCard1         c1Fill
        );

        tCIDLib::TCard4 c4CheckSum() const;

        tCIDLib::TCard4 c4CheckSum
        (
            const   tCIDLib::TCard4         c4StartInd
        )   const;

        tCIDLib::TCard4 c4CheckSum
        (
            const   tCIDLib::TCard4         c4StartInd
            , const tCIDLib::TCard4         c4Count
        )   const;

        tCIDLib::TCard4 c4ExpandIncrement() const;

        tCIDLib::TCard4 c4MaxSize() const;

        tCIDLib::TCard4 c4Size() const;

        tCIDLib::TVoid CopyIn
        (
            const   tCIDLib::TVoid* const   pData
            , const tCIDLib::TCard4         c4DataSz
            , const tCIDLib::TCard4         c4StartInd = 0
        );

        tCIDLib::TVoid CopyIn
        (
            const   TMemBuf&                mbufSrc
            , const tCIDLib::TCard4         c4DataSz
            , const tCIDLib::TCard4         c4StartInd = 0
        );

        tCIDLib::TVoid CopyOut
        (
                    tCIDLib::TVoid* const   pData
            , const tCIDLib::TCard4         c4DataSz
            , const tCIDLib::TCard4         c4StartInd = 0
        )   const;

        tCIDLib::TVoid CopyOut
        (
                    TMemBuf&                mbufSrc
            , const tCIDLib::TCard4         c4DataSz
            , const tCIDLib::TCard4         c4StartInd = 0
        )   const;

        tCIDLib::TFloat4 f4At
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::TFloat8 f8At
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::THashVal hshCalcHash
        (
            const   tCIDLib::TCard4         c4Modulus
            , const tCIDLib::TCard4         c4StartInd = 0
        )   const;

        tCIDLib::TInt1 i1At
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::TInt2 i2At
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::TInt4 i4At
        (
            const   tCIDLib::TCard4         c4Index
        );

        const tCIDLib::TVoid* pData() const;

        tCIDLib::TVoid* pData();

        const tCIDLib::TCard1* pc1Data() const;

        tCIDLib::TCard1* pc1Data();

        TBinaryStream* pstrmMakeNew();

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tch            chToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tsch           schToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutCard1
        (
            const   tCIDLib::TCard1         c1ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutCard2
        (
            const   tCIDLib::TCard2         c2ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutCard4
        (
            const   tCIDLib::TCard4         c4ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutFloat4
        (
            const   tCIDLib::TFloat4        f4ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutFloat8
        (
            const   tCIDLib::TFloat8        f8ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutInt1
        (
            const   tCIDLib::TInt1          i1ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutInt2
        (
            const   tCIDLib::TInt2          i2ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid PutInt4
        (
            const   tCIDLib::TInt4          i4ToPut
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid ReallocateTo
        (
            const   tCIDLib::TCard4         c4NewSize
            , const tCIDLib::TBoolean       bPreserve = kCIDLib::True
        )   const;

        tCIDLib::Tsch schAt
        (
            const   tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid Set
        (
            const   tCIDLib::TCard1         c1Fill
        );

        tCIDLib::TVoid Set
        (
            const   tCIDLib::TCard1         c1Fill
            , const tCIDLib::TCard4         c4StartInd
        );

        tCIDLib::TVoid Set
        (
            const   tCIDLib::TCard1         c1Fill
            , const tCIDLib::TCard4         c4StartInd
            , const tCIDLib::TCard4         c4Count
        );


    protected           :
        // ---------------------------------------------------------------------
        //  Hidden constructors and operators
        // ---------------------------------------------------------------------
        TMemBuf();

        TMemBuf
        (
            const   TMemBuf&                mbufToCopy
        );

        TMemBuf
        (
            const   tCIDLib::TCard4         c4InitSize
            , const tCIDLib::TCard4         c4MaxSize
            , const tCIDLib::TCard4         c4ExpandIncrement
            ,       tCIDLib::TVoid* const   pData
        );


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


        // ---------------------------------------------------------------------
        //  Protected, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid _ReallocateTo
        (
            const   tCIDLib::TCard4         c4NewSize
            , const tCIDLib::TBoolean       bPreserve = kCIDLib::True
        ) const = 0;


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _ReleaseBuffer();

        tCIDLib::TVoid _ValidateMembers
        (
            const   tCIDLib::Tch* const     pszMethod
            , const tCIDLib::TCard4         c4Line
        )   const;


        // ---------------------------------------------------------------------
        //  Protected data members
        //
        //  _c1Fill
        //      This is the character used to fill all buffers. Its used to
        //      init the initial allocation and the newly allocated portions
        //      when expanded. Defaults to zero.
        //  
        //  _c4ExpandIncrement
        //      This is the increment to expand the buffer by when it needs
        //      to be reallocated to hold something.
        //
        //  _c4MaxSize
        //      This is the maximum size that the buffer can grow to. The
        //      various derivatives handle the expansion differently. We
        //      just give them a common place to store it.
        //
        //  _c4Size
        //      The number of bytes pointed to by _pData.
        //
        //  _pc1Data
        //      A pointer to the data buffer. We don't allocate or deallocate
        //      these. They belong to the derived class who is responsible for
        //      them. We just know how to do things to a given buffer.
        // ---------------------------------------------------------------------
        tCIDLib::TCard1             _c1Fill;
        tCIDLib::TCard4             _c4ExpandIncrement;
        tCIDLib::TCard4             _c4MaxSize;

        // These are mutable and not counted in equality tests
        mutable tCIDLib::TCard4     _c4Size;
        mutable tCIDLib::TCard1*    _pc1Data;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TMemBuf& operator=(const TMemBuf&);


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid __CheckIndex
        (
            const   tCIDLib::TCard4     c4Line
            , const tCIDLib::TCard4     c4Index
        )   const;

        tCIDLib::TVoid __CheckRange
        (
            const   tCIDLib::TCard4     c4Line
            , const tCIDLib::TCard4     c4StartInd
            , const tCIDLib::TCard4     c4Count
        )   const;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TMemBuf,TObject)
};



// -----------------------------------------------------------------------------
//  CLASS: THeapBuf
// PREFIX: mbuf
// -----------------------------------------------------------------------------
class CIDLIBEXP THeapBuf : public TMemBuf
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        THeapBuf
        (
            const   tCIDLib::TCard4         c4Size
            , const tCIDLib::TCard4         c4MaxSize = 0
            , const tCIDLib::TCard4         c4ExpandIncrement = 0
        );

        THeapBuf
        (
            const   THeapBuf&               mbufToCopy
        );

        ~THeapBuf();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        THeapBuf& operator=
        (
            const   THeapBuf&               mbufToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   THeapBuf&               mbufToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   THeapBuf&               mbufToTest
        )   const;


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _ReallocateTo
        (
            const   tCIDLib::TCard4         c4NewSize
            , const tCIDLib::TBoolean       bPreserve
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    protected           :
        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        THeapBuf();


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(THeapBuf,TMemBuf)
        DefPolyDup(THeapBuf)
};



// -----------------------------------------------------------------------------
//  CLASS: TSysBuf
// PREFIX: mbuf
// -----------------------------------------------------------------------------
class CIDLIBEXP TSysBuf : public TMemBuf
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        TSysBuf
        (
            const   tCIDLib::TCard4         c4Size
            , const tCIDLib::TCard4         c4MaxSize
            , const tCIDLib::EMemAccFlags   eAccFlags = tCIDLib::EMemAcc_ReadWrite
        );

        TSysBuf
        (
            const   tCIDLib::TCard4         c4MaxSize
            , const tCIDLib::EMemAccFlags   eAccFlags = tCIDLib::EMemAcc_ReadWrite
        );

        TSysBuf
        (
            const   TSysBuf&                mbufSrc
        );

        ~TSysBuf();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TSysBuf& operator=
        (
            const   TSysBuf&                mbufToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TSysBuf&                mbufToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TSysBuf&                mbufToTest
        )   const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::EMemAccFlags eMemAccFlags() const;

    
    protected           :
        // ---------------------------------------------------------------------
        //  Protected methods
        // ---------------------------------------------------------------------
        TSysBuf();


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _ReallocateTo
        (
            const   tCIDLib::TCard4         c4NewSize
            , const tCIDLib::TBoolean       bPreserve
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private             :
        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TCard4 __c4RoundedUp
        (
            const   tCIDLib::TCard4         c4ToRound
        )   const;

        tCIDLib::TVoid __Create();

        tCIDLib::TVoid* __pPageAdr
        (
            const   tCIDLib::TVoid* const   pAddress
        )   const;


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __eAccess
        //      They are the access rights that were given to this object
        //      during contruction.
        // ---------------------------------------------------------------------
        tCIDLib::EMemAccFlags   __eAccess;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TSysBuf,TMemBuf)
        DefPolyDup(TSysBuf)
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  TMemBuf: Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TMemBuf::operator!=(const TMemBuf& mbufToTest) const
{
    return !operator==(mbufToTest);
}


// -----------------------------------------------------------------------------
//  TMemBuf: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TCard1 TMemBuf::c1Fill() const
{
    return _c1Fill;
}

inline tCIDLib::TCard1 TMemBuf::c1Fill(const tCIDLib::TCard1 c1Fill)
{
    _c1Fill = c1Fill;
    return _c1Fill;
}

inline tCIDLib::TCard4 TMemBuf::c4ExpandIncrement() const
{
    return _c4ExpandIncrement;
}

inline tCIDLib::TCard4 TMemBuf::c4MaxSize() const
{
    return _c4MaxSize;
}

inline tCIDLib::TCard4 TMemBuf::c4Size() const
{
    return _c4Size;
}

inline const tCIDLib::TVoid* TMemBuf::pData() const
{
    return (tCIDLib::TVoid*)_pc1Data;
}

inline tCIDLib::TVoid* TMemBuf::pData()
{
    return (tCIDLib::TVoid*)_pc1Data;
}

inline const tCIDLib::TCard1* TMemBuf::pc1Data() const
{
    return _pc1Data;
}

inline tCIDLib::TCard1* TMemBuf::pc1Data()
{
    return _pc1Data;
}


// -----------------------------------------------------------------------------
//  TMemBuf: Protected, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid TMemBuf::_ReleaseBuffer()
{
    _c4Size = 0;
    _pc1Data = 0;
}


// -----------------------------------------------------------------------------
//  THeapBuf: Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
THeapBuf::operator==(const THeapBuf& mbufToTest) const
{
    return TMemBuf::operator==(mbufToTest);
}

inline tCIDLib::TBoolean
THeapBuf::operator!=(const THeapBuf& mbufToTest) const
{
    return !TMemBuf::operator==(mbufToTest);
}


// -----------------------------------------------------------------------------
//  THeapBuf: Hidden constructors. Needed for polymorphic streaming.
// -----------------------------------------------------------------------------
inline THeapBuf::THeapBuf()
{
}



// -----------------------------------------------------------------------------
//  TSysBuf: Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TSysBuf::operator!=(const TSysBuf& mbufToTest) const
{
    return !operator==(mbufToTest);
}


// -----------------------------------------------------------------------------
//  TSysBuf: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::EMemAccFlags TSysBuf::eMemAccFlags() const
{
    return __eAccess;
}

// -----------------------------------------------------------------------------
//  TSysBuf: Hidden constructors. Needed for polymorphic streaming.
// -----------------------------------------------------------------------------
inline TSysBuf::TSysBuf() :

    TMemBuf()
    , __eAccess(tCIDLib::EMemAcc_ReadWrite)
{
}
