//
// NAME: CIDLib_Memory.Cpp
//
// DESCRIPTION:
//
//  This method provides some memory buffer management classes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/14/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TMemBuf,TObject)
RTTIData(THeapBuf,TMemBuf)
RTTIData(TSysBuf,TMemBuf)



// -----------------------------------------------------------------------------
//   CLASS: TMemBuf
//  PREFIX: mbuf
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TMemBuf: Constructors and Destructors
// -----------------------------------------------------------------------------

TMemBuf::~TMemBuf()
{
}    


// -----------------------------------------------------------------------------
//  TMemBuf: Public operators
// -----------------------------------------------------------------------------

tCIDLib::TCard1& TMemBuf::operator[](const tCIDLib::TCard4 c4Ind)
{
    __CheckIndex(__LINE__, c4Ind);
    return _pc1Data[c4Ind];
}

const tCIDLib::TCard1
TMemBuf::operator[](const tCIDLib::TCard4 c4Ind) const
{
    __CheckIndex(__LINE__, c4Ind);
    return _pc1Data[c4Ind];
}


tCIDLib::TBoolean TMemBuf::operator==(const TMemBuf& mbufToTest) const
{
    //
    //  !!The current size does not count, since its a mutable
    //  implementation detail
    //
    if (mbufToTest._c4ExpandIncrement != _c4ExpandIncrement)
        return kCIDLib::False;

    if (mbufToTest._c4MaxSize != _c4MaxSize)
        return kCIDLib::False;

    if (mbufToTest._c1Fill != _c1Fill)
        return kCIDLib::False;

    // Compare the bytes of the buffer
    if (TRawMem::eCompareMemBuf(_pc1Data, mbufToTest._pc1Data, _c4Size))
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TMemBuf: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TMemBuf::bBuffersEqual(const TMemBuf& mbufToCompare)
{
    // They have to have the same size
    if (mbufToCompare._c4Size != _c4Size)
        return kCIDLib::False;

    // Compare the bytes of the buffer
    if (TRawMem::eCompareMemBuf(_pc1Data, mbufToCompare._pc1Data, _c4Size))
        return kCIDLib::False;

    return kCIDLib::True;
}


tCIDLib::TCard1 TMemBuf::c1At(const tCIDLib::TCard4 c4Ind)
{
    __CheckIndex(__LINE__, c4Ind);
    return _pc1Data[c4Ind];
}

tCIDLib::TCard2 TMemBuf::c2At(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::TCard2));
    return *(tCIDLib::TCard2*)&_pc1Data[c4Ind];
}

tCIDLib::TCard4 TMemBuf::c4At(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::TCard4));
    return *(tCIDLib::TCard4*)&_pc1Data[c4Ind];
}

tCIDLib::Tch TMemBuf::chAt(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::Tch));
    return *((tCIDLib::Tch*)&_pc1Data[c4Ind]);
}

tCIDLib::TCard4 TMemBuf::c4CheckSum() const
{
    // Calc the checksum
    tCIDLib::TCard4         c4Sum   = 0;
    const tCIDLib::TCard1*  pc1Tmp  = _pc1Data;

    for (tCIDLib::TCard4 c4Tmp = 0; c4Tmp < _c4Size; c4Tmp++)
    {
        c4Sum += *pc1Tmp;
        pc1Tmp++;
    }
    return c4Sum;
}

tCIDLib::TCard4 TMemBuf::c4CheckSum(const tCIDLib::TCard4 c4StartInd) const
{
    __CheckIndex(__LINE__, c4StartInd);
    return c4CheckSum(c4StartInd, _c4Size - c4StartInd);
}

tCIDLib::TCard4
TMemBuf::c4CheckSum(const   tCIDLib::TCard4 c4StartInd
                    , const tCIDLib::TCard4 c4Count) const
{
    __CheckRange(__LINE__, c4StartInd, c4Count);

    // Calc the checksum
    tCIDLib::TCard4         c4Sum   = 0;
    const tCIDLib::TCard1*  pc1Tmp  = &_pc1Data[c4StartInd];

    for (tCIDLib::TCard4 c4Tmp = 0; c4Tmp < c4Count; c4Tmp++)
    {
        c4Sum += *pc1Tmp;
        pc1Tmp++;
    }
    return c4Sum;
}


tCIDLib::TVoid
TMemBuf::CopyIn(const   tCIDLib::TVoid* const   pToCopy
                , const tCIDLib::TCard4         c4DataSz
                , const tCIDLib::TCard4         c4StartInd)
{
    if (!c4DataSz)
        return;

    __CheckRange(__LINE__, c4StartInd, c4DataSz);
    TRawMem::CopyMemBuf(&_pc1Data[c4StartInd], pToCopy, c4DataSz);
}

tCIDLib::TVoid TMemBuf::CopyIn( const   TMemBuf&        mbufToCopy
                                , const tCIDLib::TCard4 c4DataSz
                                , const tCIDLib::TCard4 c4StartInd)
{
    //
    //  Check this buffer and the source one. For the source, just make
    //  sure it has at least the indicated size otherwise we cannot copy
    //  that many bytes out of it.
    //
    __CheckRange(__LINE__, c4StartInd, c4DataSz);
    mbufToCopy.__CheckRange(__LINE__, 0, c4DataSz);
    TRawMem::CopyMemBuf(&_pc1Data[c4StartInd], mbufToCopy._pc1Data, c4DataSz);
}


tCIDLib::TVoid
TMemBuf::CopyOut(           tCIDLib::TVoid* const   pToFill
                    , const tCIDLib::TCard4         c4DataSz
                    , const tCIDLib::TCard4         c4StartInd) const
{
    if (!c4DataSz)
        return;

    __CheckRange(__LINE__, c4StartInd, c4DataSz);
    TRawMem::CopyMemBuf(pToFill, &_pc1Data[c4StartInd], c4DataSz);
}

tCIDLib::TVoid
TMemBuf::CopyOut(           TMemBuf&        mbufToFill
                    , const tCIDLib::TCard4 c4DataSz
                    , const tCIDLib::TCard4 c4StartInd) const
{
    // Check both buffers, the target just needs to be at least large enough
    __CheckRange(__LINE__, c4StartInd, c4DataSz);
    mbufToFill.__CheckRange(__LINE__, 0, c4DataSz);

    TRawMem::CopyMemBuf(mbufToFill._pc1Data, &_pc1Data[c4StartInd], c4DataSz);
}


tCIDLib::TFloat4 TMemBuf::f4At(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::TFloat4));
    return *(tCIDLib::TFloat4*)&_pc1Data[c4Ind];
}

tCIDLib::TFloat8 TMemBuf::f8At(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::TFloat8));
    return *(tCIDLib::TFloat8*)&_pc1Data[c4Ind];
}


tCIDLib::THashVal
TMemBuf::hshCalcHash(   const   tCIDLib::TCard4 c4Modulus
                        , const tCIDLib::TCard4 c4StartInd) const
{
    __CheckIndex(__LINE__, c4StartInd);
    return TRawMem::hshHashBuffer
    (
        _pc1Data
        , c4Modulus
        , _c4Size - c4StartInd
    );
}

tCIDLib::TInt1 TMemBuf::i1At(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::TInt1));
    return (tCIDLib::TInt1)_pc1Data[c4Ind];
}

tCIDLib::TInt2 TMemBuf::i2At(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::TInt2));
    return *(tCIDLib::TInt2*)&_pc1Data[c4Ind];
}

tCIDLib::TInt4 TMemBuf::i4At(const tCIDLib::TCard4 c4Ind)
{
    __CheckRange(__LINE__, c4Ind, sizeof(tCIDLib::TInt4));
    return *(tCIDLib::TInt4*)&_pc1Data[c4Ind];
}

TBinaryStream* TMemBuf::pstrmMakeNew()
{
    return new TBinaryStream(new TMemBinStreamImpl(this));
}

tCIDLib::TVoid TMemBuf::PutCh(  const   tCIDLib::Tch    chToPut
                                , const tCIDLib::TCard4 c4Index)
{
    __CheckRange(__LINE__, c4Index, sizeof(chToPut));
    *((tCIDLib::Tch*)&_pc1Data[c4Index]) = chToPut;
}

tCIDLib::TVoid TMemBuf::PutCh(  const   tCIDLib::Tsch   schToPut
                                , const tCIDLib::TCard4 c4Index)
{
    __CheckIndex(__LINE__, c4Index);
    *((tCIDLib::Tsch*)&_pc1Data[c4Index]) = schToPut;
}

tCIDLib::TVoid TMemBuf::PutCard1(   const   tCIDLib::TCard1 c1ToPut
                                    , const tCIDLib::TCard4 c4Index)
{
    __CheckIndex(__LINE__, c4Index);
    *((tCIDLib::TCard1*)&_pc1Data[c4Index]) = c1ToPut;
}

tCIDLib::TVoid TMemBuf::PutCard2(   const   tCIDLib::TCard2 c2ToPut
                                    , const tCIDLib::TCard4 c4Index)
{
    __CheckRange(__LINE__, c4Index, sizeof(c2ToPut));
    *((tCIDLib::TCard2*)&_pc1Data[c4Index]) = c2ToPut;
}

tCIDLib::TVoid TMemBuf::PutCard4(   const   tCIDLib::TCard4 c4ToPut
                                    , const tCIDLib::TCard4 c4Index)
{
    __CheckRange(__LINE__, c4Index, sizeof(c4ToPut));
    *((tCIDLib::TCard4*)&_pc1Data[c4Index]) = c4ToPut;
}

tCIDLib::TVoid TMemBuf::PutFloat4(  const   tCIDLib::TFloat4    f4ToPut
                                    , const tCIDLib::TCard4     c4Index)
{
    __CheckRange(__LINE__, c4Index, sizeof(f4ToPut));
    *((tCIDLib::TFloat4*)&_pc1Data[c4Index]) = f4ToPut;
}

tCIDLib::TVoid TMemBuf::PutFloat8(  const   tCIDLib::TFloat8    f8ToPut
                                    , const tCIDLib::TCard4     c4Index)
{
    __CheckRange(__LINE__, c4Index, sizeof(f8ToPut));
    *((tCIDLib::TFloat8*)&_pc1Data[c4Index]) = f8ToPut;
}

tCIDLib::TVoid TMemBuf::PutInt1(const   tCIDLib::TInt1  i1ToPut
                                , const tCIDLib::TCard4 c4Index)
{
    __CheckIndex(__LINE__, c4Index);
    *((tCIDLib::TInt1*)&_pc1Data[c4Index]) = i1ToPut;
}

tCIDLib::TVoid TMemBuf::PutInt2(const   tCIDLib::TInt2  i2ToPut
                                , const tCIDLib::TCard4 c4Index)
{
    __CheckRange(__LINE__, c4Index, sizeof(i2ToPut));
    *((tCIDLib::TInt2*)&_pc1Data[c4Index]) = i2ToPut;
}

tCIDLib::TVoid TMemBuf::PutInt4(const   tCIDLib::TInt4  i4ToPut
                                , const tCIDLib::TCard4 c4Index)
{
    __CheckRange(__LINE__, c4Index, sizeof(i4ToPut));
    *((tCIDLib::TInt4*)&_pc1Data[c4Index]) = i4ToPut;
}


tCIDLib::TVoid TMemBuf::ReallocateTo(const  tCIDLib::TCard4     c4NewSize
                                    , const tCIDLib::TBoolean   bPreserve) const
{
    // If the enw size is larger than the max, we cannot expand
    if (c4NewSize > _c4MaxSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_CannotExpand
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4NewSize)
            , TCardinal(_c4MaxSize)
        );
    }

    // If its less than or equal to current size, then nothing to do
    if (c4NewSize <= _c4Size)
        return;

    //
    //  Take the new size and round to the next expansion increment,
    //  clipped back to the maximum size of course.
    //
    tCIDLib::TCard4 c4TmpSize = c4NewSize;
    if (c4NewSize < _c4ExpandIncrement)
    {
        c4TmpSize = _c4ExpandIncrement;
    }
     else
    {
        if (c4NewSize % _c4ExpandIncrement)
        {
            c4TmpSize = (c4NewSize / _c4ExpandIncrement) * _c4ExpandIncrement;
            c4TmpSize += _c4ExpandIncrement;
        }
    }

    if (c4TmpSize > _c4MaxSize)
        c4TmpSize = _c4MaxSize;

    // Call the protected virtual method
    _ReallocateTo(c4TmpSize, bPreserve);
}


tCIDLib::Tsch TMemBuf::schAt(const tCIDLib::TCard4 c4Ind)
{
    __CheckIndex(__LINE__, c4Ind);
    return (tCIDLib::Tsch)_pc1Data[c4Ind];
}


tCIDLib::TVoid TMemBuf::Set(const tCIDLib::TCard1 c1Fill)
{
    TRawMem::SetMemBuf(_pc1Data, c1Fill, _c4Size);
}

tCIDLib::TVoid
TMemBuf::Set(const tCIDLib::TCard1 c1Fill, const tCIDLib::TCard4 c4StartInd)
{
    __CheckIndex(__LINE__, c4StartInd);
    TRawMem::SetMemBuf(&_pc1Data[c4StartInd], c1Fill, _c4Size - c4StartInd);
}

tCIDLib::TVoid
TMemBuf::Set(   const   tCIDLib::TCard1 c1Fill
                , const tCIDLib::TCard4 c4StartInd
                , const tCIDLib::TCard4 c4Count)
{
    if (!c4Count)
        return;
    __CheckRange(__LINE__, c4StartInd, c4Count);
    TRawMem::SetMemBuf(&_pc1Data[c4StartInd], c1Fill, c4Count);
}


// -----------------------------------------------------------------------------
//  TMemBuf: Hidden constructors/operators
// -----------------------------------------------------------------------------

TMemBuf::TMemBuf() :

    _c1Fill(0)
    , _c4ExpandIncrement(0)
    , _c4MaxSize(0)
    , _c4Size(0)
    , _pc1Data(0)
{
    // Don't validate the members because they are invalid!!
}

TMemBuf::TMemBuf(const TMemBuf& mbufToCopy) :

    _c1Fill(mbufToCopy._c1Fill)
    , _c4ExpandIncrement(mbufToCopy._c4ExpandIncrement)
    , _c4MaxSize(mbufToCopy._c4MaxSize)
    , _c4Size(mbufToCopy._c4Size)
    , _pc1Data(0)
{
    _ValidateMembers(L"TMemBuf", __LINE__);
}

TMemBuf::TMemBuf(   const   tCIDLib::TCard4         c4Size
                    , const tCIDLib::TCard4         c4MaxSize
                    , const tCIDLib::TCard4         c4ExpandIncrement
                    ,       tCIDLib::TVoid* const   pData) :
    _c1Fill(0)
    , _c4ExpandIncrement(c4ExpandIncrement)
    , _c4MaxSize(c4MaxSize)
    , _c4Size(c4Size)
    , _pc1Data((tCIDLib::TCard1*)pData)
{
    // If the size is zero, then its all in vain, so just go validate and die
    if (_c4Size)
    {
        //
        //  If the max size is 0, then set it to the initial size, so this
        //  one will not be expandable.
        //
        if (!_c4MaxSize)
            _c4MaxSize = _c4Size;

        //
        //  If the expand increment is 0, then set it to the max minus the
        //  initial size. That might still be 0, but that's legal since the
        //  buffer is already the max size.
        //
        if (!_c4ExpandIncrement)
            _c4ExpandIncrement = _c4MaxSize - _c4Size;
    }
    _ValidateMembers(L"TMemBuf", __LINE__);
}


// -----------------------------------------------------------------------------
//  TMemBuf: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TMemBuf::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    //
    //  This guy just handles streaming in and out a stream marker. Because
    //  streaming in random data as a buffer size (and then allocating it!) is
    //  very dangerous. So this helps us know whether we are on a valid object.
    //  We do this for all of our derivatives, but they are responsible for
    //  the buffer members, _c4Size and _pc1Data.
    //
    TBinaryStream::EStreamMarkers eMarker;
    strmToReadFrom >> eMarker;
    if (eMarker != TBinaryStream::EStreamMarker_StartObject)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoStartMarker
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TInteger(eMarker)
        );
    }
}

tCIDLib::TVoid TMemBuf::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Stream out the marker
    strmToWriteTo << TBinaryStream::EStreamMarker_StartObject;
}


// -----------------------------------------------------------------------------
//  TMemBuf: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TMemBuf::_ValidateMembers(  const   tCIDLib::Tch* const pszMethod
                            , const tCIDLib::TCard4     c4Line) const
{
    if (!_c4MaxSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_BadState
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"the max size is 0")
        );
    }

    if (!_c4Size)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_BadState
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"the initial size is 0")
        );
    }

    //
    //  If the buffer has room to expand, then there must be an expand
    //  increment.
    //
    if ((_c4Size != _c4MaxSize) && !_c4ExpandIncrement)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_BadState
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"the expand increment is 0")
        );
    }

    if (_c4Size > _c4MaxSize)
    {
        TString strTmp(L"The size (%(1)) is > than the max size (%(2))");
        strTmp.ReplaceToken(TCardinal(_c4Size), L'1');
        strTmp.ReplaceToken(TCardinal(_c4MaxSize), L'2');
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_BadState
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , strTmp
        );
    }
}


// -----------------------------------------------------------------------------
//  TMemBuf: Private, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TMemBuf::__CheckIndex(   const   tCIDLib::TCard4 c4Line
                                        , const tCIDLib::TCard4 c4Index) const
{
    if (c4Index >= _c4MaxSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , c4Line
            , kCIDErrs::errcMBuf_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Index)
            , TCardinal(_c4MaxSize-1)
        );
    }

    //
    //  See if we need to reallocate. Add one to the index because the realloc
    //  wants a size, not an index.
    //
    if (c4Index >= _c4Size)
        ReallocateTo(c4Index+1);
}


tCIDLib::TVoid
TMemBuf::__CheckRange(  const tCIDLib::TCard4   c4Line
                        , const tCIDLib::TCard4 c4StartInd
                        ,       tCIDLib::TCard4 c4Count) const
{
    if (c4StartInd >= _c4MaxSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , c4Line
            , kCIDErrs::errcMBuf_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4StartInd)
            , TCardinal(_c4MaxSize-1)
        );
    }

    if (c4StartInd + c4Count > _c4MaxSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , c4Line
            , kCIDErrs::errcMBuf_BufOverflow
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(_c4MaxSize-1)
        );
    }

    //
    //  Now see if we need to reallocate the buffer. If any of the range
    //  would exceed the current allocation size, then we have to.
    //
    if (c4StartInd + c4Count >= _c4Size)
        ReallocateTo(c4StartInd + c4Count);
}




// -----------------------------------------------------------------------------
//   CLASS: THeapBuf
//  PREFIX: mbuf
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  THeapBuf: Constructors and Destructors
// -----------------------------------------------------------------------------

THeapBuf::THeapBuf( const   tCIDLib::TCard4 c4Size
                    , const tCIDLib::TCard4 c4MaxSize
                    , const tCIDLib::TCard4 c4ExpandIncrement) :

    TMemBuf(c4Size, c4MaxSize, c4ExpandIncrement, 0)
{
    // Allocate the buffer and store it away in TMemBuf
    _pc1Data = new tCIDLib::TCard1[c4Size];

    // Do the initial fill
    Set(_c1Fill, 0, _c4Size);
}

THeapBuf::THeapBuf(const THeapBuf& mbufSrc) :

    TMemBuf(mbufSrc)
{
    // Allocate a new buffer
    _pc1Data = new tCIDLib::TCard1[mbufSrc._c4Size];

    // Copy the data in from the source
    CopyIn(mbufSrc, mbufSrc._c4Size);
}

THeapBuf::~THeapBuf()
{
    if (_pc1Data)
    {
        delete _pc1Data;
        _ReleaseBuffer();
    }
}


// -----------------------------------------------------------------------------
//  THeapBuf: Public operators
// -----------------------------------------------------------------------------

THeapBuf& THeapBuf::operator=(const THeapBuf& mbufToAssign)
{
    if (this == &mbufToAssign)
        return *this;

    // Free the current buffer if not the same size 
    if (_c4Size != mbufToAssign._c4Size)
    {
        delete _pc1Data;
        _ReleaseBuffer();
    }

    // Copy over the members except for the buffer pointer
    _c1Fill = mbufToAssign._c1Fill;
    _c4ExpandIncrement = mbufToAssign._c4ExpandIncrement;
    _c4Size = mbufToAssign._c4Size;
    _c4MaxSize = mbufToAssign._c4MaxSize;

    // Allocate a buffer if we need to
    if (!_pc1Data)
        _pc1Data = new tCIDLib::TCard1[mbufToAssign._c4Size];

    // Copy the contents over and return a reference to ourself
    CopyIn(mbufToAssign, mbufToAssign._c4Size);
    return *this;
}


// -----------------------------------------------------------------------------
//  THeapBuf: Protected inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
THeapBuf::_ReallocateTo(const   tCIDLib::TCard4     c4NewSize
                        , const tCIDLib::TBoolean   bPreserve) const
{
    // Nothing to do
    if (c4NewSize == _c4Size)
        return;

    //
    //  Calculate the actual next size above the passed size. We have to
    //  do things in terms of the expansion increment.
    //
    tCIDLib::TCard4 c4ActualSize = c4NewSize;
    if (c4ActualSize % _c4ExpandIncrement)
        c4ActualSize += _c4ExpandIncrement - (c4ActualSize % _c4ExpandIncrement);

    // If this would overflow, then we want to clip it back.
    if (c4ActualSize > _c4MaxSize)
        c4ActualSize = _c4MaxSize;

    // Create the new buffer and protect it with a janitor
    tCIDLib::TCard1* const  pc1NewData = new tCIDLib::TCard1[c4ActualSize];
    THeapJanitor janNewData(pc1NewData);

    //
    //  Copy over the new info if we were told to. Else we want to just
    //  fill the old area.
    //
    if (bPreserve)
        TRawMem::CopyMemBuf(pc1NewData, _pc1Data, c4ActualSize);

    // Free the existing buffer now
    delete _pc1Data;
    _pc1Data = 0;

    //
    //  We need to do a fill with the fill character on the newly allocated
    //  bytes. If we were not told to preserve, then start at 0 and do the
    //  whole thing.
    //
    tCIDLib::TCard4 c4Index = _c4Size;
    if (!bPreserve)
        c4Index = 0;
    for (; c4Index < c4ActualSize; c4Index++)
        pc1NewData[c4Index] = _c1Fill;

    // Store the new size and pointer
    _c4Size = c4ActualSize;
    _pc1Data = pc1NewData;

    // Tell the janitor to orphan the new buffer
    janNewData.Orphan();
}

tCIDLib::TVoid THeapBuf::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Do our parent first
    TParent::_StreamFrom(strmToReadFrom);

    // Get the non-pointer members into temps
    tCIDLib::TCard4 c4NewSize;
    strmToReadFrom >> c4NewSize;

    //
    //  If the buffer is not currently the same size, then we need to realloc
    //  to the new size. So just delete and pretend like we never had one.
    //
    if (_pc1Data)
    {
        if (_c4Size != c4NewSize)
        {
            delete _pc1Data;
            _ReleaseBuffer();
        }
    }

    // Stream in the other non-pointer members and store the new size
    strmToReadFrom >> _c4MaxSize;
    strmToReadFrom >> _c4ExpandIncrement;
    strmToReadFrom >> _c1Fill;
    _c4Size = c4NewSize;

    // Validate the new members
    _ValidateMembers(L"THeapBuf::_StreamFrom", __LINE__);

    // If we need to allocate, then do it
    if (!_pc1Data)
    {
        _pc1Data = new tCIDLib::TCard1[c4NewSize];
        _c4Size = c4NewSize;
    }

    // And read the buffer contents in
    strmToReadFrom.ReadRawBuffer(_pc1Data, _c4Size);
}

tCIDLib::TVoid THeapBuf::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Do our parent first
    TParent::_StreamTo(strmToWriteTo);

    // Stream out the non-pointer members, then the buffer contents
    strmToWriteTo << _c4Size;
    strmToWriteTo << _c4MaxSize;
    strmToWriteTo << _c4ExpandIncrement;
    strmToWriteTo << _c1Fill;
    strmToWriteTo.WriteRawBuffer(_pc1Data, _c4Size);
}



// -----------------------------------------------------------------------------
//   CLASS: TSysBuf
//  PREFIX: mbuf
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TSysBuf: Constructors and Destructors
// -----------------------------------------------------------------------------

TSysBuf::TSysBuf(   const   tCIDLib::TCard4         c4Size
                    , const tCIDLib::EMemAccFlags   eAccessFlags) :

    TMemBuf
    (
        __c4RoundedUp(c4Size)
        , __c4RoundedUp(c4Size)
        , kCIDLib::c4MemPageSize
        , 0
    )
    , __eAccess(eAccessFlags)
{
    //
    //  Create the buffer. Because the max and current size are the same,
    //  it will effectively commit the whole thing.
    //
    __Create();
}

TSysBuf::TSysBuf(   const   tCIDLib::TCard4         c4Size
                    , const tCIDLib::TCard4         c4MaxSize
                    , const tCIDLib::EMemAccFlags   eAccessFlags) :

    TMemBuf
    (
        __c4RoundedUp(c4Size)
        , __c4RoundedUp(c4MaxSize)
        , kCIDLib::c4MemPageSize
        , 0
    )
    , __eAccess(eAccessFlags)
{
    //
    //  Create the buffer. Since the size and max size are not the same,
    //  only size amount of bytes (rounded to the next page) will actually
    //  be commited.
    //
    __Create();
}

TSysBuf::TSysBuf(const TSysBuf& mbufSrc) :

    TMemBuf(mbufSrc)
    , __eAccess(mbufSrc.__eAccess)
{
    // Create the buffer
    __Create();

    //
    //  Copy over the source buffer if they are not the same address.
    //  They could be because its a named shared buffer.
    //
    if (_pc1Data != mbufSrc._pc1Data)
        TRawMem::CopyMemBuf(_pc1Data, mbufSrc._pc1Data, mbufSrc._c4Size);
}

TSysBuf::~TSysBuf()
{
    if (_pc1Data)
    {
        try
        {
            TRawMem::FreeSysMem(_pc1Data);
        }

        catch(const TKrnlError& kerrToCatch)
        {
            // Log as a warning so we don't propogate out of here
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMBuf_FreeSys
                , kerrToCatch
                , tCIDLib::ESev_Warning
                , tCIDLib::EClass_CantDo
            );
        }
        _ReleaseBuffer();
    }
}


// -----------------------------------------------------------------------------
//  TSysBuf: Public operators
// -----------------------------------------------------------------------------

TSysBuf& TSysBuf::operator=(const TSysBuf& mbufToAssign)
{
    if (this == &mbufToAssign)
        return *this;

    // If there is an existing buffer, then free it
    if (_pc1Data)
    {
        try
        {
            TRawMem::FreeSysMem(_pc1Data);
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMBuf_FreeSys
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
        _ReleaseBuffer();
    }

    // Copy all the members over now
    _c1Fill             = mbufToAssign._c1Fill;
    _c4ExpandIncrement  = mbufToAssign._c4ExpandIncrement;
    _c4MaxSize          = mbufToAssign._c4MaxSize;
    _c4Size             = mbufToAssign._c4Size;
    __eAccess           = mbufToAssign.__eAccess;

    // Create the new buffer
    __Create();

    // Copy the data over to our new buffer
    TRawMem::CopyMemBuf
    (
        _pc1Data
        , mbufToAssign._pc1Data
        , mbufToAssign._c4Size
    );
    return *this;
}

tCIDLib::TBoolean TSysBuf::operator==(const TSysBuf& mbufToTest) const
{
    if (__eAccess != mbufToTest.__eAccess)
        return kCIDLib::False;

    // Call our parent's version now to compare the actual memory
    return TParent::operator==(mbufToTest);
}


// -----------------------------------------------------------------------------
//  TSysBuf: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TSysBuf::_ReallocateTo( const   tCIDLib::TCard4     c4NewSize
                        , const tCIDLib::TBoolean   bPreserve) const
{
    //
    //  Now commit the pages that this new expansion would cause. We first
    //  calculate the page that the current size falls onto. Note that we
    //  add 1 to get to the next page past the current one. We multiply
    //  it by the page size and add in the current buffer base to get the
    //  address of that page.
    //
    tCIDLib::TCard4 c4PageAdr = (_c4Size / kCIDLib::c4MemPageSize);
    c4PageAdr *= kCIDLib::c4MemPageSize;
    c4PageAdr += tCIDLib::TCard4(_pc1Data);

    //
    //  Now calculate how many pages we have to commit to get up to the
    //  new size. This is passed along to the commit function.
    //
    tCIDLib::TCard4 c4Pages = 1;
    if (_c4Size + kCIDLib::c4MemPageSize < c4NewSize) 
        c4Pages = (c4NewSize - _c4Size) / kCIDLib::c4MemPageSize;

    try
    {
        TRawMem::CommitPages((tCIDLib::TVoid*)c4PageAdr, c4Pages);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_Commit
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(_c4Size)
            , TCardinal(c4NewSize)
        );
    }

    //
    //  If they did not ask to preserve the current content, then we need
    //  to fill the whole buffer. Else we need to just fill the newly
    //  commited pages.
    //
    tCIDLib::TCard4 c4Index = _c4Size;
    if (!bPreserve)
        c4Index = 0;

    for (; c4Index < c4NewSize; c4Index++)
        _pc1Data[c4Index] = _c1Fill;

    // Store the new actual size now
    _c4Size = c4NewSize;
}

tCIDLib::TVoid TSysBuf::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    //  Delete the current buffer
    if (_pc1Data)
    {
        try
        {
            TRawMem::FreeSysMem(_pc1Data);
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMBuf_FreeSys
                , kerrToCatch
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
            );
        }
        _ReleaseBuffer();
    }

    // Do our parent first
    TParent::_StreamFrom(strmToReadFrom);

    // Get the new info out
    strmToReadFrom >> _c4Size;
    strmToReadFrom >> __eAccess;
    strmToReadFrom >> _c1Fill;
    strmToReadFrom >> _c4ExpandIncrement;
    strmToReadFrom >> _c4MaxSize;

    // Create the buffer
    __Create();

    // Now read in the actual data
    strmToReadFrom.ReadRawBuffer(_pc1Data, _c4Size);
}

tCIDLib::TVoid TSysBuf::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Do our parent first
    TParent::_StreamTo(strmToWriteTo);

    // Stream out our members and our parent's
    strmToWriteTo << _c4Size;
    strmToWriteTo << __eAccess;
    strmToWriteTo << _c1Fill;
    strmToWriteTo << _c4ExpandIncrement;
    strmToWriteTo << _c4MaxSize;

    // Now stream out the actual data
    strmToWriteTo.WriteRawBuffer(_pc1Data, _c4Size);
}


// -----------------------------------------------------------------------------
//  TSysBuf: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __c4Rounded
//
// DESCRIPTION:
//
//  Handles our commonly needed chore of rounding a value up to the next
//  page boundary.
// ---------------------------------------
//   INPUT: c4ToRound is the value to round
//
//  OUTPUT: None
//
//  RETURN: The rounded value
//
tCIDLib::TCard4 TSysBuf::__c4RoundedUp(const tCIDLib::TCard4 c4ToRound) const
{
    // if 0, then return 0
    if (!c4ToRound)
        return c4ToRound;

    // If less than the page size, then just return the page size
    if (c4ToRound < kCIDLib::c4MemPageSize)
        return kCIDLib::c4MemPageSize;

    tCIDLib::TCard4 c4Ret = c4ToRound / kCIDLib::c4MemPageSize;

    // If its already correct, then just return as is
    if (c4ToRound == (c4Ret * kCIDLib::c4MemPageSize))
        return c4ToRound;

    return (c4Ret +1 ) * kCIDLib::c4MemPageSize;
}


//
// FUNCTION/METHOD NAME: __Create
//
// DESCRIPTION:
//
//  This method will allocate the buffer. It will allocate the buffer to
//  the max size and commit it up to the current size.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TSysBuf::__Create()
{
    tCIDLib::TVoid*     pBuf = 0;

    // Do a sanity check on the buffer
    if (!_c4MaxSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_ZeroSize
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }

    //
    //  If the current and max size is the same, then we make it more
    //  efficient by reserving and commiting all at once. Otherwise we
    //  just reserve the max size and commit below.
    ///
    TRawMem::EMemAllFlags eAlloc = TRawMem::EReserve;

    if (_c4MaxSize == _c4Size)
        eAlloc = TRawMem::ECommit;

    try
    {
        pBuf = TRawMem::pAllocSysMem(_c4MaxSize, __eAccess, eAlloc);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_AllocSys
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(_c4MaxSize)
        );
    }

    // Store the buffer pointer
    _pc1Data = (tCIDLib::TCard1*)pBuf;

    //
    //  Now if there is any current size and its not the same as the max
    //  size, we want to commit that part of the buffer.
    //
    if (_c4Size && (_c4Size != _c4MaxSize))
    {
        try
        {
            TRawMem::CommitPages
            (
                pBuf
                , _c4Size / kCIDLib::c4MemPageSize
                , __eAccess
            );
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMBuf_Commit
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , TCardinal(0)
                , TCardinal(_c4Size)
            );
        }
    }
}


//
// FUNCTION/METHOD NAME: __pPageAdr
//
// DESCRIPTION:
//
//  Calculates the base page address of a given address.
// ---------------------------------------
//   INPUT: pAddress is the address to get the page for.
//
//  OUTPUT: None
//
//  RETURN: The base page address
//
tCIDLib::TVoid*
TSysBuf::__pPageAdr(const tCIDLib::TVoid* const pAddress) const
{
    tCIDLib::TCard4 c4Tmp = tCIDLib::TCard4(pAddress);
    c4Tmp = c4Tmp / kCIDLib::c4MemPageSize;
    return (tCIDLib::TVoid*)(c4Tmp * kCIDLib::c4MemPageSize);
}
