//
// NAME: CIDLib_Locale.Cpp
//
// DESCRIPTION:
//
//  This module implements the small amount of out of line code required
//  for the TLocale class.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 11/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TLocale,TObject)


// ----------------------------------------------------------------------------
//  Local data
//
//  __lociCache
//      A cache of the information on the currently set locale. It is updated
//      when the locale is changed.
//
//  __pcrsSync
//      A synchronization critical section used to protect the reading and
//      writing of locale info.
// ----------------------------------------------------------------------------
static TKrnlLocale::TLocaleInfo __lociCache;
static TCriticalSection*        __pcrsSync;



// ----------------------------------------------------------------------------
//  Intra-facility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid _InitTermLocale( const   tCIDLib::EInitTerm      eInitTerm
                                , const tCIDLib::EGlobalStates  eGlobals
                                , const TModule&                modInit
                                , const tCIDLib::TCard4         c4MaxChars
                                ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszReason = L"Unknown";

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Create the critical section used to sync read/write of
            //  locale information.
            //
            pszReason = modInit.pszLoadCIDMsg(kCIDMsgs::midLoc_CritSecCreate);
            __pcrsSync = new TCriticalSection;

            //
            //  And update our cache with the currently set locale, which
            //  was set up during the kernel DLL init.
            //
            TKrnlLocale::QueryLocaleInfo(__lociCache);
        }
    }

    catch(...)
    {
        TRawStr::CopyCatStr
        (
            pszFailReason
            , c4MaxChars
            , pszReason
            , L" failed"
        );
        throw;
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TLocale
//  PREFIX: loc
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TLocale: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TLocale::bNegSpacedSym()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.bNegSpacedSym;
}

tCIDLib::TBoolean TLocale::bNegSymPrecedes()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.bNegSymPrecedes;
}

tCIDLib::TBoolean TLocale::bPosSpacedSym()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.bPosSpacedSym;
}

tCIDLib::TBoolean TLocale::bPosSymPrecedes()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.bPosSymPrecedes;
}

tCIDLib::TCard4 TLocale::c4GroupSize()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.c4GroupSize;
}

tCIDLib::TCard4 TLocale::c4MonetaryDecDigits()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.c4MonetaryDecDigits;
}

tCIDLib::TCard4 TLocale::c4MonetaryGrpSize()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.c4MonetaryGrpSize;
}

tCIDLib::Tch TLocale::chDecimalSym()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.chDecimalSym;
}

tCIDLib::Tch TLocale::chThousandsSep()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.chThousandsSep;
}

tCIDLib::Tch TLocale::chMonetaryDecimalSym()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.chMonetaryDecimalSym;
}

tCIDLib::Tch TLocale::chMonetaryGrpSep()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.chMonetaryGrpSep;
}

tCIDLib::Tch TLocale::chCurrencySymbol()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.chCurrencySymbol;
}

tCIDLib::Tch TLocale::chPosSign()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.chPosSign;
}

tCIDLib::Tch TLocale::chNegSign()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.chNegSign;
}

tCIDLib::ESignPositions TLocale::eNegSignPosition()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.eNegSignPosition;
}

tCIDLib::ESignPositions TLocale::ePosSignPosition()
{
    TCritSecLock lockSync(__pcrsSync);
    return __lociCache.ePosSignPosition;
}

TString TLocale::strLocale()
{
    TCritSecLock lockSync(__pcrsSync);
    return TString(__lociCache.szLocale);
}

tCIDLib::TVoid TLocale::SetLocale(const TString& strLocale)
{
    TCritSecLock lockSync(__pcrsSync);

    try
    {
        // Set the locale for the application
        TKrnlLocale::SetLocale(strLocale.pszData());

        // And update our cache
        TKrnlLocale::QueryLocaleInfo(__lociCache);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcLoc_CantSet
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strLocale
        );
    }
}
