//
// NAME: CIDLib_KeyOps.Hpp
//
// DESCRIPTION:
//
//  There are some collection types whose elements are keys, or whose elements
//  contain key fields. The collection must do particular operations on those
//  key fields. In order to avoid hard coding these operations in, and forcing
//  every instantiation type to do it exactly the same way, 'key ops' classes
//  are provided which are derived from and given to the collection to provide
//  these key operations. This is a little more hassle and makes for a little
//  more overhead, but provides massively more flexibility.
//
//  TKeyOps is the basic key ops class for the key ops classes. It compares key
//  fields and hashes them. This is sufficient for THashMap and THashSet.
//
//  Since strings (and their derivatives) are very common key fields, a standard
//  derivative is provided strings, TStringKeyOps. It can be instantiated for
//  any type of string derived class.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/24/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//

#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TKeyOps
//  PREFIX: kops
// ----------------------------------------------------------------------------
template <class T> class TKeyOps : public TObject, public MDuplicable
{
    public :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TKeyOps()
        {
            __CheckFirstOfType();
        }

        TKeyOps(const TKeyOps<T>& kopsToCopy)
        {
            __CheckFirstOfType();
        }

        ~TKeyOps() {}


        // --------------------------------------------------------------------
        //  Public, virtual methods
        // --------------------------------------------------------------------
        virtual tCIDLib::TBoolean bCompKeys
        (
            const   T&              obj1
            , const T&              obj2
        )   const = 0;

        virtual tCIDLib::THashVal hshKey
        (
            const   T&              objToHash
            , const tCIDLib::TCard4 c4Modulus
        )   const = 0;


    private :        
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TKeyOps<T>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TKeyOpsFor", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TKeyOps<T>, TObject)
};

template <class T> tCIDLib::TBoolean TKeyOps<T>::__bFirstOfType = kCIDLib::False;
template <class T> const TClass TKeyOps<T>::clsThis;


// ----------------------------------------------------------------------------
//   CLASS: TStringKeyOps
//  PREFIX: kops
// ----------------------------------------------------------------------------
template <class T> class TStringKeyOps : public TKeyOps<T>
{
    public :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TStringKeyOps()
        {
            __CheckFirstOfType();
        }

        TStringKeyOps(const TStringKeyOps<T>& kopsToCopy)
        {
            __CheckFirstOfType();
        }

        ~TStringKeyOps() {}


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bCompKeys(const T& obj1, const T& obj2) const
        {
            return obj1.bSameText(obj2);
        }

        tCIDLib::THashVal
        hshKey(const T& objToHash, const tCIDLib::TCard4 c4Modulus) const
        {
            return objToHash.hshCalcHash(c4Modulus);
        }


    private :        
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TStringKeyOps<T>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TStringKeyOpsFor", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TStringKeyOps<T>, TKeyOps<T>)
        DefPolyDup(TStringKeyOps<T>)
};

template <class T> tCIDLib::TBoolean TStringKeyOps<T>::__bFirstOfType = kCIDLib::False;
template <class T> const TClass TStringKeyOps<T>::clsThis;

#pragma pack(pop)
