//
// NAME: CIDLib_HashSet.Hpp
//
// DESCRIPTION:
//
//  This is the public header for the THashSet class. This is a 'by value'
//  hashed collection. This type of collection cannot hold duplicate elements,
//  a fundamental attribute of set collections. So any attempt to add a
//  duplicate element will cause an exception.
//
//  THashSetNode is for internal use, and is what the actual data is
//  stored in.
//
//  Iteration order has no relationship to the order in which the nodes were
//  added to the table, and is unidirectional.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/02/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// ----------------------------------------------------------------------------
//  Forward reference some internal structures and classes
// ----------------------------------------------------------------------------
struct  THashSetList;
template <class TElem> class THashSetCursor;
template <class TElem> class THashSet;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: THashSetNode
//  PREFIX: node
// ----------------------------------------------------------------------------
template <class TElem> class THashSetNode
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        THashSetNode(const TElem& objData, THashSetNode<TElem>* pnodeNext) :

            __objData(objData)
            , __pnodeNext(pnodeNext)
        {
        }

        ~THashSetNode() {}

        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        TElem& objData()
        {
            return __objData;
        }

        const TElem& objData() const
        {
            return __objData;
        }

        THashSetNode<TElem>* pnodeNext()
        {
            return __pnodeNext;
        }

        THashSetNode<TElem>* pnodeNext(THashSetNode<TElem>* const pnodeToSet)
        {
            __pnodeNext = pnodeToSet;
            return __pnodeNext;
        }


    private :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class THashSet<TElem>;
        friend class THashSetCursor<TElem>;


        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        THashSetNode(const THashSetNode<TElem>&);

        tCIDLib::TVoid operator=(const THashSetNode<TElem>&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __objData
        //      This is the data for this node. Its a by value copy of the
        //      user's data.
        //
        //  __pnodeNext
        //      This is a pointer to the next node in the list that this
        //      node is part of.
        // --------------------------------------------------------------------
        TElem                   __objData;
        THashSetNode<TElem>*    __pnodeNext;
};



// ----------------------------------------------------------------------------
//   CLASS: THashSet
//  PREFIX: col
// ----------------------------------------------------------------------------
template <class TElem> class THashSet : public TCollection<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Nested typedefs for the cursor and node types used by a hash table
        // --------------------------------------------------------------------
        typedef class THashSetCursor<TElem> TCursor;
        typedef THashSetNode<TElem>         TNode;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        THashSet(   const   tCIDLib::TCard4         c4Modulus
                    ,       TKeyOps<TElem>* const   pkopsToAdopt
                    , const tCIDLib::TCard4         c4MaxElements = kCIDLib::c4MaxCard
                    , const tCIDLib::EMTStates      eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __aplstTable(0)
            , __c4CurElements(0)
            , __c4ListIndex(0)
            , __c4HashModulus(c4Modulus)
            , __c4MaxElements(c4MaxElements)
            , __pkopsToUse(pkopsToAdopt)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();

            // Allocate and initialize the list table
            __aplstTable = new TNode*[c4Modulus];
            TRawMem::SetMemBuf
            (
                __aplstTable
                , tCIDLib::TCard1(0)
                , sizeof(tCIDLib::TVoid*) * c4Modulus
            );
        }

        ~THashSet()
        {
            // Flush the list
            Flush();

            // Delete the comparator
            delete __pkopsToUse;

            // And delete the hash list table
            delete [] __aplstTable;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add(const TElem& objToAdd)
        {
            TLocker lockHashSet(this);

            // See if we've maxed out on elements
            if (__c4CurElements >= __c4MaxElements)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_TooManyNodes
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_OutResource
                );
            }

            // See if this element is already in the collection
            tCIDLib::THashVal hshElem;
            TNode* pnodeFind = __pnodeFind(objToAdd, hshElem);

            // If so, we cannot allow it
            if (pnodeFind)
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_DuplicateElem
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_Already
                );
            }

            //  Add it to the appropriate list. We just put it at the head
            //  since the order does not matter. We just construct the
            //  node and pass it the current head, which it will make its
            //  next node.
            //
            __aplstTable[hshElem] = new TNode(objToAdd, __aplstTable[hshElem]);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Bump up the element count
            __c4CurElements++;
        }

        tCIDLib::TBoolean bFlushCur()
        {
            TLocker lockHashSet(this);

            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Save the next node and the current list index
            const tCIDLib::TCard4 c4OldListIndex = __c4ListIndex;
            TNode* const pnodeNext = _pnodeNext(__pnodeCur, __c4ListIndex);

            // Now remove this element from the map
            __RemoveNode(__pnodeCur, c4OldListIndex);

            // Put the next node back into the iterator
            __pnodeCur = pnodeNext;

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bIsEmpty() const
        {
            TLocker lockHashSet(this);
            return (__c4CurElements == 0);
        }

        tCIDLib::TBoolean bNext()
        {
            TLocker lockHashSet(this);
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = _pnodeNext(__pnodeCur, __c4ListIndex);
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bResetIter()
        {
            TLocker lockHashSet(this);
            __pnodeCur = _pnodeFirst(__c4ListIndex);
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            TLocker lockHashSet(this);
             return __c4CurElements;
        }

        tCIDLib::TCard4 c4MaxElemCount() const
        {
            TLocker lockHashSet(this);
            return __c4MaxElements;
        }

        tCIDLib::TVoid Flush()
        {
            TLocker lockHashSet(this);

            // For each list, iterator it and delete its elements.
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4HashModulus; c4Index++)
            {
                if (!__aplstTable[c4Index])
                    continue;

                TNode* pnodeCur = __aplstTable[c4Index];
                while (pnodeCur)
                {
                    TNode* pnodeNext = pnodeCur->pnodeNext();
                    delete pnodeCur;
                    pnodeCur = pnodeNext;
                }
                __aplstTable[c4Index] = 0;
            }

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Now reset the needed members
             __c4CurElements = 0;
             __c4ListIndex = 0;
             __pnodeCur = 0;
        }

        TElem& objCur() const
        {
            TLocker lockHashSet(this);

            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }

        TElemCursor<TElem>* pcursNew() const
        {
            TLocker lockHashSet(this);
            return new TCursor(this);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bAddIfNew(const TElem& objToAdd)
        {
            TLocker lockHashSet(this);

            // See if the element exists
            tCIDLib::THashVal hshElem;
            TNode* pnodeFind = __pnodeFind(objToAdd, hshElem);

            if (pnodeFind)
                return kCIDLib::False;

            Add(objToAdd);
            return kCIDLib::True;
        }

        tCIDLib::TVoid FlushAt(TCursor& cursAt)
        {
            TLocker lockHashSet(this);

            // Make sure the cursor belongs to this bag
            if (!cursAt.bIsCursoring(*this))
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NotMyCursor
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , cursAt.clsIsA()
                    , clsIsA()
                );
            }

            // Get the node to flush
            TNode* pnodeToFlush = cursAt.pnodeCur();
            tCIDLib::TCard4 c4ListIndex = cursAt.c4ListIndex();

            // Make sure the cursor's valid
            if (!pnodeToFlush)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , cursAt.clsIsA()
                );
            }

            // Move the passed cursor up past this element
            cursAt.bNext();

            // Now remove this node
            __RemoveNode(pnodeToFlush, c4ListIndex);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // But we kept the passed cursor up to date by bumping it up
            cursAt._c4SerialNum(c4SerialNum());
        }

        tCIDLib::TCard4 c4HashModulus() const
        {
            TLocker lockHashSet(this);
            return __c4HashModulus;
        }

        TElem& objFind(const TElem& objToFind)
        {
            TLocker lockHashSet(this);

            tCIDLib::THashVal hshKey;
            TNode* pnodeFind = __pnodeFind(objToFind, hshKey);

            // Not found so return a nul reference
            if (!pnodeFind)
                return (*(TElem*)0);

            // we found it so return the object data
            return pnodeFind->objData();
        }

        const TElem& objFind(const TElem& objToFind) const
        {
            TLocker lockHashSet(this);

            tCIDLib::THashVal hshKey;
            TNode* pnodeFind = __pnodeFind(objToFind, hshKey);

            // Not found so return a nul reference
            if (!pnodeFind)
                return (*(TElem*)0);

            // we found it so return the object data
            return pnodeFind->objData();
        }

        tCIDLib::TVoid Remove(const TElem& objToRemove)
        {
            TLocker lockHashSet(this);

            // Get the hash of the element
            tCIDLib::THashVal hshElem = __pkopsToUse->hshKey
            (
                objToRemove
                , __c4HashModulus
            );

            TNode* pnodeFind = __aplstTable[hshElem];
            TNode* pnodePrev = 0;

            // Search the list if its not empty
            while (pnodeFind)
            {
                // If this key matches, then break out
                if (__pkopsToUse->bCompKeys(pnodeFind->__objData, objToRemove))
                    break;

                // Save the current as previous and move up
                pnodePrev = pnodeFind;
                pnodeFind = pnodeFind->pnodeNext();
            }

            if (!pnodeFind)
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NodeNotFound
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_NotFound
                );
            }

            //
            //  If the previous node is 0, then this is the head node,
            //  else we have to point the previous node out the find
            //  node's next node.
            //
            if (!pnodePrev)
                __aplstTable[hshElem] = pnodeFind->__pnodeNext;
            else
                pnodePrev->__pnodeNext = pnodeFind->__pnodeNext;

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // So we can now toast the found node and bump the element count
            delete pnodeFind;
            __c4CurElements--;
        }


    protected  :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class THashSetCursor<TElem>;

        friend tCIDLib::TBoolean bCompCollections
        (
            const   THashSet<TElem>&        col1
            , const THashSet<TElem>&        col2
            , const TObjEq<TElem>&          compToUse
        );

        friend tCIDLib::TVoid CopyCollectionNodes
        (
                    THashSet<TElem>&        colDest
            , const THashSet<TElem>&        colSource
        );

        friend THashSet<TElem>* pDupCollection
        (
            const   THashSet<TElem>&        colToDup
        );


        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        THashSet(const THashSet<TElem>& colToCopy) :

            TCollection<TElem>(colToCopy)
            , __aplstTable(0)
            , __c4CurElements(0)
            , __c4ListIndex(0)
            , __c4HashModulus(colToCopy.__c4HashModulus)
            , __c4MaxElements(colToCopy.__c4MaxElements)
            , __pkopsToUse(0)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();

            // Polymorphically duplicate the key ops object
            __pkopsToUse = ::pDupObject<TKeyOps<TElem> >
            (
                *colToCopy.__pkopsToUse
            );

            // Allocate and initialize the list table
            __aplstTable = new TNode*[__c4HashModulus];
            TRawMem::SetMemBuf
            (
                __aplstTable
                , tCIDLib::TCard1(0)
                , sizeof(tCIDLib::TVoid*) * __c4HashModulus
            );
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        TNode* _pnodeFirst(tCIDLib::TCard4& c4ListIndex) const
        {
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4HashModulus; c4Index++)
            {
                if (__aplstTable[c4Index])
                {
                    c4ListIndex = c4Index;
                    return __aplstTable[c4Index];
                }
            }
            return 0;
        }

        TNode* _pnodeNext(  THashSetNode<TElem>*    pnodeLast
                            , tCIDLib::TCard4&      c4ListIndex) const
        {
            // Move up to the next node
            TNode* __pnodeCur = pnodeLast->pnodeNext();
            if (__pnodeCur)
                return __pnodeCur;

            // Search subsequent lists for non-empty one
            while (1)
            {
                // If we hit the end of lists, then we are done
                c4ListIndex++;
                if (c4ListIndex == __c4HashModulus)
                    break;

                // If we find one that's not empty, we start there next
                if (__aplstTable[c4ListIndex])
                    return __aplstTable[c4ListIndex];
            }
            return 0;
        }


    private     :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        THashSet();

        tCIDLib::TVoid operator=(const THashSet<TElem>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"THashSetOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        TNode*
        __pnodeFind(const TElem& objToFind, tCIDLib::THashVal& hshElem) const
        {
            // Get the hash of the element
            hshElem = __pkopsToUse->hshKey(objToFind, __c4HashModulus);

            // If this list is empty, then obviously not here
            if (!__aplstTable[hshElem])
                return 0;

            // The list is not empty so check the list
            TNode* pnodeCur = __aplstTable[hshElem];
            while (pnodeCur)
            {
                if (__pkopsToUse->bCompKeys(pnodeCur->__objData, objToFind))
                    return pnodeCur;
                pnodeCur = pnodeCur->pnodeNext();
            }
            return 0;
        }

        tCIDLib::TVoid
        __RemoveNode(           THashSetNode<TElem>* const  pnodeToFlush
                        , const tCIDLib::TCard4             c4ListIndex)
        {
            TNode* pnodeFind = __aplstTable[c4ListIndex];
            TNode* pnodePrev = 0;

            //
            //  Loop though this list and find the passed node, remembering
            //  the previous node of each one so we can patch it out of the
            //  list.
            //
            while (pnodeFind)
            {
                // If this element matches, then we've found it
                if (pnodeToFlush == pnodeFind)
                    break;

                // Save the current as previous and move up
                pnodePrev = pnodeFind;
                pnodeFind = pnodeFind->pnodeNext();
            }

            if (!pnodeFind)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NodeNotFound
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_Internal
                );
                
            }

            //
            //  If the previous node is 0, then this is the head node,
            //  else we have to point the previous node out the find
            //  node's next node.
            //
            if (!pnodePrev)
                __aplstTable[c4ListIndex] = pnodeFind->__pnodeNext;
            else
                pnodePrev->__pnodeNext = pnodeFind->__pnodeNext;

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // So we can now toast the found node and bump the element count
            delete pnodeFind;
            __c4CurElements--;
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __aplstTable
        //      This is the array of hash table lists. Each list holds the
        //      elements that hash to that index. It is allocated to the
        //      size of the hash modulus.
        //
        //  __c4CurElements
        //      The current number of elements that the table has.
        //
        //  __c4HashModulus
        //      The modulus divisor for the hash. This is also the number
        //      elements allocated for the __alstTable array.
        //
        //  __c4MaxElements
        //      The maximum number of elements that the table can hold.
        //
        //  __c4ListIndex
        //  __pnodeCur
        //      The current list and node of the internal iterator. Just
        //      having a node pointer is not enough because we have to know
        //      whats the next list to go to when the current one is done.
        //
        //  __pkopsToUse
        //      The key ops object to use. It handles hashing and comparing
        //      the 'key' field, which is the whole object in this collection.
        // --------------------------------------------------------------------
        TNode**             __aplstTable;
        tCIDLib::TCard4     __c4CurElements;
        tCIDLib::TCard4     __c4ListIndex;
        tCIDLib::TCard4     __c4HashModulus;
        tCIDLib::TCard4     __c4MaxElements;
        TNode*              __pnodeCur;
        TKeyOps<TElem>*     __pkopsToUse;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a hash map of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(THashSet<TElem>, TCollection<TElem>)
};

template <class TElem> tCIDLib::TBoolean THashSet<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass THashSet<TElem>::clsThis;




// ----------------------------------------------------------------------------
//   CLASS: THashSetCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class THashSetCursor :

    public TElemCursor<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        THashSetCursor(const THashSet<TElem>* pcolToCursor) :

            TElemCursor<TElem>(pcolToCursor->c4SerialNum())
            , __c4CurList(0)
            , __pcolCursoring(pcolToCursor)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
            __pnodeCur = __pcolCursoring->_pnodeFirst(__c4CurList);
        }

        THashSetCursor(const THashSetCursor<TElem>& cursToCopy) :

            TElemCursor<TElem>(cursToCopy)
            , __c4CurList(cursToCopy.__c4CurList)
            , __pcolCursoring(cursToCopy.__pcolCursoring)
            , __pnodeCur(cursToCopy.__pnodeCur)
        {
            __CheckFirstOfType();
        }

        ~THashSetCursor() {}


        // --------------------------------------------------------------------
        //  Public, operator
        // --------------------------------------------------------------------
        THashSetCursor<TElem>& operator=(const THashSetCursor<TElem>& cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            TElemCursor<TElem>::operator=(cursToAssign);
            __c4CurList = cursToAssign.__c4CurList;
            __pcolCursoring = cursToAssign.__pcolCursoring;
            __pnodeCur = cursToAssign.__pnodeCur;

            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsEmpty() const
        {
            return __pcolCursoring->bIsEmpty();
        }

        tCIDLib::TBoolean bIsValid() const
        {
            // If the serial number is off, its not valid
            if (c4SerialNum() != __pcolCursoring->c4SerialNum())
                return kCIDLib::False;
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bNext()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            __pnodeCur = __pcolCursoring->_pnodeNext(__pnodeCur, __c4CurList);
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bReset()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            // Get our serial back in sync with the collection
            _c4SerialNum(__pcolCursoring->c4SerialNum());

            __pnodeCur = __pcolCursoring->_pnodeFirst(__c4CurList);
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            return __pcolCursoring->c4ElemCount();
        }

        const TElem& objCur() const
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean
        bIsCursoring(const THashSet<TElem>& colToCheck) const
        {
            return (__pcolCursoring == &colToCheck);
        }

        tCIDLib::TCard4 c4ListIndex() const
        {
            return __c4CurList;
        }

        const THashSetNode<TElem>* pnodeCur() const
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }

        THashSetNode<TElem>* pnodeCur()
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class THashSet<TElem>;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        THashSetCursor<TElem>();


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"THashSetCursorOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4CurList
        //      The index of the list within the table that we are cursoring.
        //      The hash table has multiple lists so we have to keep up with
        //      the one we are working on.
        //
        //  __pcolCursoring
        //      A pointer to the hash map collection that we are cursoring
        //
        //  __pnodeCur
        //      The current node in the hash table, 0 if not on a valid node
        //      at this time.
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4CurList;
        const THashSet<TElem>*  __pcolCursoring;
        THashSetNode<TElem>*    __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a deque of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(THashSetCursor<TElem>, TElemCursor<TElem>)
};

template <class TElem> tCIDLib::TBoolean THashSetCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass THashSetCursor<TElem>::clsThis;

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Some template functions to make duplication and copying of hash maps
//  typesafe.
// ----------------------------------------------------------------------------
template <class TElem> tCIDLib::TBoolean
bCompCollections(const  THashSet<TElem>&    col1
                , const THashSet<TElem>&    col2
                , const TObjEq<TElem>&      compToUse)
{
    // Check for compare against self
    if (&col1 == &col2)
        return kCIDLib::True;

    // Lock both of the hash maps
    TLocker lockHashSet1(&col1);
    TLocker lockHashSet2(&col2);

    if ((col1.__c4CurElements != col2.__c4CurElements)
    ||  (col1.__c4MaxElements != col2.__c4MaxElements)
    ||  (col1.__c4HashModulus != col2.__c4HashModulus))
    {
        return kCIDLib::False;
    }

    // Compare the elements
    for (tCIDLib::TCard4 c4ListInd = 0; c4ListInd < col1.__c4HashModulus; c4ListInd++)
    {
        // Get the nodes at the head of this list in each collection
        THashSet<TElem>::TNode* pnodeThis = col1.__aplstTable[c4ListInd];
        THashSet<TElem>::TNode* pnodeThat = col2.__aplstTable[c4ListInd];

        // If both null, then try next list
        if (!pnodeThis && !pnodeThat)
            continue;

        // So if one of them is null, no match
        if (!pnodeThis || !pnodeThat)
            return kCIDLib::False;

        // Compare the elements in this table
        while (1)
        {
            // If the data does not match, then not equal
            if (!compToUse.bEqual(pnodeThis->objData(), pnodeThat->objData()))
                return kCIDLib::False;

            // Move up both nodes
            pnodeThis = pnodeThis->pnodeNext();
            pnodeThat = pnodeThat->pnodeNext();

            // If both are null, then we can go on to the next list
            if (!pnodeThis && !pnodeThat)
                break;

            // If one is null, then no match
            if (!pnodeThis || !pnodeThat)
                return kCIDLib::False;
        }
    }

    // If we made it this far, they were equal
    return kCIDLib::True;
}


template <class TElem> tCIDLib::TVoid
CopyCollectionNodes(THashSet<TElem>& colDest, const THashSet<TElem>& colSource)
{
    // Check for copy to self
    if (&colDest == &colSource)
        return;

    // Lock both collections
    TLocker lockNew(&colDest);
    TLocker lockToDup(&colSource);

    if (colSource.c4ElemCount())
    {
        //
        //  Loop through all of the entries in the source collection's
        //  hash table and, for any that have entries, replicate them
        //  to the same place in the destination.
        //
        tCIDLib::TCard4 c4Index;
        for (c4Index = 0; c4Index < colSource.__c4HashModulus; c4Index++)
        {
            // If empty, do the next one
            if (!colSource.__aplstTable[c4Index])
                continue;

            THashSet<TElem>::TNode* pnodeCur = 0;
            THashSet<TElem>::TNode* pnodeLast = 0;
            THashSet<TElem>::TNode* pnodeSrc = colSource.__aplstTable[c4Index];
            while (pnodeSrc)
            {
                // Replicate this list's nodes
                pnodeCur = new THashSet<TElem>::TNode(pnodeSrc->objData(), 0);

                // Set last node's next to this one
                if (pnodeLast)
                    pnodeLast->pnodeNext(pnodeCur);

                //
                //  Set last node's next to this one, if there was a last.
                //  Else, save this as the head node in the dest.
                //
                if (pnodeLast)
                    pnodeLast->pnodeNext(pnodeCur);
                else
                    colDest.__aplstTable[c4Index] = pnodeCur;

                // Save current as last
                pnodeLast = pnodeCur;

                // And move to next source node
                pnodeSrc = pnodeSrc->pnodeNext();
            }
        }
    }

    // Set the destination element count
    colDest.__c4CurElements = colSource.__c4CurElements;
}


template <class TElem>
THashSet<TElem>* pDupCollection(const THashSet<TElem>& colToDup)
{
    // Lock the source collection
    TLocker lockToDup(&colToDup);

    // Create a new hash map with the same settings
    THashSet<TElem>* const pcolNew = new THashSet<TElem>(colToDup);
    TJanitor<THashSet<TElem> > janCol(pcolNew);

    CopyCollectionNodes(*pcolNew, colToDup);

    // Return a pointer to the new hash map, letting it go from the janitor
    janCol.Orphan();
    return pcolNew;
}
