//
// NAME: CIDLib_HashMap.Hpp
//
// DESCRIPTION:
//
//  This is the public header for the THashMap class. This is a 'by value'
//  keyed collection. Keyed means that there are two values stored for each
//  node, one is a key value and the other is the actual data that is
//  associated with that key. Elements are found by their key value, which is
//  very fast because the keys are hashed. Duplicate keys are not allowed
//  because this is a map, which maps unique keys to values.
//
//  The semantics of this class mean that the key type cannot be a fundamental
//  type, since it must provide specific methods. But it does not make sense
//  to try to hash fundamental types anyway, so its not a big deal.
//
//  There are certain operations related to the key field, which are hashing
//  it, comparing it with another key field object, and extracting the key
//  field from the data element. These are all handled by a THashMapKeyOps
//  class. A derivative of this class must be created for the key field and
//  a new object allocated and provided to each hash map. There is a standard
//  one for strings, called TStringKeyOps, since it is by far the most often
//  used hashed key field type. These are all in the CIDLib_KeyOps.Hpp file.
//
//  The element data and key classes must also provide copy constructors and
//  assignment operators.
//
//  THashMapNode is for internal use, and is what the actual data is stored in.
//
//  Iteration order has no relationship to the order in which the nodes where
//  added to the table and unidirectional only.
//
//  Note that these classes cannot use the RTTIMacros() macro because the
//  double template instantiation parameters confuse the macro processor and
//  make it think we are passing too many parameters. So it implements the
//  RTTI stuff manually. Oh well...
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/22/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// ----------------------------------------------------------------------------
//  Forward reference some internal structures and classes
// ----------------------------------------------------------------------------
struct  THashMapList;
template <class TElem,class TKey> class THashMapCursor;
template <class TElem,class TKey> class THashMap;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: THashMapNode
//  PREFIX: node
// ----------------------------------------------------------------------------
template <class TElem,class TKey> class THashMapNode
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        THashMapNode(const TElem& objData, THashMapNode<TElem,TKey>* pnodeNext) :

            __objData(objData)
            , __pnodeNext(pnodeNext)
        {
        }

        ~THashMapNode() {}

        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        const TElem& objData() const
        {
            return __objData;
        }

        TElem& objData()
        {
            return __objData;
        }

        THashMapNode<TElem,TKey>* pnodeNext()
        {
            return __pnodeNext;
        }

        THashMapNode<TElem,TKey>*
        pnodeNext(THashMapNode<TElem,TKey>* const pnodeToSet)
        {
            __pnodeNext = pnodeToSet;
            return __pnodeNext;
        }


    private :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class THashMap<TElem,TKey>;
        friend class THashMapCursor<TElem,TKey>;


        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        THashMapNode(const THashMapNode<TElem,TKey>&);

        tCIDLib::TVoid operator=(const THashMapNode<TElem,TKey>&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __objData
        //      This is the data for this node. Its a by value copy of the
        //      user's data. It contains the key field as well.
        //
        //  __pnodeNext
        //      This is a pointer to the next node in the list that this
        //      node is part of.
        // --------------------------------------------------------------------
        TElem                       __objData;
        THashMapNode<TElem,TKey>*   __pnodeNext;
};


// ----------------------------------------------------------------------------
//   CLASS: THashMap
//  PREFIX: col
// ----------------------------------------------------------------------------
template <class TElem, class TKey> class THashMap : public TCollection<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Public, static data
        // --------------------------------------------------------------------
        static const TClass clsThis;


        // --------------------------------------------------------------------
        //  Nested typedefs for the cursor, node, and key ops types used by
        //  a hash map. And a typedef for the key field extraction function
        //  that the user provides.
        // --------------------------------------------------------------------
        typedef class THashMapCursor<TElem,TKey>    TCursor;
        typedef THashMapNode<TElem,TKey>            TNode;
        typedef TKeyOps<TKey>                       TKOps;
        typedef const TKey& (*TKeyExtract)(const TElem&);


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        THashMap(   const   tCIDLib::TCard4     c4Modulus
                    ,       TKOps* const        pkopsToAdopt
                    , const TKeyExtract         pfnKeyExtract
                    , const tCIDLib::TCard4     c4MaxElements = kCIDLib::c4MaxCard
                    , const tCIDLib::EMTStates  eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __aplstTable(0)
            , __c4CurElements(0)
            , __c4ListIndex(0)
            , __c4HashModulus(c4Modulus)
            , __c4MaxElements(c4MaxElements)
            , __pfnKeyExtract(pfnKeyExtract)
            , __pkopsToUse(pkopsToAdopt)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();

            // Allocate and initialize the list table
            __aplstTable = new TNode*[c4Modulus];
            TRawMem::SetMemBuf
            (
                __aplstTable
                , tCIDLib::TCard1(0)
                , sizeof(tCIDLib::TVoid*) * c4Modulus
            );
        }

        ~THashMap()
        {
            // Flush the list
            Flush();

            // Delete the key ops object
            delete __pkopsToUse;

            // And delete the list itself
            delete [] __aplstTable;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add(const TElem& objToAdd)
        {
            TLocker lockHashMap(this);

            // See if we've maxed out on elements
            if (__c4CurElements >= __c4MaxElements)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_TooManyNodes
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_OutResource
                );
            }

            // See if this element is already in the collection
            tCIDLib::THashVal hshElem;
            TNode* pnodeFind = __pnodeFind(__pfnKeyExtract(objToAdd), hshElem);

            // If so, we cannot allow it
            if (pnodeFind)
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_DuplicateKey
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_Already
                    , __pfnKeyExtract(objToAdd)
                    , clsIsA()
                );
            }

            //  Add it to the appropriate list. We just put it at the head
            //  since the order does not matter. We just construct the
            //  node and pass it the current head, which it will make its
            //  next node.
            //
            __aplstTable[hshElem] = new TNode(objToAdd, __aplstTable[hshElem]);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Bump up the element count
            __c4CurElements++;
        }

        tCIDLib::TBoolean bIsDescendantOf(const TClass& clsTarget) const
        {
            if (clsTarget == clsThis)
                return kCIDLib::True;
            return TCollection<TElem>::bIsDescendantOf(clsTarget);
        }

        tCIDLib::TBoolean bFlushCur()
        {
            TLocker lockHashMap(this);

            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            //
            //  Save the next node before we destroy this one.
            TNode* const pnodeNext = _pnodeNext(__pnodeCur, __c4ListIndex);

            // Now remove this key from the map
            RemoveKey(__pfnKeyExtract(__pnodeCur->objData()));

            // Put the next node back into the iterator
            __pnodeCur = pnodeNext;

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bIsEmpty() const
        {
            TLocker lockHashMap(this);
            return (__c4CurElements == 0);
        }

        tCIDLib::TBoolean bNext()
        {
            TLocker lockHashMap(this);
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = _pnodeNext(__pnodeCur, __c4ListIndex);
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bResetIter()
        {
            TLocker lockHashMap(this);
            __pnodeCur = _pnodeFirst(__c4ListIndex);
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            TLocker lockHashMap(this);
             return __c4CurElements;
        }

        tCIDLib::TCard4 c4MaxElemCount() const
        {
            TLocker lockHashMap(this);
            return __c4MaxElements;
        }

        const TClass& clsIsA() const
        {
            return clsThis;
        }

        const TClass& clsParent() const
        {
            return TCollection<TElem>::clsThis;
        }

        tCIDLib::TVoid Flush()
        {
            TLocker lockHashMap(this);

            // For each list, iterator it and delete its elements.
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4HashModulus; c4Index++)
            {
                if (!__aplstTable[c4Index])
                    continue;

                TNode* pnodeCur = __aplstTable[c4Index];
                while (pnodeCur)
                {
                    TNode* pnodeNext = pnodeCur->pnodeNext();
                    delete pnodeCur;
                    pnodeCur = pnodeNext;
                }
                __aplstTable[c4Index] = 0;
            }

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Now reset the needed members
             __c4CurElements = 0;
             __c4ListIndex = 0;
             __pnodeCur = 0;
        }

        TElem& objCur() const
        {
            TLocker lockHashMap(this);

            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }

        TElemCursor<TElem>* pcursNew() const
        {
            TLocker lockHashMap(this);
            return new TCursor(this);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bAddOrUpdate(const TElem& objToAdd)
        {
            TLocker lockHashMap(this);

            // See if the element exists
            tCIDLib::THashVal hshElem;
            TNode* pnodeFind = __pnodeFind(__pfnKeyExtract(objToAdd), hshElem);

            if (pnodeFind)
            {
                pnodeFind->__objData = objToAdd;
                return kCIDLib::False;
            }

            Add(objToAdd);
            return kCIDLib::True;
        }

        tCIDLib::TBoolean bAddIfNew(const TElem& objToAdd)
        {
            TLocker lockHashSet(this);

            // See if the element exists
            tCIDLib::THashVal hshElem;
            TNode* pnodeFind = __pnodeFind(__pfnKeyExtract(objToAdd), hshElem);

            if (pnodeFind)
                return kCIDLib::False;

            Add(objToAdd);
            return kCIDLib::True;
        }

        tCIDLib::TBoolean bKeyExists(const TKey& keyToFind) const
        {
            TLocker lockHashSet(this);

            // See if the element exists
            tCIDLib::THashVal hshElem;
            TNode* pnodeFind = __pnodeFind(keyToFind, hshElem);

            return (pnodeFind != 0);
        }

        tCIDLib::TCard4 c4HashModulus() const
        {
            TLocker lockHashMap(this);
            return __c4HashModulus;
        }

        tCIDLib::TVoid FlushAt(TCursor& cursAt)
        {
            TLocker lockHashMap(this);

            // Make sure the cursor belongs to this bag
            if (!cursAt.bIsCursoring(*this))
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NotMyCursor
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , cursAt.clsIsA()
                    , clsIsA()
                );
            }

            // Get the node to flush
            TNode* pnodeToFlush = cursAt.pnodeCur();

            // Make sure the cursor's valid
            if (!pnodeToFlush)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Move the passed cursor up past this element
            cursAt.bNext();

            // Now remove this key
            RemoveKey(__pfnKeyExtract(pnodeToFlush->__objData));

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // But we kept the passed cursor up to date by bumping it up
            cursAt._c4SerialNum(c4SerialNum());
        }

        const TKey& objCurKey() const
        {
            TLocker lockHashMap(this);

            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pfnKeyExtract(__pnodeCur->__objData);
        }

        TElem& objFindByKey(const TKey& objKeyToFind)
        {
            TLocker lockHashMap(this);

            tCIDLib::THashVal hshKey;
            TNode* pnodeFind = __pnodeFind(objKeyToFind, hshKey);

            // Not found so return a nul reference
            if (!pnodeFind)
                return (*(TElem*)0);

            // we found it so return the object data
            return pnodeFind->objData();
        }

        const TElem& objFindByKey(const TKey& objKeyToFind) const
        {
            TLocker lockHashMap(this);

            tCIDLib::THashVal hshKey;
            TNode* pnodeFind = __pnodeFind(objKeyToFind, hshKey);

            // Not found so return a nul reference
            if (!pnodeFind)
                return (*(TElem*)0);

            // we found it so return the object data
            return pnodeFind->objData();
        }

        tCIDLib::TVoid RemoveKey(const TKey& objKeyToRemove)
        {
            TLocker lockHashMap(this);

            // Get the hash of the element
            tCIDLib::THashVal hshElem = __pkopsToUse->hshKey
            (
                objKeyToRemove
                , __c4HashModulus
            );

            TNode* pnodeFind = __aplstTable[hshElem];
            TNode* pnodePrev = 0;

            // Search the list if its not empty
            while (pnodeFind)
            {
                // If this key matches, then break out
                if (__pkopsToUse->bCompKeys
                (
                    __pfnKeyExtract(pnodeFind->__objData)
                    , objKeyToRemove))
                {
                    break;
                }

                // Save the current as previous and move up
                pnodePrev = pnodeFind;
                pnodeFind = pnodeFind->pnodeNext();
            }

            if (!pnodeFind)
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_KeyNotFound
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_NotFound
                    , objKeyToRemove
                    , clsIsA()
                );
            }

            //
            //  If the previous node is 0, then this is the head node,
            //  else we have to point the previous node out the find
            //  node's next node.
            //
            if (!pnodePrev)
                __aplstTable[hshElem] = pnodeFind->__pnodeNext;
            else
                pnodePrev->__pnodeNext = pnodeFind->__pnodeNext;

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // So we can now toast the found node and bump the element count
            delete pnodeFind;
            __c4CurElements--;
        }

        tCIDLib::TVoid ReplaceValue(const TElem& objNewValue)
        {
            TLocker lockHashMap(this);

            // See if this element is in the collection
            tCIDLib::THashVal hshElem;
            TNode* pnodeFind = __pnodeFind(__pfnKeyExtract(objNewValue), hshElem);

            if (!pnodeFind)
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_KeyNotFound
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_NotFound
                    , __pfnKeyExtract(objNewValue)
                    , clsIsA()
                );
            }

            // Copy in the new data
            pnodeFind->__objData = objNewValue;
        }


    protected  :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class THashMapCursor<TElem,TKey>;

        friend tCIDLib::TBoolean bCompCollections
        (
            const   THashMap<TElem,TKey>&   col1
            , const THashMap<TElem,TKey>&   col2
            , const TObjEq<TElem>&          compToUse
        );

        friend tCIDLib::TVoid CopyCollectionNodes
        (
                    THashMap<TElem,TKey>&   colDest
            , const THashMap<TElem,TKey>&   colSource
        );

        friend THashMap<TElem,TKey>* pDupCollection
        (
            const THashMap<TElem,TKey>&     colToDup
        );


        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        THashMap(const THashMap<TElem,TKey>& colToCopy) :

            TCollection<TElem>(colToCopy)
            , __aplstTable(0)
            , __c4CurElements(0)
            , __c4ListIndex(0)
            , __c4HashModulus(colToCopy.__c4HashModulus)
            , __c4MaxElements(colToCopy.__c4MaxElements)
            , __pfnKeyExtract(colToCopy.__pfnKeyExtract)
            , __pkopsToUse(0)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();

            // Polymorphically dup the key objects object
            __pkopsToUse = ::pDupObject<TKOps>
            (
                *colToCopy.__pkopsToUse
            );

            // Allocate and initialize the list table
            __aplstTable = new TNode*[__c4HashModulus];
            TRawMem::SetMemBuf
            (
                __aplstTable
                , tCIDLib::TCard1(0)
                , sizeof(tCIDLib::TVoid*) * __c4HashModulus
            );
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        TNode* _pnodeFirst(tCIDLib::TCard4& c4ListIndex) const
        {
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4HashModulus; c4Index++)
            {
                if (__aplstTable[c4Index])
                {
                    c4ListIndex = c4Index;
                    return __aplstTable[c4Index];
                }
            }
            return 0;
        }

        TNode* _pnodeNext(  THashMapNode<TElem,TKey>*   pnodeLast
                            , tCIDLib::TCard4&          c4ListIndex) const
        {
            // Move up to the next node
            TNode* __pnodeCur = pnodeLast->pnodeNext();
            if (__pnodeCur)
                return __pnodeCur;

            // Search subsequent lists for non-empty one
            while (1)
            {
                // If we hit the end of lists, then we are done
                c4ListIndex++;
                if (c4ListIndex == __c4HashModulus)
                    break;

                // If we find one that's not empty, we start there next
                if (__aplstTable[c4ListIndex])
                    return __aplstTable[c4ListIndex];
            }
            return 0;
        }


    private     :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"THashMapOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        THashMap();

        tCIDLib::TVoid operator=(const THashMap<TElem,TKey>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        TNode*
        __pnodeFind(const TKey& objKeyToFind, tCIDLib::THashVal& hshElem) const
        {
            // Get the hash of the element
            hshElem = __pkopsToUse->hshKey
            (
                objKeyToFind
                , __c4HashModulus
            );

            // If this list is empty, then obviously not here
            if (!__aplstTable[hshElem])
                return 0;

            // The list is not empty so check the list
            TNode* pnodeCur = __aplstTable[hshElem];
            while (pnodeCur)
            {
                if (__pkopsToUse->bCompKeys
                (
                    __pfnKeyExtract(pnodeCur->__objData)
                    , objKeyToFind))
                {
                    return pnodeCur;
                }
                pnodeCur = pnodeCur->pnodeNext();
            }
            return 0;
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __aplstTable
        //      This is the array of hash table lists. Each list holds the
        //      elements that hash to that index. It is allocated to the
        //      size of the hash modulus.
        //
        //  __c4CurElements
        //      The current number of elements that the table has.
        //
        //  __c4HashModulus
        //      The modulus divisor for the hash. This is also the number
        //      elements allocated for the __alstTable array.
        //
        //  __c4ListIndex
        //      The current list of the hash table that the internal iterator
        //      is on now.
        //
        //  __c4MaxElements
        //      The maximum number of elements that the table can hold.
        //
        //  __pfnKeyExtract
        //      The key extraction function provided by the user. It pulls
        //      out a reference to the key field from the data field.
        //
        //  __pkopsToUse
        //      A key ops object that provides all of the operations that
        //      we have to do on key field objects. We own and destruct it
        //      when we destruct.
        //
        //  __pnodeCur
        //      The current node of the internal iterator. Just having a node
        //      pointer is not enough because we have to know whats the next
        //      list to go to when the current one is done. So this works in
        //      conjunction with __c4ListIndex.
        // --------------------------------------------------------------------
        TNode**             __aplstTable;
        tCIDLib::TCard4     __c4CurElements;
        tCIDLib::TCard4     __c4ListIndex;
        tCIDLib::TCard4     __c4HashModulus;
        tCIDLib::TCard4     __c4MaxElements;
        TKeyExtract         __pfnKeyExtract;
        TKOps*              __pkopsToUse;
        TNode*              __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a hash map of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;
};

template <class TElem, class TKey> tCIDLib::TBoolean THashMap<TElem,TKey>::__bFirstOfType = kCIDLib::False;
template <class TElem, class TKey> const TClass THashMap<TElem,TKey>::clsThis;




// ----------------------------------------------------------------------------
//   CLASS: THashMapCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem, class TKey> class THashMapCursor :

    public TElemCursor<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Static data
        // --------------------------------------------------------------------
        static const TClass clsThis;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        THashMapCursor(const THashMap<TElem,TKey>* pcolToCursor) :

            TElemCursor<TElem>(pcolToCursor->c4SerialNum())
            , __c4CurList(0)
            , __pcolCursoring(pcolToCursor)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
            __pnodeCur = __pcolCursoring->_pnodeFirst(__c4CurList);
        }

        THashMapCursor(const THashMapCursor<TElem,TKey>& cursToCopy) :

            TElemCursor<TElem>(cursToCopy)
            , __c4CurList(cursToCopy.__c4CurList)
            , __pcolCursoring(cursToCopy.__pcolCursoring)
            , __pnodeCur(cursToCopy.__pnodeCur)
        {
            __CheckFirstOfType();
        }

        ~THashMapCursor() {}


        // --------------------------------------------------------------------
        //  Pubilc operators
        // --------------------------------------------------------------------
        THashMapCursor<TElem, TKey>&
        operator=(const THashMapCursor<TElem,TKey>& cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            TElemCursor<TElem>::operator=(cursToAssign);
            __c4CurList = cursToAssign.__c4CurList;
            __pcolCursoring = cursToAssign.__pcolCursoring;
            __pnodeCur = cursToAssign.__pnodeCur;

            return *this;
        }

        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsDescendantOf(const TClass& clsTarget) const
        {
            if (clsTarget == clsThis)
                return kCIDLib::True;
            return TElemCursor<TElem>::bIsDescendantOf(clsTarget);
        }

        tCIDLib::TBoolean bIsEmpty() const
        {
            return __pcolCursoring->bIsEmpty();
        }

        tCIDLib::TBoolean bIsValid() const
        {
            // If the serial number is off, its not valid
            if (c4SerialNum() != __pcolCursoring->c4SerialNum())
                return kCIDLib::False;
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bNext()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            __pnodeCur = __pcolCursoring->_pnodeNext(__pnodeCur, __c4CurList);
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bReset()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            // Get our serial back in sync with the collection
            _c4SerialNum(__pcolCursoring->c4SerialNum());

            __pnodeCur = __pcolCursoring->_pnodeFirst(__c4CurList);
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            return __pcolCursoring->c4ElemCount();
        }

        const TClass& clsIsA() const
        {
            return clsThis;
        }

        const TClass& clsParent() const
        {
            return TElemCursor<TElem>::clsThis;
        }

        const TElem& objCur() const
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean
        bIsCursoring(const THashMap<TElem,TKey>& colToCheck) const
        {
            return (__pcolCursoring == &colToCheck);
        }

        const THashMapNode<TElem,TKey>* pnodeCur() const
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }

        THashMapNode<TElem,TKey>* pnodeCur()
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class THashMap<TElem,TKey>;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        THashMapCursor<TElem,TKey>();


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"THashMapCursorOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4CurList
        //      The index of the list within the table that we are cursoring.
        //      The hash table has multiple lists so we have to keep up with
        //      the one we are working on.
        //
        //  __pcolCursoring
        //      A pointer to the hash map collection that we are cursoring
        //
        //  __pnodeCur
        //      The current node in the hash table, 0 if not on a valid node
        //      at this time.
        // --------------------------------------------------------------------
        tCIDLib::TCard4             __c4CurList;
        const THashMap<TElem,TKey>* __pcolCursoring;
        THashMapNode<TElem,TKey>*   __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a deque of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;
};

template <class TElem, class TKey> tCIDLib::TBoolean THashMapCursor<TElem,TKey>::__bFirstOfType = kCIDLib::False;
template <class TElem, class TKey> const TClass THashMapCursor<TElem,TKey>::clsThis;

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Some template functions to make duplication and copying of hash maps
//  typesafe.
// ----------------------------------------------------------------------------
template <class TElem,class TKey> tCIDLib::TBoolean
bCompCollections(const  THashMap<TElem,TKey>&   col1
                , const THashMap<TElem,TKey>&   col2
                , const TObjEq<TElem>&          compToUse)
{
    // Check for compare against self
    if (&col1 == &col2)
        return kCIDLib::True;

    // Lock both of the hash maps
    TLocker lockHashMap1(&col1);
    TLocker lockHashMap2(&col2);

    if ((col1.__c4CurElements != col2.__c4CurElements)
    ||  (col1.__c4MaxElements != col2.__c4MaxElements)
    ||  (col1.__c4HashModulus != col2.__c4HashModulus))
    {
        return kCIDLib::False;
    }

    // Compare the elements
    for (tCIDLib::TCard4 c4ListInd = 0; c4ListInd < col1.__c4HashModulus; c4ListInd++)
    {
        // Get the nodes at the head of this list in each collection
        THashMap<TElem,TKey>::TNode* pnodeThis = col1.__aplstTable[c4ListInd];
        THashMap<TElem,TKey>::TNode* pnodeThat = col2.__aplstTable[c4ListInd];

        // If both null, then try next list
        if (!pnodeThis && !pnodeThat)
            continue;

        // So if one of them is null, no match
        if (!pnodeThis || !pnodeThat)
            return kCIDLib::False;

        // Compare the elements in this table
        while (1)
        {
            // If the data does not match, then not equal
            if (!compToUse.bEqual(pnodeThis->objData(), pnodeThat->objData()))
                return kCIDLib::False;

            // Move up both nodes
            pnodeThis = pnodeThis->pnodeNext();
            pnodeThat = pnodeThat->pnodeNext();

            // If both are null, then we can go on to the next list
            if (!pnodeThis && !pnodeThat)
                break;

            // If one is null, then no match
            if (!pnodeThis || !pnodeThat)
                return kCIDLib::False;
        }
    }

    // If we made it this far, they were equal
    return kCIDLib::True;
}

template <class TElem, class TKey>
tCIDLib::TVoid CopyCollectionNodes
(
            THashMap<TElem,TKey>&   colDest
    , const THashMap<TElem,TKey>&   colSource)
{
    // Check for copy to self
    if (&colDest == &colSource)
        return;

    // Lock both collections
    TLocker lockNew(&colDest);
    TLocker lockToDup(&colSource);

    if (colSource.c4ElemCount())
    {
        //
        //  Loop through all of the entries in the source collection's
        //  hash table and, for any that have entries, replicate them
        //  to the same place in the destination.
        //
        tCIDLib::TCard4 c4Index;
        for (c4Index = 0; c4Index < colSource.__c4HashModulus; c4Index++)
        {
            // If empty, do the next one
            if (!colSource.__aplstTable[c4Index])
                continue;

            THashMap<TElem,TKey>::TNode* pnodeCur = 0;
            THashMap<TElem,TKey>::TNode* pnodeLast = 0;
            THashMap<TElem,TKey>::TNode* pnodeSrc = colSource.__aplstTable[c4Index];
            while (pnodeSrc)
            {
                // Replicate this list's nodes
                pnodeCur = new THashMap<TElem,TKey>::TNode
                (
                    pnodeSrc->objData()
                    , 0
                );

                //
                //  Set last node's next to this one, if there was a last.
                //  Else, save this as the head node in the dest.
                //
                if (pnodeLast)
                    pnodeLast->pnodeNext(pnodeCur);
                else
                    colDest.__aplstTable[c4Index] = pnodeCur;

                // Save current as last
                pnodeLast = pnodeCur;

                // And move to next source node
                pnodeSrc = pnodeSrc->pnodeNext();
            }
        }
    }

    // Set the destination element count
    colDest.__c4CurElements = colSource.__c4CurElements;
}

template <class TElem, class TKey>
THashMap<TElem,TKey>* pDupCollection(const THashMap<TElem,TKey>& colToDup)
{
    // Lock the source collection
    TLocker lockToDup(&colToDup);

    // Create a new hash map with the same settings
    THashMap<TElem,TKey>* const pcolNew = new THashMap<TElem,TKey>(colToDup);
    TJanitor<THashMap<TElem,TKey> > janCol(pcolNew);

    CopyCollectionNodes(*pcolNew, colToDup);

    // Return a pointer to the new hash map, letting it go from the janitor
    janCol.Orphan();
    return pcolNew;
}
