//
// NAME: CIDLib_FindBuf.Hpp
//
// DESCRIPTION:
//
//  This module implements the TFindBuf class, which is basically a wrapper
//  for the TKrnlFileSys::TRawFileFind kernel structure. So one TFindBuf
//  object represents one found file. TFileSys provides methods to build
//  them easily.
//
//  When formatting find buffers, you use a set of standard replacement
//  tokens (the same way that TTime and others are formatted".) This is
//  the most flexible way possible to do it. Each one has a default
//  format of just the file name. A couple of common ones are provided as
//  static members.
//
//  The token characters are:
//
//      'P'     - The path to the file
//      'N'     - The root part of the file name (with extension)
//      'S'     - The file size
//      'C'     - The create time
//      'c'     - The create date
//      'L'     - The last access time
//      'l'     - The last access date
//      'M'     - The last modify time
//      'm'     - The last modify date
//      'A'     - The attributes
//      'F'     - The full path and name together
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 06/12/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TFindBuf
//  PREFIX: fndb
// ----------------------------------------------------------------------------
class CIDLIBEXP TFindBuf :

    public TObject, public MDuplicable, public MStreamable, public MFormattable
{
    public  :
        // --------------------------------------------------------------------
        //  Public static methods
        //
        //  These are some prefab format strings for commonly used formats
        //  of a find buffer.
        // --------------------------------------------------------------------
        static const TString    strFull;
        static const TString    strNameAndSize;
        static const TString    strNameAndAttrs;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TFindBuf();

        TFindBuf
        (
            const   TString&                strDefFormat
        );

        TFindBuf
        (
            const   TFindBuf&               fndbToCopy
        );

        ~TFindBuf();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator<
        (
            const   TFindBuf&               fndbToTest
        )   const;

        tCIDLib::TBoolean operator>
        (
            const   TFindBuf&               fndbToTest
        )   const;

        tCIDLib::TBoolean operator==
        (
            const   TFindBuf&               fndbToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TFindBuf&               fndbToTest
        )   const;

        TFindBuf& operator=
        (
            const   TFindBuf&               fndbToAssign
        );

        operator tCIDLib::EFileAttrs() const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsArchived() const;

        tCIDLib::TBoolean bIsDirectory() const;

        tCIDLib::TBoolean bIsHidden() const;

        tCIDLib::TBoolean bIsReadOnly() const;

        tCIDLib::TBoolean bIsSystem() const;

        tCIDLib::TBoolean bIsSpecialDir() const;

        tCIDLib::EFileAttrs eAttrs() const;

        TBag<TFindBuf>& colChildren();

        tCIDLib::TVoid FormatTo
        (
                    TTextStream&            strmToWriteTo
            , const TString&                strFormat
        )   const;

        tCIDLib::TFilePos fposSize() const;

        const TPathStr& pathFileName() const;

        const TTime& tmCreate() const;

        const TTime& tmLastAccess() const;

        const TTime& tmLastModify() const;

        const TString& strDefFormat() const;

        const TString& strDefFormat
        (
            const   TString&                strNewDef
        );


    protected   :
        // --------------------------------------------------------------------
        //  Declare TFileSys and TDirIter to be friends
        // --------------------------------------------------------------------
        friend  class   TFileSys;
        friend  class   TDirIter;


        // --------------------------------------------------------------------
        //  Hidden constructors/operators
        // --------------------------------------------------------------------
        TFindBuf
        (
            const   TKrnlFileSys::TRawFileFind& rfndSrc
            , const TString&                strPath
        );


        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FromFindBuf
        (
            const   TKrnlFileSys::TRawFileFind& rfndSrc
            , const TString&                strPath
        );


    private         :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __ReplicateChildren
        (
            const   TFindBuf&               fndSrc
        );


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __eAttrs
        //      These are the standard attributes flags of the found file.
        //
        //  __fposSize
        //      This is the size of the file in bytes.
        //
        //  __pathFileName
        //      The name of the file found. Just the file name, it does not
        //      include the path or drive.
        //
        //  __pcolChildren
        //      When doing deep searches, if this node is a directory, this
        //      collection will contain all of the nodes for this directory's
        //      children that matched the search spec. Its a pointer so it
        //      has no cost unless its used. It is faulted in as needed.
        //
        //  __tmCreate
        //  __tmLastAccess
        //  __tmLastModify
        //      The create, last access, and last modification times
        //
        //  __strDefFormat
        //      This is the default formatting string. Used if one is not
        //      provided.
        // --------------------------------------------------------------------
        tCIDLib::EFileAttrs     __eAttrs;
        tCIDLib::TFilePos       __fposSize;
        TPathStr                __pathFileName;
        TBag<TFindBuf>*         __pcolChildren;
        TTime                   __tmCreate;
        TTime                   __tmLastAccess;
        TTime                   __tmLastModify;
        TString                 __strDefFormat;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFindBuf, TObject)
        DefPolyDup(TFindBuf)
};

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TFindBuf::operator<(const TFindBuf& fndbComp) const
{
    return __pathFileName < fndbComp.__pathFileName;
}

inline tCIDLib::TBoolean TFindBuf::operator>(const TFindBuf& fndbComp) const
{
    return __pathFileName > fndbComp.__pathFileName;
}

inline tCIDLib::TBoolean
TFindBuf::operator!=(const TFindBuf& fndbToTest) const
{
    return !operator==(fndbToTest);
}

inline TFindBuf::operator tCIDLib::EFileAttrs() const
{
    return __eAttrs;
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TFindBuf::bIsArchived() const
{
    return (__eAttrs & tCIDLib::EFileAttr_Archive) ?
                                kCIDLib::True : kCIDLib::False;
}

inline tCIDLib::TBoolean TFindBuf::bIsDirectory() const
{
    return (__eAttrs & tCIDLib::EFileAttr_Directory) ?
                                kCIDLib::True : kCIDLib::False;
}

inline tCIDLib::TBoolean TFindBuf::bIsHidden() const
{
    return (__eAttrs & tCIDLib::EFileAttr_Hidden) ?
                                kCIDLib::True : kCIDLib::False;
}

inline tCIDLib::TBoolean TFindBuf::bIsReadOnly() const
{
    return (__eAttrs & tCIDLib::EFileAttr_ReadOnly) ?
                                kCIDLib::True : kCIDLib::False;
}

inline tCIDLib::TBoolean TFindBuf::bIsSystem() const
{
    return (__eAttrs & tCIDLib::EFileAttr_System) ?
                                kCIDLib::True : kCIDLib::False;
}

inline tCIDLib::EFileAttrs TFindBuf::eAttrs() const
{
    return __eAttrs;
}

inline tCIDLib::TFilePos TFindBuf::fposSize() const
{
    return __fposSize;
}

inline const TPathStr& TFindBuf::pathFileName() const
{
    return __pathFileName;
}

inline const TTime& TFindBuf::tmCreate() const
{
    return __tmCreate;
}

inline const TTime& TFindBuf::tmLastAccess() const
{
    return __tmLastAccess;
}

inline const TTime& TFindBuf::tmLastModify() const
{
    return __tmLastModify;
}

inline const TString& TFindBuf::strDefFormat() const
{
    return __strDefFormat;
}

inline const TString& TFindBuf::strDefFormat(const TString& strNewDef)
{
    __strDefFormat = strNewDef;
    return __strDefFormat;
}
