//
// NAME: CIDLib_FindBuf.Cpp
//
// DESCRIPTION:
//
//  This module implements the TFindBuf class, which represents a single
//  found file during a directory search.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/13/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"

// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TFindBuf,TObject)


// ----------------------------------------------------------------------------
//  Local static data
// ----------------------------------------------------------------------------
static const tCIDLib::Tch* const __pszDefFormat = L"%(m) %(M) %(A,6) %(S,10) %(N)";


// ----------------------------------------------------------------------------
//   CLASS: TFindBuf
//  PREFIX: fnd
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TFindBuf: Public, static data
// ----------------------------------------------------------------------------
const TString   TFindBuf::strFull           = __pszDefFormat;
const TString   TFindBuf::strNameAndSize    = L"%(S,10) %(P)";
const TString   TFindBuf::strNameAndAttrs   = L"%(A,6) %(P)";


// ----------------------------------------------------------------------------
//  TFindBuf: Constructors and destructors
// ----------------------------------------------------------------------------

TFindBuf::TFindBuf() :

    __eAttrs(tCIDLib::EFileAttr_None)
    , __fposSize(0)
    , __pathFileName(kCIDLib::pszEmptyZStr)
    , __pcolChildren(0)
    , __strDefFormat(__pszDefFormat)
{
}

TFindBuf::TFindBuf(const TString& strDefFormat) :

    __eAttrs(tCIDLib::EFileAttr_None)
    , __fposSize(0)
    , __pathFileName(kCIDLib::pszEmptyZStr)
    , __pcolChildren(0)
    , __strDefFormat(strDefFormat)
{
}

TFindBuf::TFindBuf(const TFindBuf& fndbSrc) :

    __eAttrs(fndbSrc.__eAttrs)
    , __fposSize(fndbSrc.__fposSize)
    , __pathFileName(fndbSrc.__pathFileName)
    , __pcolChildren(0)
    , __tmCreate(fndbSrc.__tmCreate)
    , __tmLastAccess(fndbSrc.__tmLastAccess)
    , __tmLastModify(fndbSrc.__tmLastModify)
    , __strDefFormat(fndbSrc.__strDefFormat)
{
    // If no children in the source, then we are done
    if (!fndbSrc.__pcolChildren)
        return;

    if (fndbSrc.__pcolChildren->bIsEmpty())
        return;

    // We have children so fault in the bag
    __pcolChildren = new TBag<TFindBuf>;

    TBag<TFindBuf>::TCursor cursChildren(fndbSrc.__pcolChildren);
    do
    {
        //
        //  Add a copy of the current child to our collection. Note that
        //  this will cause a copy constructor, which will cause us to
        //  be recursively called to create the entire nested structure
        //  of find buffer objects.
        //
        __pcolChildren->Add(cursChildren.objCur());

    }   while (cursChildren.bNext());
}

TFindBuf::~TFindBuf()
{
    delete __pcolChildren;
    __pcolChildren = 0;
}


// ----------------------------------------------------------------------------
//  TFindBuf: Public operators
// ----------------------------------------------------------------------------

TFindBuf& TFindBuf::operator=(const TFindBuf& fndbSrc)
{
    if (this == &fndbSrc)
        return *this;

    // Copy over everything but the child list
    __eAttrs        = fndbSrc.__eAttrs;
    __fposSize      = fndbSrc.__fposSize;
    __pathFileName  = fndbSrc.__pathFileName;
    __tmCreate      = fndbSrc.__tmCreate;
    __tmLastAccess  = fndbSrc.__tmLastAccess;
    __tmLastModify  = fndbSrc.__tmLastModify;
    __strDefFormat  = fndbSrc.__strDefFormat;

    // If there are children in the source, then copy them also
    if (fndbSrc.__pcolChildren && !fndbSrc.__pcolChildren->bIsEmpty())
    {
        if (!__pcolChildren)
            __pcolChildren = new TBag<TFindBuf>;
        else
            __pcolChildren->Flush();

        TBag<TFindBuf>::TCursor cursNew(fndbSrc.__pcolChildren);
        do
        {
            __pcolChildren->Add(cursNew.objCur());
        }   while (cursNew.bNext());
    }
    return *this;
}


tCIDLib::TBoolean TFindBuf::operator==(const TFindBuf& fndbToTest) const
{
    // Handle the fundamental types first
    if (__eAttrs != fndbToTest.__eAttrs)
        return kCIDLib::False;

    if (__fposSize != fndbToTest.__fposSize)
        return kCIDLib::False;

    //
    //  We only test the text of the name, not all of the string attributes,
    //  which would happen if we just did an equality test.
    //
    if (__pathFileName.eCompare(fndbToTest.__pathFileName))
        return kCIDLib::False;

    if (__tmCreate != fndbToTest.__tmCreate)
        return kCIDLib::False;

    if (__tmLastAccess != fndbToTest.__tmLastAccess)
        return kCIDLib::False;

    if (__tmLastModify != fndbToTest.__tmLastModify)
        return kCIDLib::False;

    if (__strDefFormat != fndbToTest.__strDefFormat)
        return kCIDLib::False;

    //
    //  If one of us has children and the other does not, then obviously
    //  they are not equal.
    //
    if (!__pcolChildren != !fndbToTest.__pcolChildren)
        return kCIDLib::False;

    // Compare the children if any
    if (__pcolChildren)
    {
        // If the element counts are not equal, don't bother
        if (__pcolChildren->c4ElemCount() != fndbToTest.__pcolChildren->c4ElemCount())
            return kCIDLib::False;

        TBag<TFindBuf>::TCursor cursSrc(fndbToTest.__pcolChildren);
        TBag<TFindBuf>::TCursor cursThis(__pcolChildren);
        do
        {
            if (cursThis.objCur() != cursSrc.objCur())
                return kCIDLib::False;
        }   while (cursThis.bNext() && cursSrc.bNext());
    }
    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TFindBuf: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TFindBuf::bIsSpecialDir() const
{
    // Gotta be a directory obviously
    if (!bIsDirectory())
        return kCIDLib::False;

    TPathStr pathTmp(__pathFileName);
    pathTmp.bRemovePath();
    pathTmp.bRemoveExt();

    // This does not throw any exceptions!
    return TKrnlFileSys::bIsSpecialDir(pathTmp.pszData());
}


tCIDLib::TVoid
TFindBuf::FormatTo(         TTextStream&    strmToWriteTo
                    , const TString&        strFormat) const
{
    // The time/date formats used here
    static const TString strTimeFmt
    (
        L"%(r,2,0):%(h,2,0)%(a)"
        , kCIDLib::c4MaxPathLen+64
    );
    static const TString strDateFmt(L"%(M,2,0)/%(D,2,0)/%(y)");

    //
    //  Create a temp string to format into. Make it pretty large in case
    //  and precommit it all.
    //
    TString strFmt(strFormat);
    TString strTmp(kCIDLib::pszEmptyZStr, 128, 128);

    // Now check for each token and replace it
    if (strFmt.bTokenExists(L'P'))
        strFmt.ReplaceToken(__pathFileName, L'P');

    if (strFmt.bTokenExists(L'N'))
    {
        TPathStr pathTmp;
        if (__pathFileName.bQueryNameExt(pathTmp))
            strFmt.ReplaceToken(pathTmp, L'N');
    }

    if (strFmt.bTokenExists(L'S'))
        strFmt.ReplaceToken(__fposSize, L'S');

    if (strFmt.bTokenExists(L'C'))
    {
        __tmCreate.FormatToStr(strTmp, strTimeFmt);
        strFmt.ReplaceToken(strTmp, L'C');
    }

    if (strFmt.bTokenExists(L'c'))
    {
        __tmCreate.FormatToStr(strTmp, strDateFmt);
        strFmt.ReplaceToken(strTmp, L'c');
    }

    if (strFmt.bTokenExists(L'L'))
    {
        __tmLastAccess.FormatToStr(strTmp, strTimeFmt);
        strFmt.ReplaceToken(strTmp, L'L');
    }

    if (strFmt.bTokenExists(L'l'))
    {
        __tmLastAccess.FormatToStr(strTmp, strDateFmt);
        strFmt.ReplaceToken(strTmp, L'l');
    }

    if (strFmt.bTokenExists(L'M'))
    {
        __tmLastModify.FormatToStr(strTmp, strTimeFmt);
        strFmt.ReplaceToken(strTmp, L'M');
    }

    if (strFmt.bTokenExists(L'm'))
    {
        __tmLastModify.FormatToStr(strTmp, strDateFmt);
        strFmt.ReplaceToken(strTmp, L'm');
    }

    if (strFmt.bTokenExists(L'A'))
    {
        TStringStream strmTmp(&strTmp);
        strmTmp << __eAttrs;
        strFmt.ReplaceToken(strTmp, L'A');
    }

    // Now dump the formatted string to the stream
    strmToWriteTo << strFmt;
}


inline TBag<TFindBuf>& TFindBuf::colChildren()
{
    if (!__pcolChildren)
        __pcolChildren = new TBag<TFindBuf>;
    return *__pcolChildren;
}



// ----------------------------------------------------------------------------
//  TFindBuf: Hidden constructors and operators
// ----------------------------------------------------------------------------

TFindBuf::TFindBuf( const   TKrnlFileSys::TRawFileFind& FindBuf
                    , const TString&                    strPath) :

    __eAttrs(FindBuf.eAttrs)
    , __fposSize(FindBuf.fposFileSize)
    , __pathFileName(strPath)
    , __pcolChildren(0)
    , __tmCreate(FindBuf.i8CreationTime)
    , __tmLastAccess(FindBuf.i8LastAccessTime)
    , __tmLastModify(FindBuf.i8LastWriteTime)
    , __strDefFormat(__pszDefFormat)
{
    // Append the file name to the path
    __pathFileName.AddLevel(FindBuf.szName);
}


// ----------------------------------------------------------------------------
//  TFindBuf: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TFindBuf::_FormatTo(TTextStream& strmToWriteTo) const
{
    FormatTo(strmToWriteTo, __strDefFormat);
}


tCIDLib::TVoid TFindBuf::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Get the attributes out
    strmToReadFrom >> __eAttrs;
    strmToReadFrom >> __fposSize;

    // Parse all of the object members
    strmToReadFrom >> __pathFileName;
    strmToReadFrom >> __tmCreate;
    strmToReadFrom >> __tmLastAccess;
    strmToReadFrom >> __tmLastModify;

    // See if there were any children
    tCIDLib::TCard4 c4Children;
    strmToReadFrom >> c4Children;
    if (c4Children)
    {
        // Fault in the collection or flush it if it exists
        if (!__pcolChildren)
            __pcolChildren = new TBag<TFindBuf>;
        else
            __pcolChildren->Flush();

        TFindBuf fndbTmp;
        for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Children; c4Index++)
        {
            strmToReadFrom >> fndbTmp;
            __pcolChildren->Add(fndbTmp);
        }
    }
}

tCIDLib::TVoid TFindBuf::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Stream out the fundamental data type members
    strmToWriteTo << __eAttrs;
    strmToWriteTo << __fposSize;;

    // Stream out all of the object members
    strmToWriteTo << __pathFileName;
    strmToWriteTo << __tmCreate;
    strmToWriteTo << __tmLastAccess;
    strmToWriteTo << __tmLastModify;

    //
    //  If we have children, then stream them. Either way we have to write
    //  out the number of children streamed.
    //
    tCIDLib::TCard4 c4Count = 0;
    if (__pcolChildren)
    {
        c4Count = __pcolChildren->c4ElemCount();
        if (c4Count)
        {
            strmToWriteTo << c4Count;

            TBag<TFindBuf>::TCursor cursThis(__pcolChildren);
            do
            {
                strmToWriteTo << cursThis.objCur();
            }   while (cursThis.bNext());
        }
    }
     else
    {
        strmToWriteTo << c4Count;
    }
}



// ----------------------------------------------------------------------------
//  TFindBuf: Protected, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _FromFindBuf
//
// DESCRIPTION:
//
//  This method copies the info from a find operation into this find buffer
//  object.
// ---------------------------------------
//   INPUT: FindBuf is a reference to the raw find structure.
//          strPath is the path where it was found
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFindBuf::_FromFindBuf( const   TKrnlFileSys::TRawFileFind& FindBuf
                        , const TString&                    strPath)
{
    // Copy over the data
    __eAttrs        = FindBuf.eAttrs;
    __fposSize      = FindBuf.fposFileSize;
    __pathFileName  = strPath;
    __pathFileName.AddLevel(TPathStr(FindBuf.szName));
    __tmCreate      = FindBuf.i8CreationTime;
    __tmLastAccess  = FindBuf.i8LastAccessTime;
    __tmLastModify  = FindBuf.i8LastWriteTime;
}
