//
// NAME: CIDLib_FileSystem.Cpp
//
// DESCRIPTION:
//
//  This module implements the TFileSys class. This class, along with the
//  ancillary classes TFindBuf and TDirIter provide the file system services
//  for the CIDLib system.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/13/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TFileSys,TObject)



// ----------------------------------------------------------------------------
//   CLASS: TFileSys
//  PREFIX: flsys
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TFileSys: Constructors and Destructors
// ----------------------------------------------------------------------------

TFileSys::TFileSys()
{
}

TFileSys::~TFileSys()
{
}


// ----------------------------------------------------------------------------
//  TFileSys: Constructors and Destructors
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TFileSys::bExists(const TString& strSpec)
{
    return bExists(strSpec.pszData(), tCIDLib::EFileAttr_All);
}

tCIDLib::TBoolean
TFileSys::bExists(const TString& strDirectory, const TString& strName)
{
    TPathStr pathTmp(strDirectory);
    pathTmp.AddLevel(strName);
    return bExists(pathTmp);
}

tCIDLib::TBoolean
TFileSys::bExists(  const   TString&                strSpec
                    , const tCIDLib::EFileAttrs     eCanBe
                    , const tCIDLib::EFileAttrs     eMustBe)
{
    try
    {
        return TKrnlFileSys::bExists(strSpec.pszData(), eCanBe, eMustBe);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        //
        //  If its not just that the file/path is not found, then throw
        //  the error.
        //
        if ((kerrToCatch.errcId() != kKrnlErrors::errcPathNotFound)
        &&  (kerrToCatch.errcId() != kKrnlErrors::errcNoMoreFiles)
        &&  (kerrToCatch.errcId() != kKrnlErrors::errcFileNotFound))
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcFile_Search
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , strSpec
            );
        }
    }
    return kCIDLib::False;
}

tCIDLib::TBoolean
TFileSys::bExists(  const   TString&                strDirectory
                    , const TString&                strName
                    , const tCIDLib::EFileAttrs     eCanBe
                    , const tCIDLib::EFileAttrs     eMustBe)
{
    TPathStr pathTmp(strDirectory);
    pathTmp.AddLevel(strName);
    return bExists(pathTmp, eCanBe, eMustBe);
}

tCIDLib::TBoolean
TFileSys::bExists(  const   TString&                strSpec
                    , const tCIDLib::EFileAttrs     eCanBe
                    , const tCIDLib::EFileAttrs     eMustBe
                    ,       TFindBuf&               fndbToFill)
{
    try
    {
        TDirIter diterTmp;

        if (!diterTmp.bFindFirst(strSpec, fndbToFill, eCanBe, eMustBe))
            return kCIDLib::False;
    }

    catch (...)
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}

tCIDLib::TBoolean
TFileSys::bExists(  const   TString&            strDirectory
                    , const TString&            strName
                    , const tCIDLib::EFileAttrs eCanBe
                    , const tCIDLib::EFileAttrs eMustBe
                    ,       TFindBuf&           fndbToFill)
{
    TPathStr pathTmp(strDirectory);
    pathTmp.AddLevel(strName);
    return bExists(pathTmp, eCanBe, eMustBe, fndbToFill);
}


tCIDLib::TBoolean
TFileSys::bExistsInPath(const   TString&    strSpec
                        , const TString&    strSearchPath)
{
    TFindBuf fndTmp;
    return bExistsInPath
    (
        strSearchPath
        , strSpec
        , fndTmp
        , tCIDLib::EFileAttr_None
    );
}

tCIDLib::TBoolean
TFileSys::bExistsInPath(const   TString&                strSpec
                        , const TString&                strSearchPath
                        , const tCIDLib::EFileAttrs     eCanBe
                        , const tCIDLib::EFileAttrs     eMustBe)
{
    TFindBuf            fndTmp;
    return bExistsInPath(strSearchPath, strSpec, eCanBe, eMustBe, fndTmp);
}

tCIDLib::TBoolean
TFileSys::bExistsInPath(const   TString&            strSpec
                        , const TString&            strSearchPath
                        , const tCIDLib::EFileAttrs eCanBe
                        , const tCIDLib::EFileAttrs eMustBe
                        ,       TFindBuf&           fndTarget)
{
    tCIDLib::Tch    szTmp[kCIDLib::c4MaxPathLen+1];
    try
    {
        TKrnlFileSys::FindInPath
        (
            strSearchPath.pszData()
            , strSpec.pszData()
            , szTmp
            , kCIDLib::c4MaxPathLen
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if ((kerrToCatch.errcId() != kKrnlErrors::errcPathNotFound)
        &&  (kerrToCatch.errcId() != kKrnlErrors::errcNoMoreFiles)
        &&  (kerrToCatch.errcId() != kKrnlErrors::errcFileNotFound))
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcFile_PathSearch
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , strSpec
                , strSearchPath
            );
        }
        return kCIDLib::False;
    }
    return bExists(szTmp, eCanBe, eMustBe, fndTarget);
}


tCIDLib::TBoolean TFileSys::bIsDirectory(const TString& strPath)
{
    return TKrnlFileSys::bExists
    (
        strPath.pszData()
        , tCIDLib::EFileAttr_Directory
        , tCIDLib::EFileAttr_Directory
    );
}


tCIDLib::TCard4
TFileSys::c4BuildDirTree(const TString& strWildCard, TFindBuf& fndbMatches)
{
    // Make sure the find buffer is valid
    if (!(fndbMatches.eAttrs() & tCIDLib::EFileAttr_Directory)
    ||   fndbMatches.bIsSpecialDir())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_BadFindBuf
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
        );
    }

    // Looks ok, so kick off the recursive operation
    tCIDLib::TCard4  c4Matches = 0;
    _FindDirs(strWildCard, fndbMatches, c4Matches);
    return c4Matches;
}


tCIDLib::TCard4
TFileSys::c4BuildFileTree(  const   TString&            strWildCard
                            ,       TFindBuf&           fndbMatches
                            , const tCIDLib::EFileAttrs eCanBe
                            , const tCIDLib::EFileAttrs eMustBe)
{
    // Make sure the find buffer is valid
    if (!(fndbMatches.eAttrs() & tCIDLib::EFileAttr_Directory)
    ||   fndbMatches.bIsSpecialDir())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_BadFindBuf
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
        );
    }

    //
    //  Looks ok, so kick off the recursive operation that finds all of
    //  the directories.
    //
    tCIDLib::TCard4  c4Matches = 0;
    _FindFiles(strWildCard, fndbMatches, c4Matches, eCanBe, eMustBe);

    return c4Matches;
}


tCIDLib::TCard4
TFileSys::c4SearchDir(  const   TString&                strPath
                        , const TString&                strWildCard
                        ,       TCollection<TFindBuf>&  colTarget
                        , const tCIDLib::EFileAttrs     eCanBe
                        , const tCIDLib::EFileAttrs     eMustBe)
{
    TPathStr    pathToSearch(strPath);
    pathToSearch.AddLevel(strWildCard);
    return c4SearchDir(pathToSearch, colTarget, eCanBe, eMustBe);
}

tCIDLib::TCard4
TFileSys::c4SearchDir(  const   TString&                strPath
                        ,       TCollection<TFindBuf>&  colTarget
                        , const tCIDLib::EFileAttrs     eCanBe
                        , const tCIDLib::EFileAttrs     eMustBe)
{
    tCIDLib::TCard4 c4Matches = 0;
    TFindBuf        fndBuf;
    TDirIter        iterSearch;
    if (iterSearch.bFindFirst(strPath, fndBuf, eCanBe, eMustBe))
    {
        do
        {
            colTarget.Add(fndBuf);
            c4Matches++;
        }   while (iterSearch.bFindNext(fndBuf));
    }
    return c4Matches;
}


tCIDLib::TVoid
TFileSys::CopyFile( const   TString&            strSourcePath
                    , const TString&            strTargetPath
                    , const tCIDLib::TBoolean   bFailIfExists)
{
    try
    {
        TKrnlFileSys::CopyFile
        (
            strSourcePath.pszData()
            , strTargetPath.pszData()
            , bFailIfExists
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Copy
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strSourcePath
            , strTargetPath
        );
    }
}


tCIDLib::TVoid TFileSys::DeleteFile(const TString& strSpec)
{
    try
    {
        TKrnlFileSys::DeleteFile(strSpec.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Delete
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strSpec
        );
    }
}

tCIDLib::TVoid
TFileSys::DeleteFile(const TString& strDirectory, const TString& strName)
{
    TPathStr pathTmp(strDirectory);
    pathTmp.AddLevel(strName);
    DeleteFile(pathTmp);
}


tCIDLib::TVoid TFileSys::MakeDirectory( const   TString&    strToCreate
                                        , const TString&    strTemplate)
{
    const tCIDLib::Tch* pszTemplate = 0;

    if (&strTemplate)
        pszTemplate = strTemplate.pszData();

    try
    {
        TKrnlFileSys::MakeDirectory(strToCreate.pszData(), pszTemplate);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_DirCreation
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strToCreate
        );
    }
}


tCIDLib::TVoid
TFileSys::MakeSubDirectory( const   TString& strToCreate
                            , const TString& strParent
                            , const TString& strTemplate)
{
    const tCIDLib::Tch* pszTemplate = 0;

    if (&strTemplate)
        pszTemplate = strTemplate.pszData();

    if (!bIsDirectory(strParent))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_MustBeDir
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strParent
        );
    }

    TPathStr pathTmp(strParent);
    pathTmp.AddLevel(strToCreate);
    MakeDirectory(pathTmp, pszTemplate);
}


tCIDLib::TCard4
TFileSys::c4QueryAvailableVolumePaths(TCollection<TString>& colToFill)
{
    // Flush the collection first
    colToFill.Flush();

    //
    //  Ask the kernel to give us the list of available volumes. It is
    //  in a raw string buffer that contains nul terminated strings,
    //  followed by a double null at the end.
    //
    tCIDLib::Tch* pszVols = 0;
    try
    {
        pszVols = TKrnlFileSys::pszQueryVolumeList();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_VolumeListQuery
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    //
    //  Put a janitor on the returned string to insure its cleaned up. This
    //  means we can now modify the pszVols pointers without worry.
    //
    THeapJanitor janVols(pszVols);

    //
    //  Now lets loop through it and get all of the volumes out. We create
    //  a string for each one and put it into the collection. We loop until
    //  we get a nul, which will catch an empty string and the last nul of
    //  the terminating double nul.
    //
    tCIDLib::TCard4 c4Count = 0;
    while (*pszVols)
    {
        // Add the current volume to the collection
        colToFill.Add(TString(pszVols));

        // Bump our counter
        c4Count++;

        // Move up to the terminating null
        while (*pszVols)
            pszVols++;

        // And now move past it to the next volume or terminating nul
        pszVols++;
    }

    // Return the count of volumes we got
    return c4Count;
}


tCIDLib::TCard4
TFileSys::c4QueryAvailableVolumes(TCollection<TVolumeInfo>& colToFill)
{
    // Flush the collection first
    colToFill.Flush();

    //
    //  Ask the kernel to give us the list of available volumes. It is
    //  in a raw string buffer that contains nul terminated strings,
    //  followed by a double null at the end.
    //
    tCIDLib::Tch* pszVols = 0;
    try
    {
        pszVols = TKrnlFileSys::pszQueryVolumeList();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_VolumeListQuery
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    //
    //  Put a janitor on the returned string to insure its cleaned up. This
    //  means we can now modify the pszVols pointers without worry.
    THeapJanitor janVols(pszVols);

    //
    //  Now lets loop through it and get all of the volumes out. We create
    //  a TVolumeInfo object for each one and put it into the collection.
    //  We loop until we get a nul, which will catch an empty string and
    //  the last nul of the terminating double nul.
    //
    tCIDLib::TCard4 c4Count = 0;
    while (*pszVols)
    {
        //
        //  Try to get the volume info, but watch for exceptions indicating
        //  that the volume is not ready.
        //
        try
        {
            // Add the current volume to the collection
            colToFill.Add(TVolumeInfo(pszVols));

            // Bump our counter
            c4Count++;
        }

        catch(const TError& errToCatch)
        {
            //
            //  If its anything besides a not ready error, or an unsupported
            //  file system type error, let it propogate.
            //
            if ((errToCatch.errcKrnlId() != kKrnlErrors::errcNotReady)
            &&  (errToCatch.errcKrnlId() != kKrnlErrors::errcUnsupportedFileSystem))
            {
                throw;
            }
        }

        // Move up to the terminating null
        while (*pszVols)
            pszVols++;

        // And now move past it to the next volume or terminating nul
        pszVols++;
    }

    // Return the count of volumes we got
    return c4Count;
}


tCIDLib::TVoid TFileSys::QueryCurrentDir(TString& strToFill)
{
    tCIDLib::Tch szPath[kCIDLib::c4MaxPathLen+1];

    try
    {
        TKrnlFileSys::QueryCurrentDir(szPath, c4MaxBufChars(szPath));
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_QueryCurDir
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    strToFill = szPath;
}


tCIDLib::TVoid
TFileSys::QueryFullPath(const   TString& strPartialPath
                        ,       TString& strFullReturnedPath)
{
    tCIDLib::Tch szBuffer[kCIDLib::c4MaxPathLen+1];

    try
    {
        TKrnlFileSys::QueryFullPath
        (
            strPartialPath.pszData()
            , szBuffer
            , kCIDLib::c4MaxPathLen
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_PathCompletion
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strPartialPath
        );
    }
    strFullReturnedPath = szBuffer;

    // If it ends with a separator, then throw that away
    if (strFullReturnedPath.chLast() == kCIDLib::chPathSeparator)
        strFullReturnedPath.DeleteLast();
}


tCIDLib::TVoid
TFileSys::QueryVolumeLabel(         TString&    strToFill
                            ,       TString&    strSerialNum
                            , const TString&    strVolumePath)
{
    TKrnlFileSys::TRawVolumeInfo VolInfo;

    try
    {
        TKrnlFileSys::QueryVolumeInfo(strVolumePath.pszData(), VolInfo);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_VolumeQuery
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strVolumePath
        );
    }

    // Copy over the volume label
    strToFill = VolInfo.szVolumeLabel;

    //
    //  And copy over the volume serial number. This is formatted as
    //  the high word, a dash, then the low word.
    //
    strSerialNum = L"%(1,4)-%(2,4)";
    strSerialNum.ReplaceToken
    (
        TCardinal(VolInfo.c4VolumeSerialNum >> 16, tCIDLib::ERadix_Hex)
        , L'1'
    );
    strSerialNum.ReplaceToken
    (
        TCardinal(VolInfo.c4VolumeSerialNum & 0xFFFF, tCIDLib::ERadix_Hex)
        , L'2'
    );
}


tCIDLib::TVoid TFileSys::RemoveDirectory(const TString& strSpec)
{
    try
    {
        TKrnlFileSys::RemoveDirectory(strSpec.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_RemoveDir
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strSpec
        );
    }
}


tCIDLib::TVoid TFileSys::RemovePath(const TString& strStartDir)
{
    try
    {
        TKrnlFileSys::RemovePath(strStartDir.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_PathDelete
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strStartDir
        );
    }
}


tCIDLib::TVoid
TFileSys::Rename(const TString& strSource, const TString& strDest)
{
    try
    {
        TKrnlFileSys::Rename(strSource.pszData(), strDest.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Rename
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strSource
            , strDest
        );
    }
}


// ----------------------------------------------------------------------------
//  TFileSys: Protected, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _FindDirs
//
// DESCRIPTION:
//
//  This method is called by public methods that needs to find all of the
//  directory structure under a starting point.
// ---------------------------------------
//   INPUT: strWildCard is the wild card spec
//
//  OUTPUT: fndbTarget is the find buffer that represents the root of this
//              level.
//          c4Matches is bump up by the number of matches. It is assumed that
//              the public caller is initializing the count.
//
//  RETURN: None
//
tCIDLib::TVoid
TFileSys::_FindDirs(const   TString&            strWildCard
                    ,       TFindBuf&           fndbTarget
                    ,       tCIDLib::TCard4&    c4Matches)
{
    // Flush the child collection to make sure we start empty
    fndbTarget.colChildren().Flush();

    //
    //  Find all of the matches at this level. Then we will go back and
    //  recursively search each of them. If we don't find any matches
    //  here, then we are done.
    //
    tCIDLib::TCard4 c4Count = c4SearchDir
    (
        fndbTarget.pathFileName()
        , strWildCard
        , fndbTarget.colChildren()
        , tCIDLib::EFileAttr_Directory
        , tCIDLib::EFileAttr_Directory
    );

    if (!c4Count)
        return;

    // Add the matches we got
    c4Matches += c4Count;

    //
    //  We found some, so iterate the bag of child directories and
    //  recurse for each one.
    //
    fndbTarget.colChildren().bResetIter();
    do
    {
        _FindDirs(strWildCard, fndbTarget.colChildren().objCur(), c4Matches);
    }   while (fndbTarget.colChildren().bNext());
}


//
// FUNCTION/METHOD NAME: _FindFiles
//
// DESCRIPTION:
//
//  This method is called by the public version who has set the cMatches to
//  whatever desired value already, we do not init it to zero. This is
//  because this recursive.
// ---------------------------------------
//   INPUT: strWildCard is the wild card spec
//
//  OUTPUT: fndbTarget is the find buffer that represents the root of this
//              level.
//          c4Matches is bump up by the number of matches. It is assumed that
//              the public caller is initializing the count.
//
//  RETURN: None
//
tCIDLib::TVoid
TFileSys::_FindFiles(   const   TString&            strWildCard
                        ,       TFindBuf&           fndbTarget
                        ,       tCIDLib::TCard4&    c4Matches
                        , const tCIDLib::EFileAttrs eCanBe
                        , const tCIDLib::EFileAttrs eMustBe)
{
    // Flush the child collection to make sure we start empty
    fndbTarget.colChildren().Flush();

    //
    //  Create a search spec that includes directories. It might have
    //  already, but we need to insure it does. This avoids the need
    //  to do two steps, one for directories and one for files.
    //
    tCIDLib::EFileAttrs eActualCanBe = tCIDLib::EFileAttrs
    (
        eCanBe
        | tCIDLib::EFileAttr_Directory
    );

    //
    //  Find all of the matches at this level. Then we will go back and
    //  recursively search each of them. If we don't find any matches
    //  here, then we are done.
    //
    tCIDLib::TCard4 c4Count = c4SearchDir
    (
        fndbTarget.pathFileName()
        , strWildCard
        , fndbTarget.colChildren()
        , eActualCanBe
        , eMustBe
    );

    if (!c4Count)
        return;

    // Add the matches we got
    c4Matches += c4Count;

    //
    //  We found some, so iterate the bag of child directories and
    //  recurse for each one.
    //
    fndbTarget.colChildren().bResetIter();
    do
    {
        //
        //  If this one is a directory and not a special directory, then
        //  search it.
        //
        if (fndbTarget.colChildren().objCur().bIsDirectory()
        &&  !fndbTarget.colChildren().objCur().bIsSpecialDir())
        {
            _FindFiles
            (
                strWildCard
                , fndbTarget.colChildren().objCur()
                , c4Matches
                , eCanBe
                , eMustBe
            );
        }
    }   while (fndbTarget.colChildren().bNext());
}
