//
// NAME: CIDLib_FileStreamImpl.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_FileStreamImpl.Cpp module, which
//  implements the TFileBinStreamImpl and TFileTextStreamImpl classes. These
//  classes provide the data source/sink for a binary or text stream. Streams
//  get their data from a plugged in implementation object, allowing streams
//  to get their data from many different sources.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/24/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  This class does not have to track its current position because the
//      file object it encapsulates does this for us.
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TBinaryFile;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: MFileStreamImpl
//  PREFIX: strmi
//
//  A lot of the text and binary stream implementations are similar, so a
//  mixin is provided to handle the common stuff.
// -----------------------------------------------------------------------------
class CIDLIBEXP MFileStreamImpl
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        MFileStreamImpl
        (
                    TBinaryFile* const      pflToUse
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
        );

        ~MFileStreamImpl();


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid Reset();

        tCIDLib::TVoid TruncateAtZero();


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid AdoptFile();

        tCIDLib::TBoolean bFileIsAdopted() const;

        tCIDLib::TVoid Close();

        tCIDLib::TVoid Open
        (
            const   tCIDLib::EAccessModes   eAccess
            , const tCIDLib::ECreateActions eAction
            , const tCIDLib::EFileAttrs     eAttrs
            , const tCIDLib::EFileFlags     eFlags
        );


    protected  :
        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _CheckCount
        (
            const   tCIDLib::TCard4         c4Line
            , const tCIDLib::TCard4         c4Count
            , const tCIDLib::TCard4         c4Actual
        );

        // ---------------------------------------------------------------------
        //  Protected data members
        //
        //  _eAdopted
        //      Indicates whether we have adopted the file object.
        //
        //  _flData
        //      This is the binary file object that provides the data.
        // ---------------------------------------------------------------------
        tCIDLib::EAdoptOpts     _eAdopted;
        TBinaryFile*            _pflData;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        MFileStreamImpl();

        tCIDLib::TVoid operator=(const MFileStreamImpl&);
};


// -----------------------------------------------------------------------------
//   CLASS: TFileBinStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------
class CIDLIBEXP TFileBinStreamImpl :

    public TBinStreamImpl, public MFileStreamImpl
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TFileBinStreamImpl
        (
                    TBinaryFile* const      pflToUse
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
        );

        ~TFileBinStreamImpl();


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid ReadBytes
        (
                    tCIDLib::TVoid* const   pBuffer
            , const tCIDLib::TCard4         c4BytesToRead
            ,       tCIDLib::TCard4&        c4Actual
        );

        tCIDLib::TVoid Reset();

        tCIDLib::TVoid TruncateAtZero();

        tCIDLib::TVoid WriteBytes
        (
            const   tCIDLib::TVoid* const   pBuffer
            , const tCIDLib::TCard4         c4BytesToWrite
            ,       tCIDLib::TCard4&        c4Actual
        );


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TFileBinStreamImpl(const TFileBinStreamImpl&);

        tCIDLib::TVoid operator=(const TFileBinStreamImpl&);


    private :
        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFileBinStreamImpl,TBinStreamImpl)
};


// -----------------------------------------------------------------------------
//   CLASS: TFileTextStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------
class CIDLIBEXP TFileTextStreamImpl :

    public TTextStreamImpl, public MFileStreamImpl
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TFileTextStreamImpl
        (
                    TBinaryFile* const      pflToUse
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
        );

        TFileTextStreamImpl
        (
                    TBinaryFile* const      pflToUse
            , const tCIDLib::ETextFormats   eTextFmt
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
        );

        ~TFileTextStreamImpl();


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TCard4 c4GetLine
        (
                    tCIDLib::Tch* const     pszBufToFill
            , const tCIDLib::TCard4         c4BufferSize
            , const tCIDLib::TBoolean       bStripWhitespace = kCIDLib::True
        );

        tCIDLib::Tch chGet();

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tch            chToWrite
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tsch           schToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tch* const     pszBufToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tsch* const    pszBufToWrite
        );

        tCIDLib::TVoid Reset();

        tCIDLib::TVoid TruncateAtZero();

        tCIDLib::Tsch schGet();


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TFileTextStreamImpl(const TFileTextStreamImpl&);

        tCIDLib::TVoid operator=(const TFileTextStreamImpl&);


    private :
        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFileTextStreamImpl,TTextStreamImpl)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  MFileStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean MFileStreamImpl::bEndOfStream() const
{
    return _pflData->bEndOfFile();
}

inline tCIDLib::TFilePos MFileStreamImpl::fposCurPos() const
{
    return _pflData->fposQueryCurPos();
}

inline tCIDLib::TFilePos MFileStreamImpl::fposLogicalEnd() const
{
    return _pflData->fposQuerySize();
}

inline tCIDLib::TFilePos MFileStreamImpl::fposPhysicalEnd() const
{
    return tCIDLib::TFilePos(kCIDLib::i8MaxInt);
}

inline tCIDLib::TFilePos MFileStreamImpl::fposSeekToEnd()
{
    tCIDLib::TFilePos fposActual;

    _pflData->Seek(0, tCIDLib::ESeekFrom_End, fposActual);
    return fposActual;
}

inline tCIDLib::TVoid MFileStreamImpl::Reset()
{
    _pflData->Seek(0);
}

inline tCIDLib::TVoid MFileStreamImpl::TruncateAtZero()
{
    _pflData->TruncateAt(0);
}


// -----------------------------------------------------------------------------
//  MFileStreamImpl: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid MFileStreamImpl::AdoptFile()
{
    _eAdopted = tCIDLib::EAdoptOpt_Adopt;
}

inline tCIDLib::TBoolean MFileStreamImpl::bFileIsAdopted() const
{
    if (_eAdopted)
        return kCIDLib::True;
    else
        return kCIDLib::False;
}

inline tCIDLib::TVoid MFileStreamImpl::Close()
{
    _pflData->Close();
}

inline tCIDLib::TVoid
MFileStreamImpl::Open(  const   tCIDLib::EAccessModes   eAccess
                        , const tCIDLib::ECreateActions eAction
                        , const tCIDLib::EFileAttrs     eAttrs
                        , const tCIDLib::EFileFlags     eFlags)
{
    _pflData->Open(eAccess, eAction, eAttrs, eFlags);
}



// -----------------------------------------------------------------------------
//  TFileBinStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------

inline tCIDLib::TBoolean TFileBinStreamImpl::bEndOfStream() const
{
    return MFileStreamImpl::bEndOfStream();
}

inline tCIDLib::TFilePos TFileBinStreamImpl::fposCurPos() const
{
    return MFileStreamImpl::fposCurPos();
}

inline tCIDLib::TFilePos TFileBinStreamImpl::fposLogicalEnd() const
{
    return MFileStreamImpl::fposLogicalEnd();
}

inline tCIDLib::TFilePos TFileBinStreamImpl::fposPhysicalEnd() const
{
    return MFileStreamImpl::fposPhysicalEnd();
}

inline tCIDLib::TFilePos TFileBinStreamImpl::fposSeekToEnd()
{
    return MFileStreamImpl::fposSeekToEnd();
}

inline tCIDLib::TFilePos
TFileBinStreamImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    return MFileStreamImpl::fposSkipForwardBy(c4SkipBy);
}

inline tCIDLib::TVoid TFileBinStreamImpl::Reset()
{
    MFileStreamImpl::Reset();
}

inline tCIDLib::TVoid TFileBinStreamImpl::TruncateAtZero()
{
    MFileStreamImpl::TruncateAtZero();
}


// -----------------------------------------------------------------------------
//  TFileTextStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TFileTextStreamImpl::bEndOfStream() const
{
    return MFileStreamImpl::bEndOfStream();
}

inline tCIDLib::TVoid TFileTextStreamImpl::Reset()
{
    MFileStreamImpl::Reset();
}

inline tCIDLib::TVoid TFileTextStreamImpl::TruncateAtZero()
{
    MFileStreamImpl::TruncateAtZero();
}
