//
// NAME: CIDLib_FileStreamImpl.Cpp
//
// DESCRIPTION: 
//
//  This module implements the TFileBinStreamImpl and TFileTextStreamImpl classes.
//  These classes are the plug in derivatives of TTextStreamImpl and TBinStreamImpl
//  that can be the sink/source of a stream's data. 
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/24/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TFileBinStreamImpl,TBinStreamImpl)
RTTIData(TFileTextStreamImpl,TTextStreamImpl)



// -----------------------------------------------------------------------------
//   CLASS: MFileStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  MFileStreamImpl: Constructors and destructors
// -----------------------------------------------------------------------------

MFileStreamImpl::MFileStreamImpl(   TBinaryFile* const          pflToUse
                                    , const tCIDLib::EAdoptOpts eAdopt) :
    _eAdopted(eAdopt)
    , _pflData(pflToUse)
{
}

MFileStreamImpl::~MFileStreamImpl()
{
    if (_eAdopted)
    {
        _pflData->Close();
        delete _pflData;
        _pflData = 0;
    }
}


// -----------------------------------------------------------------------------
//  MFileStreamImpl: Public, non-virtual methods
// -----------------------------------------------------------------------------

inline tCIDLib::TFilePos
MFileStreamImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    // Remember the current position
    tCIDLib::TFilePos fposCurrent = fposCurPos();

    tCIDLib::TFilePos fposActual;
    _pflData->Seek(c4SkipBy, tCIDLib::ESeekFrom_Current, fposActual);

    //
    //  If the current position plus the skip is not equal to actual, then
    //  we did not succeed.
    //
    if (fposCurrent + c4SkipBy != fposActual)
    {
        // Put the position back
        _pflData->Seek(fposCurrent, tCIDLib::ESeekFrom_Start, fposActual);

        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InvalidRelSeek
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(c4SkipBy)
        );
    }
    return fposActual;
}




// -----------------------------------------------------------------------------
//  MFileStreamImpl: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
MFileStreamImpl::_CheckCount(   const   tCIDLib::TCard4     c4Line
                                , const tCIDLib::TCard4     c4Count
                                , const tCIDLib::TCard4     c4Actual)
{
    if (c4Actual != c4Count)
    {
        tCIDLib::TErrCode errcToLog;
        if (!c4Actual)
            errcToLog = kCIDErrs::errcStrm_EndOfStream;
        else
            errcToLog = kCIDErrs::errcStrm_NotAllData;

        // Seek back to the original position if any was read or written
        if (c4Actual)
        {
            tCIDLib::TFilePos fposSeek = c4Actual;
            _pflData->Seek(-fposSeek, tCIDLib::ESeekFrom_Current);
        }

        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , errcToLog
            , _pflData->strName()
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_OutResource
        );
    }
}



// -----------------------------------------------------------------------------
//   CLASS: TFileBinStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFileBinStreamImpl: Constructors and destructors
// -----------------------------------------------------------------------------

TFileBinStreamImpl::TFileBinStreamImpl( TBinaryFile* const          pflToUse
                                        , const tCIDLib::EAdoptOpts eAdopt) :
    MFileStreamImpl(pflToUse, eAdopt)
{
}

TFileBinStreamImpl::~TFileBinStreamImpl()
{
}


// -----------------------------------------------------------------------------
//  TFileBinStreamImpl: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TFileBinStreamImpl::ReadBytes(          tCIDLib::TVoid* const   pBufToFill
                                , const tCIDLib::TCard4         c4Count
                                ,       tCIDLib::TCard4&        c4Actual)
{
    // And write the buffer
    tCIDLib::TCard4  c4TmpActual;
    _pflData->ReadBuf(pBufToFill, c4Count, c4TmpActual);

    if (&c4Actual)
        c4Actual = c4TmpActual;
     else
        _CheckCount(__LINE__, c4Count, c4TmpActual);
}


tCIDLib::TVoid
TFileBinStreamImpl::WriteBytes( const   tCIDLib::TVoid* const   pBufToUse
                                , const tCIDLib::TCard4         c4Count
                                ,       tCIDLib::TCard4&        c4Actual)
{
    // And write the buffer
    tCIDLib::TCard4 c4TmpActual;
    _pflData->WriteBuf(pBufToUse, c4Count, c4TmpActual);

    if (&c4Actual)
        c4Actual = c4TmpActual;
     else
        _CheckCount(__LINE__, c4Count, c4TmpActual);
}



// -----------------------------------------------------------------------------
//   CLASS: TFileTextStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFileTextStreamImpl: Constructors and destructors
// -----------------------------------------------------------------------------

TFileTextStreamImpl::TFileTextStreamImpl(       TBinaryFile* const  pflToUse
                                        , const tCIDLib::EAdoptOpts eAdopt) :
    MFileStreamImpl(pflToUse, eAdopt)
{
}

TFileTextStreamImpl::
TFileTextStreamImpl(            TBinaryFile* const      pflToUse
                        , const tCIDLib::ETextFormats   eTextFmt
                        , const tCIDLib::EAdoptOpts     eAdopt) :

    MFileStreamImpl(pflToUse, eAdopt)
    , TTextStreamImpl(eTextFmt)
{
}

TFileTextStreamImpl::~TFileTextStreamImpl()
{
}


// -----------------------------------------------------------------------------
//  TFileTextStreamImpl: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4
TFileTextStreamImpl::c4GetLine(         tCIDLib::Tch* const pszBufToFill
                                , const tCIDLib::TCard4     c4BufSize
                                , const tCIDLib::TBoolean   bStripWhitespace)
{
    // If we are at the end of file, then log an 'end of stream' error
    if (_pflData->bEndOfFile())
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_EndOfStream
            , _pflData->strName()
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_OutResource
        );
    }

    tCIDLib::Tch    chCur;
    tCIDLib::TCard4 c4Count = 0;
    tCIDLib::TCard4 c4Actual;

    while (1)
    {
        // Get the next character
        if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        {
            _pflData->ReadBuf(&chCur, sizeof(chCur), c4Actual);
        }
         else
        {
            tCIDLib::Tsch schCur;
            _pflData->ReadBuf(&schCur, sizeof(schCur), c4Actual);
            chCur = TRawStr::chConvert(schCur);
        }

        //
        //  If we got no bytes, then we are at the end of the stream
        //  so we need to stop here.
        //
        if (!c4Actual)
        {
            pszBufToFill[c4Count] = 0;
            break;
        }

        // If a newline, then cap the string and return
        if (chCur == kCIDLib::chLF)
        {
            pszBufToFill[c4Count] = 0;
            break;
        }

        // Put it in the buffer and bump up the count
        pszBufToFill[c4Count] = chCur;
        c4Count++;

        // Don't overflow the buffer
        if (c4Count == c4BufSize-1)
        {
            pszBufToFill[c4BufSize-1] = 0;
            break;
        }
    }

    //
    //  If the last character was a carraige return, then do away with
    //  it, since its just part of a CR/LF at the line end.
    //
    if (c4Count)
    {
        if (pszBufToFill[c4Count-1] == kCIDLib::chCR)
        {
            pszBufToFill[c4Count-1] = kCIDLib::chNull;
            c4Count--;
        }
    }

    // Strip off white space if asked to
    if (bStripWhitespace)
    {
        TRawStr::StripStr
        (
            pszBufToFill
            , kCIDLib::szWhitespace
            , tCIDLib::EStripMode_LeadTrail
        );

        // Update the count to the stripped length
        c4Count = TRawStr::c4StrLen(pszBufToFill);
    }

    return c4Count;
}


tCIDLib::Tch TFileTextStreamImpl::chGet()
{
    tCIDLib::Tch    chRet;
    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4ToWrite;
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        c4ToWrite = sizeof(chRet);
        _pflData->ReadBuf(&chRet, c4ToWrite, c4Actual);
    }
     else
    {
        tCIDLib::Tsch schTmp;
        c4ToWrite = sizeof(schTmp);
        _pflData->ReadBuf(&schTmp, c4ToWrite, c4Actual);
        chRet = TRawStr::chConvert(schTmp);
    }
    _CheckCount(__LINE__, c4ToWrite, c4Actual);
    return chRet;
}


tCIDLib::TFilePos TFileTextStreamImpl::fposCurPos() const
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MFileStreamImpl::fposCurPos();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}


tCIDLib::TFilePos TFileTextStreamImpl::fposLogicalEnd() const
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MFileStreamImpl::fposLogicalEnd();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}


tCIDLib::TFilePos TFileTextStreamImpl::fposPhysicalEnd() const
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MFileStreamImpl::fposPhysicalEnd();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}


tCIDLib::TFilePos TFileTextStreamImpl::fposSeekToEnd()
{
    // Get the raw file position
    tCIDLib::TFilePos fposRet = MFileStreamImpl::fposSeekToEnd();

    // If the data is UNICode, then turn into a character offset
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}


tCIDLib::TFilePos
TFileTextStreamImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    // We have to convert if UNICode
    const tCIDLib::TBoolean bUNIFmt = eTextFormat() == tCIDLib::ETextFmt_UNICode;

    // If the format is UNICode we need to convert skip count to bytes
    tCIDLib::TCard4 c4Skip = c4SkipBy;
    if (bUNIFmt)
        c4Skip *= kCIDLib::c4CharBytes;

    // Get the raw new file position
    tCIDLib::TFilePos fposRet = MFileStreamImpl::fposSkipForwardBy(c4SkipBy);

    // If the data is UNICode, then turn into a character offset
    if (bUNIFmt)
        fposRet /= kCIDLib::c4CharBytes;
    return fposRet;
}


tCIDLib::TVoid TFileTextStreamImpl::PutCh(const tCIDLib::Tch chToWrite)
{
    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4ToWrite;
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        c4ToWrite = sizeof(chToWrite);
        _pflData->WriteBuf(&chToWrite, c4ToWrite, c4Actual);
    }
     else
    {
        tCIDLib::Tsch   schTmp = TRawStr::schConvert(chToWrite);
        c4ToWrite = sizeof(schTmp);
        _pflData->WriteBuf(&schTmp, c4ToWrite, c4Actual);
    }
    _CheckCount(__LINE__, c4ToWrite, c4Actual);
}


tCIDLib::TVoid TFileTextStreamImpl::PutCh(const tCIDLib::Tsch schToWrite)
{
    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4ToWrite;
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        tCIDLib::Tch chTmp = TRawStr::chConvert(schToWrite);
        c4ToWrite = sizeof(chTmp);
        _pflData->WriteBuf(&chTmp, c4ToWrite, c4Actual);
    }
     else
    {
        c4ToWrite = sizeof(schToWrite);
        _pflData->WriteBuf(&schToWrite, c4Actual);
    }
    _CheckCount(__LINE__, c4ToWrite, c4Actual);
}


tCIDLib::TVoid
TFileTextStreamImpl::PutLine(const tCIDLib::Tch* const pszBufToWrite)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszBufToWrite);
    if (!c4Len)
        return;

    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4ToWrite;
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        c4ToWrite = c4Len * kCIDLib::c4CharBytes;
        _pflData->WriteBuf(pszBufToWrite, c4ToWrite, c4Actual);
    }
     else
    {
        tCIDLib::Tsch*  pszTmp = TRawStr::pszConvert(pszBufToWrite);
        THeapJanitor    janTmpStr(pszTmp);

        c4ToWrite = c4Len;
        _pflData->WriteBuf(pszTmp, c4ToWrite, c4Actual);
    }
    _CheckCount(__LINE__, c4ToWrite, c4Actual);
}


tCIDLib::TVoid
TFileTextStreamImpl::PutLine(const tCIDLib::Tsch* const pszBufToWrite)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszBufToWrite);
    if (!c4Len)
        return;

    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4ToWrite;
    if (eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        tCIDLib::Tch*   pszTmp = TRawStr::pszConvert(pszBufToWrite);
        THeapJanitor    janTmpStr(pszTmp);

        c4ToWrite = c4Len * kCIDLib::c4CharBytes;
        _pflData->WriteBuf(pszTmp, c4ToWrite, c4Actual);
    }
     else
    {
        c4ToWrite = c4Len;
        _pflData->WriteBuf(pszBufToWrite, c4ToWrite, c4Actual);
    }
    _CheckCount(__LINE__, c4ToWrite, c4Actual);
}


tCIDLib::Tsch TFileTextStreamImpl::schGet()
{
    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4ToWrite;
    tCIDLib::Tsch   schRet;
    if (eTextFormat() == tCIDLib::ETextFmt_ASCII)
    {
        c4ToWrite = sizeof(schRet);
        _pflData->ReadBuf(&schRet, c4ToWrite, c4Actual);
    }
     else
    {
        tCIDLib::Tch chTmp;
        c4ToWrite = sizeof(chTmp);
        _pflData->ReadBuf(&chTmp, c4ToWrite, c4Actual);
        schRet = TRawStr::schConvert(chTmp);
    }
    _CheckCount(__LINE__, c4ToWrite, c4Actual);
    return schRet;
}
