//
// NAME: CIDLib_FileLogger.Cpp
//
// DESCRIPTION:
//
//  This module implements a file logger, which sends logged errors to a
//  binary output file, as flattened objects.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/27/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  In order to speed things up, it formats first into a memory buffer
//      then writes the whole thing to disk as a binary blob. This is much
//      faster than flattening to a disk file stream directly.
//


// -----------------------------------------------------------------------------
//  Includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TFileLogger,TLogger)


// -----------------------------------------------------------------------------
//  TFileLogger: Constructors and operators
// -----------------------------------------------------------------------------

TFileLogger::TFileLogger() :

    __pmtxSync(0)
{
}

TFileLogger::TFileLogger(   const   tCIDLib::Tch* const pszFile
                            , const TResourceName&      rsnMutex) :
    __pmtxSync(0)
{
    Create(pszFile, rsnMutex);
}

TFileLogger::~TFileLogger()
{
    delete __pmtxSync;
}


// -----------------------------------------------------------------------------
//  TFileLogger: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFileLogger::LogMsg(const TError& errToLog)
{
    //
    //  Create a static memory buffer stream for formatting the errors
    //  to. We can get away with this because calls to this method are
    //  synced between threads in this process.
    //
    //  The stream does not adopt by default, which is cool for this
    //  case.
    //
    static TSysBuf          mbufFormat(kCIDLib::c4MemPageSize * 4);
    static TBinaryStream    strmFormat(new TMemBinStreamImpl(&mbufFormat));

    // Reset the stream pointer for the new message
    strmFormat.Reset();

    // And flatten the object to it
    strmFormat << errToLog;

    if (__pmtxSync)
    {
        TMtxLock mtxlSync(__pmtxSync);

        __flTarget.WriteBuf
        (
            mbufFormat.pData()
            , tCIDLib::TCard4(strmFormat.fposCurPos())
        );
    }
     else
    {
        __flTarget.WriteBuf
        (
            mbufFormat.pData()
            , tCIDLib::TCard4(strmFormat.fposCurPos())
        );
    }
}

tCIDLib::TVoid TFileLogger::Create( const   tCIDLib::Tch* const pszFile
                                    , const TResourceName&      rsnMutex)
{
    __flTarget.strName(pszFile);
    __flTarget.Open
    (
        tCIDLib::EAccess_Write
        , tCIDLib::ECreateAct_CreateIfNew
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    // Seek to the end for appending
    __flTarget.Seek(0, tCIDLib::ESeekFrom_End);

    // Create the new mutex that we will use to sync if they want us to
    if (&rsnMutex)
        __pmtxSync = new TMutex(rsnMutex, tCIDLib::ELockState_Unlocked);
}
