//
// NAME: CIDLib_FileBase.Cpp
//
// DESCRIPTION:
//
//  This module implements the TFileBase class, which is the lowest level
//  file class.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/20/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"

// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TFileBase,TObject)


// -----------------------------------------------------------------------------
//   CLASS: TFileBase
//  PREFIX: fbase
// -----------------------------------------------------------------------------

// ---------------------------------------------------------------------
//  TFileBase: Constructors and Destructors
// ---------------------------------------------------------------------

TFileBase::TFileBase(const TString& strFileName) :

    __strName(strFileName.pszData(), kCIDLib::c4MaxPathLen)
    , __kflThis(strFileName.pszData())
{
}

TFileBase::TFileBase(const tCIDLib::Tch* const pszFileName) :

    __strName(pszFileName, kCIDLib::c4MaxPathLen)
    , __kflThis(pszFileName)
{
}

TFileBase::TFileBase() :

    __strName(kCIDLib::pszEmptyZStr, kCIDLib::c4MaxPathLen)
{
}

TFileBase::~TFileBase()
{
    try
    {
        __kflThis.Close();
    }

    // Log a warning but don't let it propogate out of destructor!
    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Close
            , kerrToCatch
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
}


// ---------------------------------------------------------------------
//  TFileBase: Public, non-virtual methods
// ---------------------------------------------------------------------

tCIDLib::TBoolean TFileBase::bEndOfFile()
{
    tCIDLib::TFilePos   fposCurPos;
    tCIDLib::TFilePos   fposCurSize;

    tCIDLib::TErrCode errcToLog;
    try
    {
        // Query the current file size
        errcToLog = kCIDErrs::errcFile_QuerySize;
        fposCurSize = __kflThis.fposCurSize();

        // If current size is 0, then has to be
        if (!fposCurSize)
            return kCIDLib::True;

        // Query the current file position
        errcToLog = kCIDErrs::errcFile_Seek;
        __kflThis.Seek(0, tCIDLib::ESeekFrom_Current, fposCurPos);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , errcToLog
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }

    if (fposCurPos >= fposCurSize)
        return kCIDLib::True;
    else
        return kCIDLib::False;
}


tCIDLib::TBoolean
TFileBase::bExists( const   tCIDLib::EFileAttrs eCanBe
                    , const tCIDLib::EFileAttrs eMustBe) const
{
    return TKrnlFileSys::bExists(__strName.pszData(), eCanBe, eMustBe);
}

tCIDLib::TBoolean
TFileBase::bExists( const   tCIDLib::EFileAttrs eCanBe
                    , const tCIDLib::EFileAttrs eMustBe
                    ,       tCIDLib::TBoolean&  bUnique) const
{
    return TKrnlFileSys::bExists(__strName.pszData(), eCanBe, eMustBe, bUnique);
}


tCIDLib::TVoid TFileBase::Close()
{
    if (__kflThis.bValid())
    {
        // Bump down the number of opened files
        _pmtrgCore->OffsetCard(tCIDLib_::eCoreMetric_OpenFiles, -1);

        try
        {
            __kflThis.Close();
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcFile_Close
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_AppError
                , __strName
            );
        }
    }
}


tCIDLib::TVoid
TFileBase::Open(const   tCIDLib::EAccessModes   eAccess
                , const tCIDLib::ECreateActions eAction
                , const tCIDLib::EFileAttrs     eAttrs
                , const tCIDLib::EFileFlags     eFlags)
{
    try
    {
        __kflThis.Open(eAccess, eAction, eAttrs, eFlags);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Open
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }

    // Bump up the number of opened files
    _pmtrgCore->OffsetCard(tCIDLib_::eCoreMetric_OpenFiles, 1);
}

tCIDLib::TVoid TFileBase::Open(const tCIDLib::EStdFiles eStdFile)
{
    try
    {
        __kflThis.Open(eStdFile);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Open
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(__kflThis.pszName())
        );
    }

    //
    //  Get the name out of the kernel file object that it selected for
    //  the standard file that the user wants.
    //
    __strName = __kflThis.pszName();

    // Bump up the number of opened files
    _pmtrgCore->OffsetCard(tCIDLib_::eCoreMetric_OpenFiles, 1);
}

tCIDLib::TFilePos TFileBase::fposQueryCurPos()
{
    tCIDLib::TFilePos   fposCurPos = 0;

    // Query the current file position
    try
    {
        __kflThis.Seek(0, tCIDLib::ESeekFrom_Current, fposCurPos);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Seek
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
    return fposCurPos;
}


tCIDLib::TFilePos TFileBase::fposQuerySize()
{
    tCIDLib::TFilePos fposSize;

    try
    {
        fposSize = __kflThis.fposCurSize();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_QuerySize
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
    return fposSize;
}


tCIDLib::TVoid TFileBase::TruncateAt(const tCIDLib::TFilePos fposPosition)
{
    try
    {
        __kflThis.TruncateAt(fposPosition);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Truncate
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
}
 
tCIDLib::TVoid TFileBase::TruncateAt()
{
    try
    {
        __kflThis.TruncateAt();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Truncate
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
}


const TString& TFileBase::strName(const TString& strNew)
{
    if (__kflThis.bValid())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_AlreadyOpen
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
            , __strName
        );
    }

    // Store the new name and return a ref to our member
    __strName = strNew;
    return __strName;
}


// ---------------------------------------------------------------------
//  TFileBase: Protected, non-virtual methods
// ---------------------------------------------------------------------

tCIDLib::TVoid
TFileBase::_ReadBuf(        tCIDLib::TVoid* const   pBuf
                    , const tCIDLib::TCard4         c4BufSz
                    ,       tCIDLib::TCard4&        c4Actual)
{
    try
    {
        __kflThis.ReadBuffer(pBuf, c4BufSz, c4Actual);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Read
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(c4BufSz)
            , __strName
        );
    }
}


tCIDLib::TVoid
TFileBase::_Seek(   const   tCIDLib::TFilePos   fposOffset
                    , const tCIDLib::ESeekFrom  eSeekFrom
                    ,       tCIDLib::TFilePos&  fposActual)
{
    tCIDLib::TFilePos fposActualPos;

    try
    {
        __kflThis.Seek(fposOffset, eSeekFrom, fposActualPos);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Seek
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }

    if (&fposActual)
        fposActual = fposActualPos;
}


tCIDLib::TVoid
TFileBase::_WriteBuf(   const   tCIDLib::TVoid* const pBuf
                        , const tCIDLib::TCard4       c4BufSz
                        ,       tCIDLib::TCard4&      c4Actual)
{
    tCIDLib::TCard4     c4ActualActual;

    // Do the write of the requested number of bytes
    try
    {
        __kflThis.WriteBuffer(pBuf, c4BufSz, c4ActualActual);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_Write
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(c4BufSz)
            , __strName
        );
    }

    // If the whole buffer was not written, log a warning in debug mode
    #if CID_DEBUG_ON
    if (c4ActualActual != c4BufSz)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_NotAllData
            , __strName
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_AppStatus
            , TString(L"write")
            , TCardinal(c4BufSz)
            , TCardinal(c4Actual)
        );
    }
    #endif

    if (&c4Actual)
        c4Actual = c4ActualActual;
}
