//
// NAME: CIDLib_Facility.Hpp
//
// DESCRIPTION:
//
//  This is the header file for the CIDLib_Facility.Cpp module. This
//  module provides the basic facility class. Every facility defines a
//  facility class and instantiates a single instance of it. Simple facilities
//  can just instantiate a TFacility object, but more complex ones will
//  create a derivative with its own data and functionality.
//
//  This class builds upon the TModule class. TModule is the simplest public
//  representation of a DLL or Exe module.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/25/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//

#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//  CLASS: TFacility
// PREFIX: fac
// ----------------------------------------------------------------------------
class CIDLIBEXP TFacility : public TModule
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors.
        // --------------------------------------------------------------------
        TFacility
        (
            const   TString&                strFacName
            , const tCIDLib::EModTypes      eModType
            , const tCIDLib::TCard4         c4MajVer
            , const tCIDLib::TCard4         c4MinVer
            , const tCIDLib::TBoolean       bLoad = kCIDLib::False
        );

        TFacility
        (
            const   tCIDLib::TModHandle     hmodToUse
            , const tCIDLib::EModTypes      eModType
            , const tCIDLib::TCard4         c4MajVer
            , const tCIDLib::TCard4         c4MinVer
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_NoAdopt
            , const tCIDLib::TBoolean       bViaLoad = kCIDLib::False
        );

        ~TFacility();


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TCard4 c4MajVersion() const;

        tCIDLib::TCard4 c4MinVersion() const;

        tCIDLib::TCard4 c4Version() const;

        const TString& strVersion() const;



    private :
        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TFacility();

        TFacility(const TFacility&);

        tCIDLib::TVoid operator=(const TFacility&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c2MajVer
        //  __c2MinVer
        //      This facility's major and minor version.
        //
        //  __eFacType
        //      The type of this facility, set during the constructor.
        //
        //  __kmodThis
        //      The Facility module object. This is set by the constructor,
        //      using the facility name provided by the caller.
        //
        //  __strName
        //      The name of the facility, passed in the constructor and
        //      stored here.
        //
        //  __strVersion
        //      This is the version string, which is formatted during
        //      construction, using the provided version values.
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4MajVer;
        tCIDLib::TCard4         __c4MinVer;
        TString                 __strVersion;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFacility,TObject)
};

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TCard4 TFacility::c4MajVersion() const
{
    return __c4MajVer;
}

inline tCIDLib::TCard4 TFacility::c4MinVersion() const
{
    return __c4MinVer;
}

inline tCIDLib::TCard4 TFacility::c4Version() const
{
    return TRawBits::c4From16
    (
        tCIDLib::TCard2(__c4MinVer)
        , tCIDLib::TCard2(__c4MajVer)
    );
}

inline const TString& TFacility::strVersion() const
{
    return __strVersion;
}
