//
// NAME: CIDLib_FArea.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_FArea.Cpp module, which implements
//  the TFArea class. This class represents a rectangular area  in a
//  floating point coordinate space. The internal representation is as
//  a left, right, top, and bottom value.
//
//  Include this header by defining:
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TFArea
// PREFIX: farea
// -----------------------------------------------------------------------------
class CIDLIBEXP TFArea :

    public TObject, public MStreamable, public MDuplicable, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and destructors
        // ---------------------------------------------------------------------
        TFArea();

        TFArea
        (
            const   TArea&                  areaFrom
            , const tCIDLib::TFloat8&       f8ScaleBy = 1.0
        );

        TFArea
        (
            const   tCIDLib::TFloat8&       f8Left
            , const tCIDLib::TFloat8&       f8Right
            , const tCIDLib::TFloat8&       f8Bottom
            , const tCIDLib::TFloat8&       f8Top
        );

        TFArea
        (
            const   T2DVector&              vecLL
            , const T2DVector&              vecUR
        );

        TFArea
        (
            const   TFArea&                 fareaToCopy
        );

        ~TFArea() {}


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TFArea&                 fareaToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TFArea&                 fareaToCompare
        )   const;

        TFArea& operator=
        (
            const   TFArea&                 fareaToAssign
        );


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TFloat8 f8Area() const;

        tCIDLib::TFloat8 f8Bottom() const;

        tCIDLib::TFloat8 f8Bottom
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8CX() const;

        tCIDLib::TFloat8 f8CXDiv
        (
            const   tCIDLib::TCard4         c4DivBy
        )   const;

        tCIDLib::TFloat8 f8CY() const;

        tCIDLib::TFloat8 f8CYDiv
        (
            const   tCIDLib::TCard4         c4DivBy
        )   const;

        tCIDLib::TFloat8 f8Left() const;

        tCIDLib::TFloat8 f8Left
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Right() const;

        tCIDLib::TFloat8 f8Right
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Top() const;

        tCIDLib::TFloat8 f8Top
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TVoid SetAll
        (
            const   tCIDLib::TFloat8&       f8Left
            , const tCIDLib::TFloat8&       f8Right
            , const tCIDLib::TFloat8&       f8Bottom
            , const tCIDLib::TFloat8&       f8Top
        );

        tCIDLib::TVoid SetOrg
        (
            const   T2DVector&              vecToSet
        );

        tCIDLib::TVoid SetLowerRight
        (
            const   T2DVector&              vecToSet
        );

        T2DVector vecLL() const;

        T2DVector vecLR() const;

        T2DVector vecUL() const;

        T2DVector vecUR() const;


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        ) const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToWriteTo
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid __AdjustCoordinates();


        // ---------------------------------------------------------------------
        // Private data members
        //
        //  __f8Left
        //  __f8Right
        //  __f8Bottom
        //  __f8Top
        //      These are the 8 coordinates the define the rectangular area.
        // ---------------------------------------------------------------------
        tCIDLib::TFloat8        __f8Left;
        tCIDLib::TFloat8        __f8Right;
        tCIDLib::TFloat8        __f8Bottom;
        tCIDLib::TFloat8        __f8Top;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFArea,TObject)
        DefPolyDup(TFArea)
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Constructors 
// -----------------------------------------------------------------------------
inline TFArea::TFArea() :

    __f8Bottom(0.0)
    , __f8Left(0.0)
    , __f8Right(0.0)
    , __f8Top(0.0)
{
}

inline TFArea::TFArea(const TFArea& areaFrom) :

    __f8Bottom(areaFrom.__f8Bottom)
    , __f8Left(areaFrom.__f8Left)
    , __f8Right(areaFrom.__f8Right)
    , __f8Top(areaFrom.__f8Top)
{
}

inline TFArea::TFArea(  const   TArea&              areaFrom
                        , const tCIDLib::TFloat8&   f8ScaleBy) :

    __f8Bottom(tCIDLib::TFloat8(areaFrom.i4Bottom()) * f8ScaleBy)
    , __f8Left(tCIDLib::TFloat8(areaFrom.i4Left()) * f8ScaleBy)
    , __f8Right(tCIDLib::TFloat8(areaFrom.i4Right()) * f8ScaleBy)
    , __f8Top(tCIDLib::TFloat8(areaFrom.i4Top()) * f8ScaleBy)
{
}

inline TFArea::TFArea(  const   tCIDLib::TFloat8&   f8Left
                        , const tCIDLib::TFloat8&   f8Right
                        , const tCIDLib::TFloat8&   f8Bottom
                        , const tCIDLib::TFloat8&   f8Top) :
    __f8Bottom(f8Bottom)
    , __f8Left(f8Left)
    , __f8Right(f8Right)
    , __f8Top(f8Top)
{
    __AdjustCoordinates();
}


// -----------------------------------------------------------------------------
//  Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TFArea::operator!=(const TFArea& fareaToCompare) const
{
    return !operator==(fareaToCompare);
}


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TFloat8 TFArea::f8Area() const
{
    return ((__f8Right - __f8Left) * (__f8Top - __f8Bottom));
}

inline tCIDLib::TFloat8 TFArea::f8Bottom() const
{
    return __f8Bottom;
}

inline tCIDLib::TFloat8 TFArea::f8CX() const
{
    return __f8Right - __f8Left;
}

inline tCIDLib::TFloat8 TFArea::f8CXDiv(const tCIDLib::TCard4 c4DivBy) const
{
    return (__f8Right - __f8Left) / tCIDLib::TFloat8(c4DivBy);
}

inline tCIDLib::TFloat8 TFArea::f8CY() const
{
    return __f8Top - __f8Bottom;
}

inline tCIDLib::TFloat8 TFArea::f8CYDiv(const tCIDLib::TCard4 c4DivBy) const
{
    return (__f8Top - __f8Bottom) / tCIDLib::TFloat8(c4DivBy);
}

inline tCIDLib::TFloat8 TFArea::f8Left() const
{
    return __f8Left;
}

inline tCIDLib::TFloat8 TFArea::f8Right() const
{
    return __f8Right;
}

inline tCIDLib::TFloat8 TFArea::f8Top() const
{
    return __f8Top;
}

inline tCIDLib::TVoid
TFArea::SetAll( const   tCIDLib::TFloat8&   f8Left
                , const tCIDLib::TFloat8&   f8Right
                , const tCIDLib::TFloat8&   f8Bottom
                , const tCIDLib::TFloat8&   f8Top)
{
    __f8Bottom  = f8Bottom;
    __f8Left    = f8Left;
    __f8Right   = f8Right;
    __f8Top     = f8Top;

    __AdjustCoordinates();
}

inline T2DVector TFArea::vecLL() const
{
    return T2DVector(__f8Left, __f8Bottom);
}

inline T2DVector TFArea::vecLR() const
{
    return T2DVector(__f8Right, __f8Bottom);
}

inline T2DVector TFArea::vecUL() const
{
    return T2DVector(__f8Left, __f8Top);
}

inline T2DVector TFArea::vecUR() const
{
    return T2DVector(__f8Right, __f8Top);
}
