//
// NAME: CIDLib_FArea.Cpp
//
// DESCRIPTION:
//
//	This module implements the TFArea class, a 2D rectangular area in a
//  floating point coordinate space.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(TFArea,TObject)



// -----------------------------------------------------------------------------
//   CLASS: TFArea
//  PREFIX: farea
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFArea: Constructors and destructors
// -----------------------------------------------------------------------------

TFArea::TFArea(const T2DVector& vecLL, const T2DVector& vecUR) :

    __f8Bottom(vecLL.f8X())
    , __f8Left(vecLL.f8Y())
    , __f8Right(vecUR.f8X())
    , __f8Top(vecUR.f8Y())
{
    __AdjustCoordinates();
}


// -----------------------------------------------------------------------------
//  TFArea: Public operators
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TFArea::operator==(const TFArea& fareaToCompare) const
{
    if ((__f8Bottom == fareaToCompare.__f8Bottom)
    &&  (__f8Left == fareaToCompare.__f8Left)
    &&  (__f8Right == fareaToCompare.__f8Right)
    &&  (__f8Top == fareaToCompare.__f8Top))
    {
        return kCIDLib::True;
    }
    return kCIDLib::False;
}

TFArea& TFArea::operator=(const TFArea& fareaToAssign)
{
    __f8Bottom  = fareaToAssign.__f8Bottom;
    __f8Left    = fareaToAssign.__f8Left;
    __f8Right   = fareaToAssign.__f8Right;
    __f8Top     = fareaToAssign.__f8Top;
    return *this;
}


// -----------------------------------------------------------------------------
//  TFArea: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8 TFArea::f8Bottom(const tCIDLib::TFloat8& f8New)
{
    if (f8New > __f8Top)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BadCoord
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TString(L"Bottom")
            , TFloat(f8New)
            , TString(L"top")
            , TFloat(__f8Top)
        );
    }
    __f8Bottom = f8New;
    return f8New;
}

tCIDLib::TFloat8 TFArea::f8Left(const tCIDLib::TFloat8& f8New)
{
    if (f8New > __f8Right)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BadCoord
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TString(L"Left")
            , TFloat(f8New)
            , TString(L"right")
            , TFloat(__f8Right)
        );
    }
    __f8Left = f8New;
    return f8New;
}

tCIDLib::TFloat8 TFArea::f8Right(const tCIDLib::TFloat8& f8New)
{
    if (f8New < __f8Left)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BadCoord
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TString(L"Right")
            , TFloat(f8New)
            , TString(L"left")
            , TFloat(__f8Left)
        );
    }
    __f8Right = f8New;
    return f8New;
}

tCIDLib::TFloat8 TFArea::f8Top(const tCIDLib::TFloat8& f8New)
{
    if (f8New < __f8Bottom)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcArea_BadCoord
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TString(L"Top")
            , TFloat(f8New)
            , TString(L"bottom")
            , TFloat(__f8Bottom)
        );
    }
    __f8Top = f8New;
    return f8New;
}


tCIDLib::TVoid TFArea::SetOrg(const T2DVector& vecToSet)
{
    f8Left(vecToSet.f8X());
    f8Top(vecToSet.f8Y());
}

tCIDLib::TVoid TFArea::SetLowerRight(const T2DVector& vecToSet)
{
    f8Right(vecToSet.f8X());
    f8Bottom(vecToSet.f8Y());
}


// -----------------------------------------------------------------------------
//  TFArea: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFArea::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Save the stream state and set up a new one
    TStreamJanitor janWrite(&strmToWriteTo);
    strmToWriteTo   << TTextStream::Width(0)
                    << TTextStream::Precision(4)
                    << TTextStream::Justify(tCIDLib::EHJustify_Right)
                    << TTextStream::Fill(kCIDLib::chSpace);

    strmToWriteTo   << L"{"
                    << __f8Left << L","
                    << __f8Bottom << L"-"
                    << __f8Right << L","
                    << __f8Top
                    << L"}";
}


tCIDLib::TVoid TFArea::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __f8Bottom;
    strmToReadFrom >> __f8Left;
    strmToReadFrom >> __f8Right;
    strmToReadFrom >> __f8Top;
}

tCIDLib::TVoid TFArea::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __f8Bottom;
    strmToWriteTo << __f8Left;
    strmToWriteTo << __f8Right;
    strmToWriteTo << __f8Top;
}



// -----------------------------------------------------------------------------
//  TFArea: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __AdjustCoordinates
//
// DESCRIPTION:
//
//  This is called after the coordinates are set by the outside world. It will
//  flip them as needed to keep the left and bottom less than the top and
//  right. Its only called when they are all set at once. When individual
//  members are set, its illegal to do them wrong.
//
//  !We have to export this private because its accessed via inlines!
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFArea::__AdjustCoordinates()
{
    if (__f8Bottom > __f8Top)
        Swap(__f8Bottom, __f8Top);

    if (__f8Left > __f8Right)
        Swap(__f8Left, __f8Right);
}
