//
// NAME: CIDLib_ExternalProcess.Cpp
//
// DESCRIPTION:
//
//  This module implements the TExternalProcess class, which supports the
//  execution and control of other processes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 09/13/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TExternalProcess,TObject)


// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------

static tCIDLib::TVoid
__CleanUpStrArray(const tCIDLib::Tch** apszToClean, const tCIDLib::TCard4 c4Count)
{
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
        delete (tCIDLib::Tch*)apszToClean[c4Index];

    delete [] apszToClean;
}


// -----------------------------------------------------------------------------
//   CLASS: TExternalProcess
//  PREFIX: extp
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TExternalProcess: Constructors and destructors
// -----------------------------------------------------------------------------

TExternalProcess::TExternalProcess() :

    __envThis(kCIDLib::True)
{
}

TExternalProcess::TExternalProcess( const   TString&        strPath
                                    , const TEnvironment&   envToUse) :

    __envThis(envToUse)
    , __strPath(strPath)
{
}

TExternalProcess::TExternalProcess(const TString& strPath) :

    __envThis(kCIDLib::True)
    , __strPath(strPath)
{
}

TExternalProcess::~TExternalProcess()
{
}


// -----------------------------------------------------------------------------
//  TExternalProcess: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TExternalProcess::bIsRunning() const
{
    try
    {
        return __kextpThis.bIsRunning();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.ThrowKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_TestState
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strPath
        );
    }
}

tCIDLib::EPrioClasses TExternalProcess::ePriorityClass() const
{
    if (!bIsRunning())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_NotRunning
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_NotReady
            , __strPath
        );
    }

    try
    {
        return __kextpThis.ePriorityClass();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.ThrowKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_GetPriority
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strPath
        );
    }
}

tCIDLib::EPrioClasses 
TExternalProcess::ePriorityClass(const tCIDLib::EPrioClasses eNewClass)
{
    if (!bIsRunning())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_NotRunning
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_NotReady
            , __strPath
        );
    }

    try
    {
        return __kextpThis.ePriorityClass(eNewClass);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.ThrowKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_SetPriority
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strPath
        );
    }
}

tCIDLib::EExitCodes
TExternalProcess::eWaitForDeath(const tCIDLib::TCard4 c4MilliSecs) const
{
    try
    {
        return __kextpThis.eWaitForDeath(c4MilliSecs);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        tCIDLib::TErrCode       errcToThrow = kCIDErrs::errcExtP_WaitFor;
        tCIDLib::EErrClasses    eClass = tCIDLib::EClass_CantDo;

        if (kerrToCatch.errcId() == kKrnlErrors::errcTimeout)
        {
            errcToThrow = kCIDErrs::errcExtP_Timeout;
            eClass = tCIDLib::EClass_Timeout;
        }

        facCIDLib.ThrowKrnlErr
        (
            __FILE__
            , __LINE__
            , errcToThrow
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , eClass
            , __strPath
        );
    }
}

TEnvironment& TExternalProcess::envThis()
{
    if (bIsRunning())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_NotWhileRunning
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
            , __strPath
        );
    }
    return __envThis;
}

tCIDLib::TVoid TExternalProcess::Kill()
{
    try
    {
        __kextpThis.Kill();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        tCIDLib::TErrCode       errcToThrow = kCIDErrs::errcExtP_Kill;
        tCIDLib::EErrClasses    eClass = tCIDLib::EClass_CantDo;

        if (kerrToCatch.errcId() == kKrnlErrors::errcNotRunning)
        {
            errcToThrow = kCIDErrs::errcExtP_NotRunning;
            eClass = tCIDLib::EClass_NotReady;
        }

        facCIDLib.ThrowKrnlErr
        (
            __FILE__
            , __LINE__
            , errcToThrow
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , eClass
            , __strPath
        );
    }
}

tCIDLib::TVoid TExternalProcess::Start(const tCIDLib::TBoolean bDetached)
{
    Start(NUL_TString, *((TCollection<TString>*)0), NUL_TEnvironment, bDetached);
}

tCIDLib::TVoid TExternalProcess::Start( const   TString&            strNewPath
                                        , const tCIDLib::TBoolean   bDetached)
{
    Start
    (
        NUL_TString
        , *((TCollection<TString>*)0)
        , NUL_TEnvironment
        , bDetached
    );
}

tCIDLib::TVoid
TExternalProcess::Start(const   TString&                strNewPath
                        , const TCollection<TString>&   colParms
                        , const tCIDLib::TBoolean       bDetached)
{
    Start(NUL_TString, colParms, NUL_TEnvironment, bDetached);
}

tCIDLib::TVoid
TExternalProcess::Start(const   TString&                strNewPath
                        , const TCollection<TString>&   colParms
                        , const TEnvironment&           envNew
                        , const tCIDLib::TBoolean       bDetached)
{
    if (bIsRunning())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_NotWhileRunning
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
            , __strPath
        );
    }

    // If there is a new path, then store it as our new path.
    if (&strNewPath)
        __strPath = strNewPath;

    // If there is a new environment, then store it as our new environment
    if (&envNew)
        __envThis = envNew;

    //
    //  If we ended up with an empty path, then its obviously not been set
    //  up correctly.
    //
    if (__strPath.bEmpty())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_EmptyPath
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    //
    //  Ok, so lets build up the raw parameter and environment strings that
    //  the kernel needs to do its thing.
    //
    const tCIDLib::Tch**    apszEnv = 0;
    const tCIDLib::Tch**    apszCmdLine = 0;
    tCIDLib::TCard4         c4EnvCount = 0;
    tCIDLib::TCard4         c4ParmCount = 0;
    TElemCursor<TString>*   pcursParms = 0;
    TString                 strTmp;
    try
    {
        c4EnvCount = __envThis.c4Entries();
        if (c4EnvCount)
        {
            apszEnv = new const tCIDLib::Tch*[c4EnvCount];

            TEnvironment::TCursor cursEnv(&__envThis);
            cursEnv.bReset();
            tCIDLib::TCard4 c4Index = 0;
            do
            {
                strTmp = cursEnv.objCur().strKey();
                strTmp.Append(kCIDLib::chEquals);
                strTmp.Append(cursEnv.objCur().strValue());
                apszEnv[c4Index++] = TRawStr::pszReplicate(strTmp.pszData());
            }   while (cursEnv.bNext());
        }

        if (&colParms && colParms.c4ElemCount())
        {
            // Our actual count must include the program, so add !
            c4ParmCount = colParms.c4ElemCount() + 1;
            apszCmdLine = new const tCIDLib::Tch*[c4ParmCount];

            // Set up the program name as the 0th element
            apszCmdLine[0] = TRawStr::pszReplicate(__strPath.pszData());

            // Get a new cursor from the collection
            pcursParms = colParms.pcursNew();
            pcursParms->bReset();

            // Start the index at 1 since we put the program name in 0
            tCIDLib::TCard4 c4Index = 1;
            do
            {
                apszCmdLine[c4Index++] =
                            TRawStr::pszReplicate(pcursParms->objCur().pszData());
            }   while (pcursParms->bNext());
        }
         else
        {
            // There are no parms, but we still need to put the program name in
            c4ParmCount = 1;
            apszCmdLine = new const tCIDLib::Tch*[c4ParmCount];
            apszCmdLine[0] = TRawStr::pszReplicate(__strPath.pszData());
        }

        //
        //  Ok, lets start up this puppy. We just call our kernel object
        //  and pass along this info we've set up.
        //
        __kextpThis.Start
        (
            0
            , apszCmdLine
            , (apszCmdLine ? c4ParmCount : 0)
            , apszEnv
            , (apszEnv ? c4EnvCount : 0)
            , bDetached
        );

        // Clean up the buffers
        __CleanUpStrArray(apszEnv, c4EnvCount);
        apszEnv = 0;
        __CleanUpStrArray(apszCmdLine, c4ParmCount);
        apszCmdLine = 0;
        delete pcursParms;
        pcursParms = 0;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        __CleanUpStrArray(apszEnv, c4EnvCount);
        apszEnv = 0;
        __CleanUpStrArray(apszCmdLine, c4ParmCount);
        apszCmdLine = 0;
        delete pcursParms;
        pcursParms = 0;

        facCIDLib.ThrowKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcExtP_CantStart
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __strPath
        );
    }

    catch(...)
    {
        __CleanUpStrArray(apszEnv, c4EnvCount);
        apszEnv = 0;
        __CleanUpStrArray(apszCmdLine, c4ParmCount);
        apszCmdLine = 0;
        delete pcursParms;
        pcursParms = 0;

        throw;
    }
}
