//
// NAME: CIDLib_Event.Cpp
//
// DESCRIPTION:
//
//  This module implements the TEvent class.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/20/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TEvent,TObject)



// ----------------------------------------------------------------------------
//   CLASS: TEvent
//  PREFIX: ev
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TEvent: Constructors and Destructors
// ----------------------------------------------------------------------------

TEvent::TEvent( const   tCIDLib::EEventStates   eInitState
                , const tCIDLib::EAutoModes     eAutoReset
                , const tCIDLib::EShareStates   eShareState) :

    __bNamed(kCIDLib::False)
    , __kevImpl(eShareState)
{
    try
    {
        __kevImpl.Create(eInitState, eAutoReset);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Create
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
        );
    }
}

TEvent::TEvent( const   TResourceName&          rsnToUse
                , const tCIDLib::EAutoModes     eAutoReset
                , const tCIDLib::EEventStates   eInitState
                , const tCIDLib::ECreateActions eAction) :

    __bNamed(kCIDLib::True)
    , __kevImpl(rsnToUse.strFullName(tCIDLib::ENamedRsc_Event).pszData())
    , __rsnThis(rsnToUse)
{
    #if CID_DEBUG_ON
    //
    //  Check for illegal create actions for this type of operation.
    //  
    if ((eAction == tCIDLib::ECreateAct_ReplaceIfExists)
    ||  (eAction == tCIDLib::ECreateAct_TruncateIfExists))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_CreateAction
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TString(L"'Replace/truncate if exists'")
        );
    }
    #endif

    //
    //  First try to open the semaphore. Then see if the open action
    //  makes this a bad thing.
    //
    try
    {
        __kevImpl.Open();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        //
        //  If it failed and we were asked only to open if it exists,
        //  then log and error.
        //
        if ((kerrToCatch.errcId() == kKrnlErrors::errcFileNotFound)
        &&  (eAction == tCIDLib::ECreateAct_OpenIfExists))
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEv_Open
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
            );
        }

        //
        //  If we were asked to create it, then lets try that and see if
        //  it works any better. If we were not asked to create it, then
        //  give up.
        //
        if ((eAction == tCIDLib::ECreateAct_CreateIfNew)
        ||  (eAction == tCIDLib::ECreateAct_OpenOrCreate))
        {
            try
            {
                __kevImpl.Create(eInitState, eAutoReset);
            }

            catch(const TKrnlError& kerrToCatch)
            {
                facCIDLib.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcEv_Create
                    , kerrToCatch
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                    , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
                );
            }
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEv_Open
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
            );
        }
    }

    //
    //  We opened it. If the open action indicates that we should fail if
    //  it exists, then close it an fail.
    //
    if (eAction == tCIDLib::ECreateAct_FailIfExists)
    {
        __kevImpl.Close();
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_AlreadyExists
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
            , TString(L"TEvent")
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
        );
    }
}

TEvent::~TEvent()
{
    try
    {
        __kevImpl.Close();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        //
        //  If we get an error, log it but don't throw out of the destructor
        //  so we use a warning message.
        //
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Close
            , kerrToCatch
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_Internal
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
        );
    }
}


// ----------------------------------------------------------------------------
//  TEvent: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TEvent::Pulse()
{
    try
    {
        __kevImpl.Pulse();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Pulse
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
        );
    }
}


tCIDLib::TVoid TEvent::Reset()
{
    try
    {
        __kevImpl.Reset();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Reset
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
        );
    }
}


tCIDLib::TVoid TEvent::Trigger()
{
    try
    {
        __kevImpl.Trigger();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Trigger
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
        );
    }
}


tCIDLib::TVoid TEvent::WaitFor(const tCIDLib::TCard4 c4Timeout)
{
    try
    {
        __kevImpl.WaitFor(c4Timeout);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if (kerrToCatch.errcId() == kKrnlErrors::errcTimeout)
        {
            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEv_Timeout
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEv_Wait
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Event)
            );
        }
    }
}


// ----------------------------------------------------------------------------
//  TEvent: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TEvent::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << L"Name="
                    << __rsnThis.strFullName(tCIDLib::ENamedRsc_Event);
}
