//
// NAME: CIDLib_Error.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_Error.Cpp module, which implements
//  the standard object thrown when errors are logged in the CIDLib system.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 02/07/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//  CLASS: TError
// PREFIX: err
// ----------------------------------------------------------------------------
class CIDLIBEXP TError :

    public TObject, public MDuplicable, public MStreamable, public MFormattable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors.
        // --------------------------------------------------------------------
        TError();

        //
        //  For logging inline text messages. The default severity will cause
        //  it to always go to the logs. The error code is set to 0.
        //
        TError
        (
            const   TString&                strModName
            , const TString&                strFileName
            , const tCIDLib::TCard4         c4LineNum
            , const TString&                strMsgText
            , const tCIDLib::ESeverities    eSev = tCIDLib::ESev_Status
            , const tCIDLib::EErrClasses    eClass = tCIDLib::EClass_None
        );

        TError
        (
            const   TString&                strModName
            , const TString&                strFileName
            , const tCIDLib::TCard4         c4LineNum
            , const tCIDLib::TErrCode       errcCIDErr
            , const TString&                strErrText
            , const TString&                strAuxText
            , const tCIDLib::ESeverities    eSeverity
            , const tCIDLib::EErrClasses    eClass
        );

        TError
        (
            const   TString&                strModName
            , const TString&                strFileName
            , const tCIDLib::TCard4         c4LineNum
            , const tCIDLib::TErrCode       errcId
            , const TKrnlError&             kerrIds
            , const TString&                strErrText
            , const TString&                strAuxText
            , const tCIDLib::ESeverities    eSeverity
            , const tCIDLib::EErrClasses    eClass
        );

        TError
        (
            const   TError&                 errToCopy
        );

        ~TError();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TError& operator=
        (
            const   TError&                 errToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TError&                 errToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TError&                 errToTest
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bAnyError() const;

        tCIDLib::TBoolean bCheckError
        (
            const   TModule&                modToCheck
            , const tCIDLib::TErrCode       errcToCheck
        )   const;

        tCIDLib::TCard4 c4LineNum() const;

        tCIDLib::TCard4 c4LineNum
        (
            const   tCIDLib::TCard4         c4NewLineNum
        );

        tCIDLib::TOSErrCode errcHostId() const;

        tCIDLib::TOSErrCode errcHostId
        (
            const   tCIDLib::TOSErrCode     errcNew
        );

        tCIDLib::TErrCode errcId() const;

        tCIDLib::TErrCode errcId
        (
            const   tCIDLib::TErrCode       errcNew
        );

        tCIDLib::TErrCode errcKrnlId() const;

        tCIDLib::TErrCode errcKrnlId
        (
            const   tCIDLib::TErrCode       errcNew
        );

        tCIDLib::EErrClasses eClass() const;

        tCIDLib::EErrClasses eClass
        (
            const   tCIDLib::EErrClasses    eNewClass
        );

        tCIDLib::ESeverities eSeverity() const;

        tCIDLib::ESeverities eSeverity
        (
            const   tCIDLib::ESeverities    eNewSev
        );

        const TString& strAuxText() const;

        const TString& strAuxText
        (
            const   TString&                strNewText
        );

        const TString& strErrText() const;

        const TString& strErrText
        (
            const   TString&                strNewText
        );

        const TString& strModName() const;

        const TString& strModName
        (
            const   TString&                strNewName
        );

        const TString& strFileName() const;

        const TString& strFileName
        (
            const   TString&                strNewName
        );

        const TString& strThread() const;

        const TString& strThread
        (
            const   TString&                strNewName
        );


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private       :
        // --------------------------------------------------------------------
        //  Private data
        //
        //  __c4LineNum
        //      The number of the line on which the error is reported. It can
        //      easily be provided by the __LINE__ macro and most facilities
        //      provide an internal macro to provide these automatically.
        //
        //  __eClass
        //      And indicator of the cause of the error. It may be EClass_None
        //      if the cause is unknown.
        //
        //  __errcHostId
        //      The underlying host error code, if any. If none, then it will
        //      be 0.
        //
        //  __errcId
        //      This is the CIDLib error code. What error it really maps to
        //      is of course relative to the module that it came from.
        //
        //  __errcKrnlId
        //      This the underlying kernel error code, if any. If none, then
        //      it will be 0.
        //
        //  __eSeverity
        //      The severity level of the error, tells the caller how bad
        //      it was.
        //
        //  __strAuxText
        //      This is the storage for the auxiliary text string. This is to
        //      give an extra spin on the error text.
        //
        //  __strErrText
        //      This is the storage for the error text. This may be set
        //      manually with a string or by loading the error text from a
        //      message file and storing it here.
        //
        //  __strModName
        //      This is the name of the module that returned the error. It
        //      is provided for convenience and so that it can be reported
        //      along with the other information.
        //
        //  __strFileName
        //      This is the name of the file in which the error occured.
        //      This is easily provided via the __FILE__ macro.
        //
        //  __strThread
        //      This is the name of the thread that had the error
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4LineNum;
        tCIDLib::EErrClasses    __eClass;
        tCIDLib::TOSErrCode     __errcHostId;
        tCIDLib::TErrCode       __errcId;
        tCIDLib::TErrCode       __errcKrnlId;
        tCIDLib::ESeverities    __eSeverity;
        TString                 __strAuxText;
        TString                 __strErrText;
        TString                 __strModName;
        TString                 __strFileName;
        TString                 __strThread;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TError,TObject)
        DefPolyDup(TError)
};

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TError::operator!=(const TError& errToTest) const
{
    return !operator==(errToTest);
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TError::bAnyError() const
{
    return (__errcHostId || __errcKrnlId || __errcId);
}

inline tCIDLib::TCard4 TError::c4LineNum() const
{
    return __c4LineNum;
}

inline tCIDLib::TCard4 TError::c4LineNum(const tCIDLib::TCard4 c4NewLineNum)
{
    __c4LineNum = c4NewLineNum;
    return __c4LineNum;
}

inline tCIDLib::TOSErrCode TError::errcHostId() const
{
    return __errcHostId;
}

inline tCIDLib::TOSErrCode
TError::errcHostId(const tCIDLib::TOSErrCode errcNew)
{
    __errcHostId = errcNew;
    return __errcHostId;
}

inline tCIDLib::TErrCode TError::errcId() const
{
    return __errcId;
}

inline tCIDLib::TErrCode TError::errcId(const tCIDLib::TErrCode errcNew)
{
    __errcId = errcNew;
    return __errcId;
}

inline tCIDLib::TErrCode TError::errcKrnlId() const
{
    return __errcKrnlId;
}

inline tCIDLib::TErrCode TError::errcKrnlId(const tCIDLib::TErrCode errcNew)
{
    __errcKrnlId = errcNew;
    return __errcKrnlId;
}

inline tCIDLib::EErrClasses TError::eClass() const
{
    return __eClass;
}

inline tCIDLib::EErrClasses
TError::eClass(const tCIDLib::EErrClasses eNewCause)
{
    __eClass = eNewCause;
    return __eClass;
}

inline tCIDLib::ESeverities TError::eSeverity() const
{
    return __eSeverity;
}

inline tCIDLib::ESeverities
TError::eSeverity(const tCIDLib::ESeverities eNewSev)
{
    __eSeverity = eNewSev;
    return __eSeverity;
}

inline const TString& TError::strAuxText() const
{
    return __strAuxText;
}

inline const TString& TError::strAuxText(const TString& strNewText)
{
    __strAuxText = strNewText;
    return __strAuxText;
}

inline const TString& TError::strErrText() const
{
    return __strErrText;
}

inline const TString& TError::strErrText(const TString& strNewText)
{
    __strErrText = strNewText;
    return __strErrText;
}

inline const TString& TError::strModName() const
{
    return __strModName;
}

inline const TString& TError::strModName(const TString& strNewName)
{
    __strModName = strNewName;
    return __strModName;
}

inline const TString& TError::strFileName() const
{
    return __strFileName;
}

inline const TString& TError::strFileName(const TString& strNewName)
{
    __strFileName = strNewName;
    return __strFileName;
}

inline const TString& TError::strThread() const
{
    return __strThread;
}

inline const TString& TError::strThread(const TString& strNewName)
{
    __strThread = strNewName;
    return __strThread;
}
