//
// NAME: CIDLib_Error.Cpp
//
// DESCRIPTION:
//
//  This module provides support for error reporting. It implements the very
//  simple TError class. This class is used by all Facilities to report
//  errors.
//
//  TError objects are almost never created manually. TModule provides a set
//  of logging/throwing methods that will create the error and log it or throw
//  it or both. All facilities inherit this capability, so there is never any
//  reason to build error objects out in the outside world.
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 02/07/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TError,TObject)



// -----------------------------------------------------------------------------
//   CLASS: TError
//  PREFIX: err
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TError: Constructors and Destructors
// -----------------------------------------------------------------------------

TError::TError() :

    __c4LineNum(0)
    , __eClass(tCIDLib::EClass_Unknown)
    , __errcHostId(0)
    , __errcId(0)
    , __eSeverity(tCIDLib::ESev_Information)
{
}

TError::TError( const   TString&                strModName
                , const TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const TString&                strErrText
                , const tCIDLib::ESeverities    eSev
                , const tCIDLib::EErrClasses    eClass) :

    __c4LineNum(c4LineNum)
    , __eClass(eClass)
    , __errcHostId(0)
    , __errcId(0)
    , __errcKrnlId(0)
    , __eSeverity(eSev)
    , __strErrText(strErrText)
    , __strModName(strModName)
{
    //
    //  Strip the path part off the file name, because its almost always
    //  fed by the file macro that puts the whole friggin path on there.
    //
    TPathStr pathTmp(strFileName);
    pathTmp.bQueryNameExt(__strFileName);

    // Get the calling thread's name
    __strThread = TThread::pthrCaller()->strName();
}

TError::TError( const   TString&                strModName
                , const TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const tCIDLib::TErrCode       errcId
                , const TString&                strErrText
                , const TString&                strAuxText
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass) :

    __c4LineNum(c4LineNum)
    , __eClass(eClass)
    , __errcHostId(0)
    , __errcId(errcId)
    , __errcKrnlId(0)
    , __eSeverity(eSeverity)
    , __strAuxText(strAuxText)
    , __strErrText(strErrText)
    , __strModName(strModName)
{
    //
    //  Strip the path part off the file name, because its almost always
    //  fed by the file macro that puts the whole friggin path on there.
    //
    TPathStr pathTmp(strFileName);
    pathTmp.bQueryNameExt(__strFileName);

    // Get the calling thread's name
    __strThread = TThread::pthrCaller()->strName();
}

TError::TError( const   TString&                strModName
                , const TString&                strFileName
                , const tCIDLib::TCard4         c4LineNum
                , const tCIDLib::TErrCode       errcId
                , const TKrnlError&             kerrIds
                , const TString&                strErrText
                , const TString&                strAuxText
                , const tCIDLib::ESeverities    eSeverity
                , const tCIDLib::EErrClasses    eClass) :

    __c4LineNum(c4LineNum)
    , __eClass(eClass)
    , __errcHostId(kerrIds.errcHostId())
    , __errcId(errcId)
    , __errcKrnlId(kerrIds.errcId())
    , __eSeverity(eSeverity)
    , __strAuxText(strAuxText)
    , __strErrText(strErrText)
    , __strModName(strModName)
{
    //
    //  Strip the path part off the file name, because its almost always
    //  fed by the file macro that puts the whole friggin path on there.
    //
    TPathStr pathTmp(strFileName);
    pathTmp.bQueryNameExt(__strFileName);

    // Get the calling thread's name
    __strThread = TThread::pthrCaller()->strName();
}

TError::TError(const TError& errSource) :

    __c4LineNum(errSource.__c4LineNum)
    , __eClass(errSource.__eClass)
    , __errcHostId(errSource.__errcHostId)
    , __errcId(errSource.__errcId)
    , __errcKrnlId(errSource.__errcKrnlId)
    , __eSeverity(errSource.__eSeverity)
    , __strAuxText(errSource.__strAuxText)
    , __strErrText(errSource.__strErrText)
    , __strModName(errSource.__strModName)
    , __strFileName(errSource.__strFileName)
    , __strThread(errSource.__strThread)
{
}

TError::~TError()
{
}


// -----------------------------------------------------------------------------
//  TError: Public operators
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TError::operator==(const TError& errToTest) const
{
    //
    //  Put the stuff most likely to be different in different error objects
    //  first, so we don't waste more time than necessary.
    //
    if ((errToTest.__errcId != __errcId)
    ||  (errToTest.__errcKrnlId != __errcKrnlId)
    ||  (errToTest.__errcHostId != __errcHostId))
    {
        return kCIDLib::False;
    }

    // Then do the stuff that's less likely but fast to compare
    if ((errToTest.__c4LineNum != __c4LineNum)
    ||  (errToTest.__eClass != __eClass)
    ||  (errToTest.__eSeverity != __eSeverity))
    {
        return kCIDLib::False;
    }

    // And then just bull through the high overhead suff
    if (errToTest.__strAuxText != __strAuxText)
        return kCIDLib::False;

    if (errToTest.__strErrText != __strErrText)
        return kCIDLib::False;

    if (errToTest.__strModName != __strModName)
        return kCIDLib::False;

    if (errToTest.__strFileName != __strFileName)
        return kCIDLib::False;

    if (errToTest.__strThread != __strThread)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TError: Public, virtual methods
// -----------------------------------------------------------------------------

TError& TError::operator=(const TError& errToAssign)
{
    if (this == &errToAssign)
        return *this;

    __errcHostId    = errToAssign.__errcHostId;
    __c4LineNum     = errToAssign.__c4LineNum;
    __eClass        = errToAssign.__eClass;
    __eSeverity     = errToAssign.__eSeverity;
    __strAuxText    = errToAssign.__strAuxText;
    __errcHostId    = errToAssign.__errcHostId;
    __errcId        = errToAssign.__errcId;
    __errcKrnlId    = errToAssign.__errcKrnlId;
    __strErrText    = errToAssign.__strErrText;
    __strModName    = errToAssign.__strModName;
    __strFileName   = errToAssign.__strFileName;
    __strThread     = errToAssign.__strThread;

    return *this;
}


// -----------------------------------------------------------------------------
//  TError: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean
TError::bCheckError(const   TModule&            modToCheck
                    , const tCIDLib::TErrCode   errcToCheck) const
{
    if ((__errcId == errcToCheck) && (modToCheck.strName() == __strModName))
        return kCIDLib::True;
    return kCIDLib::False;
}


// -----------------------------------------------------------------------------
//  TError: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TError::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the error info into the string
    strmToWriteTo   << L"(" << __strModName << L"," << __strThread << L","
                    << __strFileName << L"." << __c4LineNum
                    << L"," << __eSeverity << L"/" << __eClass
                    << L",Err=" << __errcId << L") "
                    << __strErrText;

    // If there is a kernel error, then format it
    if (__errcKrnlId)
        strmToWriteTo << L", Kernel Err=" << TCardinal(__errcKrnlId);

    // If there is a host OS error, then format it
    if (__errcHostId)
        strmToWriteTo << L", Host Err=" << TCardinal(__errcHostId);

    /// If there is an aux text then format it
    if (__strAuxText.c4Length())
        strmToWriteTo << L", " << __strAuxText;
}


tCIDLib::TVoid TError::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Parse out the fundamental data type members
    strmToReadFrom >> __errcHostId;
    strmToReadFrom >> __errcId;
    strmToReadFrom >> __errcKrnlId;
    strmToReadFrom >> __c4LineNum;
    strmToReadFrom >> __eClass;
    strmToReadFrom >> __eSeverity;

    // Now parse out our object members
    strmToReadFrom >> __strAuxText;
    strmToReadFrom >> __strErrText;
    strmToReadFrom >> __strModName;
    strmToReadFrom >> __strFileName;
    strmToReadFrom >> __strThread;
}

tCIDLib::TVoid TError::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the fundamental data type members
    strmToWriteTo << __errcHostId;
    strmToWriteTo << __errcId;
    strmToWriteTo << __errcKrnlId;
    strmToWriteTo << __c4LineNum;
    strmToWriteTo << __eClass;
    strmToWriteTo << __eSeverity;

    // Now write out our object members
    strmToWriteTo << __strAuxText;
    strmToWriteTo << __strErrText;
    strmToWriteTo << __strModName;
    strmToWriteTo << __strFileName;
    strmToWriteTo << __strThread;
}
