//
// NAME: CIDLib_Environment.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_Environment.Cpp module, which implements
//  the TProcEnvironment and TEnvironment classes.
//
//  TProcEnvironment is all static and operates on this process' environment,
//  which all threads of this process share.
//
//  TProcEnvLocker is a lock janitor that allows you to lock the environment
//  when you need to atomically do multiple operations. Single methods of
//  TProcEnvironment are thread safe, but that won't help you if you have to
//  do multiple operations atomically.
//
//  TEnvironment is a means of creating copies of the environment and modifying
//  it without affecting others. This is usually done to create a new
//  environment to pass to an external process which is being spawned, via the
//  TExternalProcess class. TEnvironment objects have their own environment
//  variable data and only operate on it, not affecting the process environment.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 08/15/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//

// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
class   TEnvironCursor;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TProcEnvironment
//  PREFIX: env
// ----------------------------------------------------------------------------
class CIDLIBEXP TProcEnvironment : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TProcEnvironment();

        ~TProcEnvironment();


        // --------------------------------------------------------------------
        //  Public static methods
        // --------------------------------------------------------------------
        static tCIDLib::TVoid Add
        (
            const   TString&                strKey
            , const TString&                strNewValue
        );

        static tCIDLib::TBoolean bAddOrUpdate
        (
            const   TString&                strKey
            , const TString&                strNewValue
        );

        static tCIDLib::TBoolean bFind
        (
            const   TString&                strKey
            ,       TString&                strToFill
        );

        static tCIDLib::TCard4 c4Entries();

        static tCIDLib::TCard4 chCharsInValue
        (
            const   TString&                strKey
        );

        static TString strFind
        (
            const   TString&                strKey
        );

        static tCIDLib::TVoid Update
        (
            const   TString&                strKey
            , const TString&                strNewValue
        );


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TProcEnvironment(const TProcEnvironment&);

        tCIDLib::TVoid operator=(const TProcEnvironment&);

        // --------------------------------------------------------------------
        //  Do any magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TProcEnvironment,TObject)
};


// ----------------------------------------------------------------------------
//   CLASS: TProcEnvLocker
//  PREFIX: lock
// ----------------------------------------------------------------------------
class CIDLIBEXP TProcEnvLocker : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TProcEnvLocker();

        ~TProcEnvLocker();


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TProcEnvLocker(const TProcEnvLocker&);

        tCIDLib::TVoid operator=(const TProcEnvLocker&);


        // --------------------------------------------------------------------
        //  Magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TProcEnvLocker, TObject)
};


// ----------------------------------------------------------------------------
//   CLASS: TEnvironment
//  PREFIX: env
// ----------------------------------------------------------------------------
class CIDLIBEXP TEnvironment

    : public TObject, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Public nested types
        //
        //  TElem
        //      Our element type that we store our info in in the hash map
        //      that provides our storage.
        //
        //  TCursor
        //      A cursor to use when iterating the object. Its really just
        //      a typedef to our simple derivative of the hash map cursor.
        // --------------------------------------------------------------------
        class TElem : public TObject
        {
            public :
                // ------------------------------------------------------------
                //  Constructors and destructors
                // ------------------------------------------------------------
                TElem(const TString& strKey, const TString& strValue) :

                    __strKey(strKey)
                    , __strValue(strValue)
                {
                }

                TElem(const TElem& elemToCopy) :

                    __strKey(elemToCopy.__strKey)
                    , __strValue(elemToCopy.__strValue)
                {
                }

                ~TElem()
                {
                }

                tCIDLib::TVoid operator=(const TElem& elemToAssign)
                {
                    __strKey = elemToAssign.__strKey;
                    __strValue = elemToAssign.__strValue;
                }

                const TString& strKey() const
                {
                    return __strKey;
                }

                TString& strKey()
                {
                    return __strKey;
                }

                const TString& strValue() const
                {
                    return __strValue;
                }

                TString& strValue()
                {
                    return __strValue;
                }


            private :
                // ------------------------------------------------------------
                //  Private data
                //
                //  __strKey
                //  __strValue
                //      The key and value strings for this element
                // ------------------------------------------------------------
                TString     __strKey;
                TString     __strValue;

                RTTIMacros(TElem,TObject)
        };

        typedef TEnvironCursor  TCursor;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TEnvironment
        (
            const   tCIDLib::TBoolean       bInherit = kCIDLib::False
        );

        TEnvironment
        (
            const   TEnvironment&           envToCopy
        );

        ~TEnvironment();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TEnvironment& operator=(const TEnvironment& envToAssign);


        // --------------------------------------------------------------------
        //  Public non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add
        (
            const   TString&                strKey
            , const TString&                strNewValue
        );

        tCIDLib::TBoolean bAddOrUpdate
        (
            const   TString&                strKey
            , const TString&                strNewValue
        );

        tCIDLib::TBoolean bFind
        (
            const   TString&                strKey
            ,       TString&                strToFill
        );

        tCIDLib::TCard4 c4Entries();

        tCIDLib::TCard4 chCharsInValue
        (
            const   TString&                strKey
        );

        TString strFind
        (
            const   TString&                strKey
        );

        tCIDLib::TVoid Update
        (
            const   TString&                strKey
            , const TString&                strNewValue
        );

    protected :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TEnvironCursor;

    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __hshmEnv
        //      This is a hash map that is used to hold the environment
        //      variable data.
        // --------------------------------------------------------------------
        THashMap<TElem,TString>*  __phshmEnv;


        // --------------------------------------------------------------------
        //  Do any magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TEnvironment,TObject)
        DefPolyDup(TEnvironment)
};


// ----------------------------------------------------------------------------
//   CLASS: TEnvironCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
#pragma warning(disable : 4275)
class CIDLIBEXP TEnvironCursor

    : public THashMapCursor<TEnvironment::TElem, TString>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TEnvironCursor(TEnvironment* const penvToCursor) :

            THashMapCursor<TEnvironment::TElem, TString>(penvToCursor->__phshmEnv)
        {
        }

        TEnvironCursor(const TEnvironCursor& cursToCopy) :

            THashMapCursor<TEnvironment::TElem, TString>(cursToCopy)
        {
        }

        ~TEnvironCursor()
        {
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TEnvironCursor();

        tCIDLib::TVoid operator=(const TEnvironCursor&);

        tCIDLib::TBoolean operator==(const TEnvironCursor&);

        tCIDLib::TBoolean operator!=(const TEnvironCursor&);
};
#pragma warning(default : 4275)

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Constructors and destructors
// ----------------------------------------------------------------------------
inline TProcEnvironment::TProcEnvironment()
{
}

inline TProcEnvironment::~TProcEnvironment()
{
}
