//
// NAME: CIDLib_Environment.Cpp
//
// DESCRIPTION:
//
//  TProcEnvironment and TEnvironment are mechanism for manipulating the
//  process wide environment and copies of it.
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 08/15/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TProcEnvironment,TObject)
RTTIData(TProcEnvLocker,TObject)
RTTIData2(TEnvironment,TObject)
RTTIData(TEnvironment::TElem,TObject)


// -----------------------------------------------------------------------------
//  Local, const data
//
//  __c4Modulus
//      This is the modulus used in the hashing of the key strings.
// -----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4Modulus = 23;


// -----------------------------------------------------------------------------
//  Static functions
// -----------------------------------------------------------------------------

static const TString& __strGetKey(const TEnvironment::TElem& elemData)
{
    return elemData.strKey();
}


// ----------------------------------------------------------------------------
//   CLASS: TProcEnvironment
//  PREFIX: env
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TProcEnvironment: Public static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TProcEnvironment::Add(const TString& strKey, const TString& strNewValue)
{
    try
    {
        TKrnlEnvironment::Add(strKey.pszData(), strNewValue.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        tCIDLib::TErrCode errcToThrow = kCIDErrs::errcEnv_AddError;
        tCIDLib::EErrClasses eClass = tCIDLib::EClass_CantDo;
        if (kerrToCatch.errcId() == kKrnlErrors::errcAlreadyExists)
        {
            errcToThrow = kCIDErrs::errcEnv_VarAlreadyExists;
            eClass = tCIDLib::EClass_Already;
        }

        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , errcToThrow
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , eClass
            , strKey
        );
    }
}

tCIDLib::TBoolean
TProcEnvironment::bAddOrUpdate(const TString& strKey, const TString& strNewValue)
{
    tCIDLib::TBoolean bRes;
    try
    {
        bRes = TKrnlEnvironment::bAddOrUpdate
        (
            strKey.pszData()
            , strNewValue.pszData()
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnv_AddError
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strKey
        );
    }
    return bRes;
}

tCIDLib::TBoolean
TProcEnvironment::bFind(const TString& strKey, TString& strToFill)
{
    tCIDLib::TBoolean bRes;
    try
    {
        tCIDLib::Tch szTmp[2048];
        bRes = TKrnlEnvironment::bFind
        (
            strKey.pszData()
            , szTmp
            , c4MaxBufChars(szTmp)
        );

        strToFill = szTmp;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnv_FindError
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strKey
        );
    }
    return bRes;
}

tCIDLib::TCard4 TProcEnvironment::c4Entries()
{
    return TKrnlEnvironment::c4Entries();
}

tCIDLib::TCard4 TProcEnvironment::chCharsInValue(const TString& strKey)
{
    tCIDLib::TCard4 c4Chars;
    try
    {
        c4Chars = TKrnlEnvironment::chCharsInValue(strKey.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        tCIDLib::TErrCode errcToThrow = kCIDErrs::errcEnv_FindError;
        tCIDLib::EErrClasses eClass = tCIDLib::EClass_CantDo;
        if (kerrToCatch.errcId() == kKrnlErrors::errcNotFound)
        {
            errcToThrow = kCIDErrs::errcEnv_VarNotFound;
            eClass = tCIDLib::EClass_NotFound;
        }

        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , errcToThrow
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , eClass
            , strKey
        );
    }
    return c4Chars;
}

TString TProcEnvironment::strFind(const TString& strKey)
{
    tCIDLib::TBoolean   bRes;
    tCIDLib::Tch        szTmp[2048];
    try
    {
        bRes = TKrnlEnvironment::bFind
        (
            strKey.pszData()
            , szTmp
            , c4MaxBufChars(szTmp)
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnv_FindError
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strKey
        );
    }

    if (!bRes)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnv_VarNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_NotFound
            , strKey
        );
    }

    return TString(szTmp);
}

tCIDLib::TVoid
TProcEnvironment::Update(   const   TString&    strKey
                            , const TString&    strNewValue)
{
    try
    {
        TKrnlEnvironment::Update(strKey.pszData(), strNewValue.pszData());
    }

    catch(const TKrnlError& kerrToCatch)
    {
        tCIDLib::TErrCode errcToThrow = kCIDErrs::errcEnv_UpdateError;
        tCIDLib::EErrClasses eClass = tCIDLib::EClass_CantDo;
        if (kerrToCatch.errcId() == kKrnlErrors::errcNotFound)
        {
            errcToThrow = kCIDErrs::errcEnv_VarNotFound;
            eClass = tCIDLib::EClass_NotFound;
        }

        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , errcToThrow
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , eClass
            , strKey
        );
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TProcEnvLocker
//  PREFIX: lock
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TProcEnvLocker: Constructors and Destructors
// ----------------------------------------------------------------------------

TProcEnvLocker::TProcEnvLocker()
{
    TKrnlEnvironment::Lock();
}

TProcEnvLocker::~TProcEnvLocker()
{
    TKrnlEnvironment::Unlock();
}



// ----------------------------------------------------------------------------
//   CLASS: TEnvironment
//  PREFIX: env
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TEnvironment: Public static methods
// ----------------------------------------------------------------------------

TEnvironment::TEnvironment(const tCIDLib::TBoolean bInherit) :

    __phshmEnv(0)
{
    __phshmEnv = new THashMap<TElem, TString>
    (
        __c4Modulus
        , new TStringKeyOps<TString>
        , __strGetKey
        , kCIDLib::c4MaxCard
        , tCIDLib::EMTState_Unsafe
    );

    //
    //  If the bInherit parm is set, then we need to query a snapshot of
    //  the environment and load ourself up with the same info.
    //
    if (bInherit)
    {
        // Load up a list of environment element objects
        TKrnlEnvironment::TElem* pelemList;

        tCIDLib::TCard4 c4Count = 0;
        try
        {
            c4Count = TKrnlEnvironment::c4QueryState(pelemList);
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEnv_Inherit
                , kerrToCatch
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
            );
        }

        if (c4Count)
        {
            for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
            {
                __phshmEnv->Add
                (
                    TElem
                    (
                        pelemList[c4Index].pszKey
                        , pelemList[c4Index].pszValue
                    )
                );
            }

            // Delete the list
            delete [] pelemList;
        }
    }
}

TEnvironment::TEnvironment(const TEnvironment& envToCopy) :

    __phshmEnv(0)
{
    // Duplicate the hash map of the source
    __phshmEnv = pDupCollection(*envToCopy.__phshmEnv);
}

TEnvironment::~TEnvironment()
{
    delete __phshmEnv;
    __phshmEnv = 0;
}


// ----------------------------------------------------------------------------
//  TEnvironment: Public operators
// ----------------------------------------------------------------------------

TEnvironment& TEnvironment::operator=(const TEnvironment& envToAssign)
{
    if (this == &envToAssign)
        return *this;

    // Flush the collection and copy the elements of the source
    __phshmEnv->Flush();
    CopyCollectionNodes(*__phshmEnv, *envToAssign.__phshmEnv);

    return *this;
}


// ----------------------------------------------------------------------------
//  TEnvironment: Public non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TEnvironment::Add(const TString& strKey, const TString& strNewValue)
{
    try
    {
        __phshmEnv->Add(TElem(strKey, strNewValue));
    }

    catch(const TError& errToCatch)
    {
        // Translate a duplicate key into an environment error
        if (errToCatch.errcId() == kCIDErrs::errcCol_DuplicateKey)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEnv_VarAlreadyExists
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_Already
                , strKey
            );
        }

        // Wasn't expected, so just propogate it
        throw;
    }
}

tCIDLib::TBoolean
TEnvironment::bAddOrUpdate(const TString& strKey, const TString& strNewValue)
{
    return __phshmEnv->bAddOrUpdate(TElem(strKey, strNewValue));
}

tCIDLib::TBoolean TEnvironment::bFind(const TString& strKey, TString& strToFill)
{
    TElem& elemFind = __phshmEnv->objFindByKey(strKey);
    if (!(&elemFind))
        return kCIDLib::False;

    strToFill = elemFind.strValue();
    return kCIDLib::True;
}

tCIDLib::TCard4 TEnvironment::c4Entries()
{
    return __phshmEnv->c4ElemCount();
}

tCIDLib::TCard4 TEnvironment::chCharsInValue(const TString& strKey)
{
    TElem& elemFind = __phshmEnv->objFindByKey(strKey);
    if (!(&elemFind))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnv_VarNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_NotFound
            , strKey
        );
    }
    return elemFind.strValue().c4Length();
}

TString TEnvironment::strFind(const TString& strKey)
{
    TElem& elemFind = __phshmEnv->objFindByKey(strKey);
    if (!(&elemFind))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnv_VarNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_NotFound
            , strKey
        );
    }
    return elemFind.strValue();
}

tCIDLib::TVoid TEnvironment::Update(const   TString&    strKey
                                    , const TString&    strNewValue)
{
    TElem& elemFind = __phshmEnv->objFindByKey(strKey);
    if (!(&elemFind))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnv_VarNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_NotFound
            , strKey
        );
    }
    elemFind.strValue() = strNewValue;
}
