//
// NAME: CIDLib_Encrypt.Cpp
//
// DESCRIPTION:
//
//  This module implements the TEncrypter class. This is an abstract base
//  class interface that is used by the system to do all encryption. It
//  allows many encryption algorithms to be implemented and used without
//  change to the using code.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 12/02/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TEncrypter,TObject)


// ----------------------------------------------------------------------------
//   CLASS: TEncrypter
//  PREFIX: cryp
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TEncrypter: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TCard4
TEncrypter::c4Decrypt(  const   TMemBuf&            mbufCypher
                        ,       TMemBuf&            mbufPlain
                        , const tCIDLib::TCard4     c4CypherBytes)
{
    const tCIDLib::TCard4 c4SrcSize = mbufCypher.c4Size();
    const tCIDLib::TCard4 c4TargetMax = mbufPlain.c4MaxSize();

    //
    //  If the cypher byte parm is larger than the size of the cypher text
    //  memory buffer's size, that's an error.
    //
    if (c4CypherBytes > c4SrcSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCryp_BadCount
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TCardinal(c4CypherBytes)
            , TCardinal(c4SrcSize)
        );
    }

    //
    //  If the cypher byte parm is 0, then use the allocated size of the
    //  source buffer.
    //
    tCIDLib::TCard4 c4ActualSize = c4CypherBytes;
    if (!c4ActualSize)
        c4ActualSize = c4SrcSize;

    //
    //  Create a temp buffer that is large enough to hold the maximum
    //  size that the plain text buffer could hold. Put a janitor on it
    //  so it gets cleaned up.
    //
    tCIDLib::TCard1*    pac1Tmp = new tCIDLib::TCard1[c4TargetMax];
    THeapJanitor        janTmp(pac1Tmp);

    //
    //  And call the protected virtual method. We pass the raw cypher
    //  text buffer as the source, the raw temp buffer as the target,
    //  the actual number of bytes from the source to decrypt, and 
    //  a variable to fill in with the number of bytes put into the
    //  temp buffer.
    //
    tCIDLib::TCard4 c4Decrypted = _c4Decrypt
    (
        mbufCypher.pc1Data()
        , c4ActualSize
        , pac1Tmp
        , c4TargetMax
    );

    //
    //  Copy the decrypted number of bytes from the temp buffer to the
    //  plaintext memory buffer.
    //
    if (c4Decrypted)
        mbufPlain.CopyIn(pac1Tmp, c4Decrypted);

    return c4Decrypted;
}

tCIDLib::TCard4
TEncrypter::c4Decrypt(  const   TMemBuf&            mbufCypher
                        ,       TString&            strPlain
                        , const tCIDLib::TCard4     c4CypherBytes)
{
    //
    //  Get the number of allocated bytes in the source cypher buffer
    //  and the maximum bytes the target string could hold (which is the
    //  max char size times the bytes per char.
    //
    const tCIDLib::TCard4 c4SrcSize = mbufCypher.c4Size();
    const tCIDLib::TCard4 c4TargetMax = strPlain.c4MaxChars() * kCIDLib::c4CharBytes;

    //
    //  If the cypher byte parm is larger than the size of the cypher text
    //  memory buffer's size, that's an error.
    //
    if (c4CypherBytes > c4SrcSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCryp_BadCount
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TCardinal(c4CypherBytes)
            , TCardinal(c4SrcSize)
        );
    }

    //
    //  If the cypher byte parm is 0, then use the allocated size of the
    //  source buffer.
    //
    tCIDLib::TCard4 c4ActualSize = c4CypherBytes;
    if (!c4ActualSize)
        c4ActualSize = c4SrcSize;

    //
    //  Create a temp buffer that is large enough to hold the maximum
    //  size that the plain text string could hold. Put a janitor on it
    //  so it gets cleaned up. Add an extra char's worth at the end in
    //  order to tack on a nul terminator.
    //
    tCIDLib::TCard1* pac1Tmp =
                     new tCIDLib::TCard1[c4TargetMax + kCIDLib::c4CharBytes];
    THeapJanitor     janTmp(pac1Tmp);

    //
    //  And call the protected virtual method. We pass the raw cypher text
    //  buffer as the source, the raw temp buffer as the target, the actual
    //  number of bytes from the source to decrypt, and a variable to fill
    //  in with the number of bytes put into the temp buffer.
    //
    tCIDLib::TCard4 c4Decrypted = _c4Decrypt
    (
        mbufCypher.pc1Data()
        , c4ActualSize
        , pac1Tmp
        , c4TargetMax
    );

    //
    //  Copy the decrypted number of bytes from the temp buffer to the
    //  plaintext memory buffer.
    //
    if (c4Decrypted)
    {
        // Look at the temp buffer as a UNICode string
        tCIDLib::Tch* pszTmp = reinterpret_cast<tCIDLib::Tch*>(pac1Tmp);

        // Cap it off at the number of chars decrypted
        pszTmp[c4Decrypted / kCIDLib::c4CharBytes] = kCIDLib::chNull;

        // And put the temp text into the plain text string
        strPlain = pszTmp;
    }
     else
    {
        // No bytes were decryped, so just clear the plain text string
        strPlain.Clear();
    }
    return c4Decrypted;
}



tCIDLib::TCard4
TEncrypter::c4Encrypt(  const   TMemBuf&            mbufPlain
                        ,       TMemBuf&            mbufCypher
                        , const tCIDLib::TCard4     c4PlainBytes)
{
    const tCIDLib::TCard4 c4SrcSize = mbufPlain.c4Size();
    const tCIDLib::TCard4 c4TargetMax = mbufCypher.c4MaxSize();

    //
    //  If the plain bytes parm is larger than the size of the plain text
    //  memory buffer's size, that's an error.
    //
    if (c4PlainBytes > c4SrcSize)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCryp_BadCount
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TCardinal(c4PlainBytes)
            , TCardinal(c4SrcSize)
        );
    }

    //
    //  If the cypher byte parm is 0, then use the allocated size of the
    //  source buffer.
    //
    tCIDLib::TCard4 c4ActualSize = c4PlainBytes;
    if (!c4ActualSize)
        c4ActualSize = c4SrcSize;

    //
    //  Create a temp buffer that is large enough to hold the maximum
    //  size that the cypher text buffer could hold. Put a janitor on it
    //  so it gets cleaned up.
    //
    tCIDLib::TCard1*    pac1Tmp = new tCIDLib::TCard1[c4TargetMax];
    THeapJanitor        janTmp(pac1Tmp);

    //
    //  And call the protected virtual method. We pass the raw plain
    //  text buffer as the source, the raw temp buffer as the target,
    //  the actual number of bytes from the source to encrypt, and 
    //  a variable to fill in with the number of bytes put into the
    //  temp buffer.
    //
    tCIDLib::TCard4 c4Encrypted = _c4Encrypt
    (
        mbufPlain.pc1Data()
        , c4ActualSize
        , pac1Tmp
        , c4TargetMax
    );

    //
    //  Copy the encrypted number of bytes from the temp buffer to the
    //  cypher text memory buffer.
    //
    if (c4Encrypted)
        mbufCypher.CopyIn(pac1Tmp, c4Encrypted);

    return c4Encrypted;
}

tCIDLib::TCard4
TEncrypter::c4Encrypt(const TString& strPlain, TMemBuf& mbufCypher)
{
    // Watch for an empty source string
    if (strPlain.bEmpty())
        return 0;

    //
    //  Get the number of bytes that are available in the plain text
    //  string, and the maximum number of bytes the target buffer can
    //  hold.
    //
    const tCIDLib::TCard4 c4SrcSize = strPlain.c4Length() * kCIDLib::c4CharBytes;
    const tCIDLib::TCard4 c4TargetMax = mbufCypher.c4MaxSize();

    //
    //  Create a temp buffer that is large enough to hold the maximum
    //  size that the cypher text buffer could hold. Put a janitor on it
    //  so it gets cleaned up.
    //
    tCIDLib::TCard1* pac1Tmp = new tCIDLib::TCard1[c4TargetMax];
    THeapJanitor     janTmp(pac1Tmp);

    //
    //  And call the protected virtual method. We pass the raw plain text
    //  string buffer as the source, the raw temp buffer as the target, the
    //  number of bytes from the source to encrypt, and a variable
    //  to fill in with the number of bytes put into the temp buffer.
    //
    tCIDLib::TCard4 c4Encrypted = _c4Encrypt
    (
        reinterpret_cast<const tCIDLib::TCard1*>(strPlain.pszData())
        , c4SrcSize
        , pac1Tmp
        , c4TargetMax
    );

    // Copy the temp buffer data into the cypher text buffer
    if (c4Encrypted)
        mbufCypher.CopyIn(pac1Tmp, c4Encrypted);

    return c4Encrypted;
}
