//
// NAME: CIDLib_DirIter.Cpp
//
// DESCRIPTION:
//
//  This module implements the TDirIter class, which allows for iteration of the
//  files in a directory.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/13/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"

// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TDirIter,TObject)



// -----------------------------------------------------------------------------
//   CLASS: TDirIter
//  PREFIX: diter
//
//  This class provides directory search services
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TDirIter: Constructors/Destructors
// -----------------------------------------------------------------------------

TDirIter::TDirIter()
{
}

TDirIter::~TDirIter()
{
}


// -----------------------------------------------------------------------------
//  TDirIter: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TDirIter::bFindNext(TFindBuf& fndBuf)
{
    tCIDLib::TCard4             c4Count;
    TKrnlFileSys::TRawFileFind  FindBuf;

    // If there is no open search, then log an error and abort
    if (!__kdsSearch.bSearchIsOpen())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcFile_NoOpenSearch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
        );
    }

    // Try to find another match
    c4Count = 1;
    try
    {
        if (!__kdsSearch.bFindNext(FindBuf))
            return kCIDLib::False;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcDIter_FindFailure
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Unknown
            , __pathSearchSpec
        );
    }

    // We got a match, so copy the needed stuff to this object
    fndBuf._FromFindBuf(FindBuf, __pathSearchSpec);
    return kCIDLib::True;
}


tCIDLib::TBoolean
TDirIter::bFindFirst(   const   TString&            strSpec
                        ,       TFindBuf&           fndBuf
                        , const tCIDLib::EFileAttrs eCanBe
                        , const tCIDLib::EFileAttrs eMustBe)
{
    tCIDLib::TCard4             c4Count;
    TKrnlFileSys::TRawFileFind  FindBuf;

    // Store away the spec and attributes
    __pathSearchSpec    = strSpec;

    // Strip the name and extension off of the search spec
    __pathSearchSpec.bRemoveNameExt();

    // Try to start the search
    c4Count = 1;
    try
    {
        if (!__kdsSearch.bFindFirst
        (
            strSpec.pszData()
            , eCanBe
            , eMustBe
            , FindBuf))
        {
            return kCIDLib::False;
        }
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcDIter_FindFailure
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Unknown
            , strSpec
        );
    }

    // Copy the needed stuff back
    fndBuf._FromFindBuf(FindBuf, __pathSearchSpec);

    return kCIDLib::True;
}

tCIDLib::TBoolean
TDirIter::bFindFirst(   const   TString&            strPath
                        , const TString&            strWildCard
                        ,       TFindBuf&           fndBuf
                        , const tCIDLib::EFileAttrs eCanBe
                        , const tCIDLib::EFileAttrs eMustBe)
{
    //
    //  Just build a full spec out of the passed path and wildcards, then
    //  call the other version
    //
    TPathStr pathTmp(strPath);

    // Add the wild cards to the path in the temp string
    pathTmp.AddLevel(strWildCard);

    // Pass along our data and get the respose
    return bFindFirst(pathTmp, fndBuf, eCanBe, eMustBe);
}


tCIDLib::TVoid TDirIter::Reset()
{
    __kdsSearch.Close();
}
