//
// NAME: CIDLib_Deque.Hpp
//
// DESCRIPTION:
//
//  This is the header for the deque template collection. It implements the
//  TDeque class template, which provides LIFO and FIFO storage of its elements
//  by value.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 02/06/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// ----------------------------------------------------------------------------
//  Forward reference the deque cursor class
// ----------------------------------------------------------------------------
template <class TElem> class TDequeCursor;
template <class TElem> class TDeque;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TDequeNode
//  PREFIX: node
// ----------------------------------------------------------------------------
template <class TElem> class TDequeNode : public TDLstNode
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TDequeNode(const TElem& objData) :

            __objData(objData)
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TDequeNodeOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        ~TDequeNode() {}


    protected   :
        // --------------------------------------------------------------------
        //  The deque and its cursor are our friends
        // --------------------------------------------------------------------
        friend class TDeque<TElem>;
        friend class TDequeCursor<TElem>;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TDequeNode();

        TDequeNode(const TDequeNode<TElem>&);

        tCIDLib::TVoid operator=(const TDequeNode<TElem>&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __objData
        //      The user data object.
        // --------------------------------------------------------------------
        TElem   __objData;


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean    __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TDequeNode<TElem>,TDLstNode)
};

template <class TElem> tCIDLib::TBoolean TDequeNode<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TDequeNode<TElem>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TDeque
//  PREFIX: deq
// ----------------------------------------------------------------------------
template <class TElem> class TDeque : public TCollection<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Nested typedefs for the cursor and node types used by a deque.
        // --------------------------------------------------------------------
        typedef class TDequeCursor<TElem>   TCursor;
        typedef TDequeNode<TElem>           TNode;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TDeque(const tCIDLib::EMTStates eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __llstDeque(0)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        TDeque( const   tCIDLib::TCard4     c4MaxElems
                , const tCIDLib::EMTStates  eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __llstDeque(c4MaxElems)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        ~TDeque() {}


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add(const TElem& objNew)
        {
            TLocker LockDeque(this);
            PushBottom(objNew);
            _c4IncSerialNum();
        }

        tCIDLib::TBoolean bFlushCur()
        {
            TLocker LockDeque(this);

            // Make sure the iterator is valid
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Save the next node
            TNode* const pnodeNext = (TNode*)__pnodeCur->pnodeNext();

            // Now flush the iterator node
            __llstDeque.FlushNode(__pnodeCur);

            // Bump serial number to invalidate cursors
            _c4IncSerialNum();

            // Put the next node back into the iterator
            __pnodeCur = pnodeNext;

            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bIsEmpty() const
        {
            TLocker LockDeque(this);
            tCIDLib::TBoolean bRet = __llstDeque.bIsEmpty();
            return bRet;
        }

        tCIDLib::TBoolean bNext()
        {
            TLocker LockDeque(this);
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            return (__pnodeCur != 0);
        }
        
        tCIDLib::TBoolean bResetIter()
        {
            TLocker LockDeque(this);
            __pnodeCur = (TNode*)__llstDeque.pnodeHead();
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            TLocker LockDeque(this);
            tCIDLib::TCard4 c4Ret = __llstDeque.c4ElemCount();
            return c4Ret;
        }

        tCIDLib::TCard4 c4MaxElemCount() const
        {
            TLocker LockDeque(this);
            tCIDLib::TCard4 c4Ret = __llstDeque.c4MaxElemCount();
            return c4Ret;
        }

        tCIDLib::TVoid Flush()
        {
            TLocker LockDeque(this);
            __pnodeCur = 0;
            __llstDeque.Flush();
            _c4IncSerialNum();
        }

        TElem& objCur() const
        {
            TLocker LockDeque(this);
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }

        TElemCursor<TElem>* pcursNew() const
        {
            TLocker LockDeque(this);
            return new TCursor(this);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid FlushAt(TCursor& cursAt)
        {
            TLocker LockDeque(this);

            // Make sure the cursor belongs to this deque
            if (!cursAt.bIsCursoring(*this))
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NotMyCursor
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , cursAt.clsIsA()
                    , clsIsA()
                );
            }

            // Get the node to flush
            TNode* pnodeToFlush = cursAt.pnodeCur();

            // Make sure the cursor's valid
            if (!pnodeToFlush)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , cursAt.clsIsA()
                );
            }

            // Bump up the cursor
            cursAt.bNext();

            // If the flush node is the current iterator node, move it up
            if (pnodeToFlush == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Now flush the node
            __llstDeque.FlushNode(pnodeToFlush);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // But we kept the passed cursor up to date by bumping it up
            cursAt._c4SerialNum(c4SerialNum());
        }

        TElem objPeekBottom() const
        {
            TLocker LockDeque(this);

            // See if there are any nodes. If not, throw an exception
            if (!__llstDeque.c4ElemCount())
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Get a pointer to the first node
            TNode* pnodeFirst = (TNode*)__llstDeque.pnodeHead();

            // Get a copy of the stored object
            return pnodeFirst->__objData;
        }

        TElem objPeekTop() const
        {
            TLocker LockDeque(this);

            // See if there are any nodes. If not, throw an exception
            if (!__llstDeque.c4ElemCount())
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Get a pointer to the last node
            TNode* pnodeLast = (TNode*)__llstDeque.pnodeTail();

            // Return a copy of the stored object
            return pnodeLast->__objData;
         }

        TElem objPopBottom()
        {
            TLocker LockDeque(this);

            // See if there are any nodes. If not, throw an exception
            if (!__llstDeque.c4ElemCount())
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Get a pointer to the first node
            TNode* pnodeHead = (TNode*)__llstDeque.pnodeHead();

            // If the popped node is the current iterator node, move it up
            if (pnodeHead == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Get a copy of the stored object
            TElem objRet = pnodeHead->__objData;

            // Delete the node.
            __llstDeque.FlushNode(pnodeHead);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            return objRet;
        }

        TElem objPopTop()
        {
            TLocker LockDeque(this);

            // See if there are any nodes. If not, throw an exception
            if (!__llstDeque.c4ElemCount())
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Get a pointer to the last node
            TNode* pnodeLast = (TNode*)__llstDeque.pnodeTail();

            // If the flush node is the current iterator node, move it up
            if (pnodeLast == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Get a copy of the stored object
            TElem objRet = pnodeLast->__objData;

            // Delete the node.
            __llstDeque.FlushNode(pnodeLast);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            return objRet;
        }

        tCIDLib::TVoid PushBottom(const TElem& objSrc)
        {
            TLocker LockDeque(this);
            __llstDeque.PrependNode(new TNode(objSrc));
            _c4IncSerialNum();
        }

        tCIDLib::TVoid PushTop(const TElem& objSrc)
        {
            TLocker LockDeque(this);
            __llstDeque.AppendNode(new TNode(objSrc));
            _c4IncSerialNum();
        }


    protected       :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TDequeCursor<TElem>;

        friend tCIDLib::TBoolean bCompCollections
        (
            const   TDeque<TElem>&          col1
            , const TDeque<TElem>&          col2    
            , const TObjEq<TElem>&          compToUse
        );

        friend TDeque<TElem>* pDupCollection
        (
            const   TDeque<TElem>&          colToDup
        );


        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TDeque(const TDeque<TElem>& colToCopy) :

            TCollection<TElem>(colToCopy)
            , __llstDeque(colToCopy.c4MaxElemCount())
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        const TDLinkedList& _llstDeque() const
        {
            return __llstDeque;
        }

        TDLinkedList& _llstDeque()
        {
            return __llstDeque;
        }


    private     :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TDeque<TElem>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TDequeOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __llstDeque
        //      This is a doubly linked list that is the storage for the deque
        //      items. Deques are bidirectional so we used a doubly linked
        //      list.
        //
        //  __pnodeCur
        //      The current node of the internal iterator.
        // --------------------------------------------------------------------
        TDLinkedList            __llstDeque;
        TNode*                  __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a deque of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TDeque<TElem>,TCollection<TElem>)
};

template <class TElem> tCIDLib::TBoolean TDeque<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TDeque<TElem>::clsThis;




// ----------------------------------------------------------------------------
//   CLASS: TDequeCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class TDequeCursor : public TBiElemCursor<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TDequeCursor(const TDeque<TElem>* const pcolToCursor) :

            TBiElemCursor<TElem>(pcolToCursor->c4SerialNum())
            , __pllstCursoring(&pcolToCursor->_llstDeque())
            , __pcolCursoring(pcolToCursor)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();

            __pnodeCur = (TDequeNode<TElem>*)__pllstCursoring->pnodeHead();
        }

        TDequeCursor(const TDequeCursor<TElem>& cursToCopy) :

            TBiElemCursor<TElem>(cursToCopy)
            , __pllstCursoring(cursToCopy.__pllstCursoring)
            , __pcolCursoring(cursToCopy.__pcolCursoring)
            , __pnodeCur(cursToCopy.__pnodeCur)
        {
            __CheckFirstOfType();
        }

        ~TDequeCursor() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TDequeCursor<TElem>& operator=(const TDequeCursor<TElem>& cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            TParent::operator=(cursToAssign);
            __pllstCursoring = cursToAssign.__pllstCursoring;
            __pcolCursoring = cursToAssign.__pcolCursoring;
            __pnodeCur = cursToAssign.__pnodeCur;

            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsEmpty() const
        {
            return __pllstCursoring->bIsEmpty();
        }

        tCIDLib::TBoolean bIsValid() const
        {
            // If the serial number is off, its not valid
            if (c4SerialNum() != __pcolCursoring->c4SerialNum())
                return kCIDLib::False;
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bNext()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TDequeNode<TElem>*)__pnodeCur->pnodeNext();
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bPrevious()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TDequeNode<TElem>*)__pnodeCur->pnodePrev();
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bReset()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            // Get the head node of the list
            __pnodeCur = (TDequeNode<TElem>*)__pllstCursoring->pnodeHead();

            // Get our serial back in sync with the collection
            _c4SerialNum(__pcolCursoring->c4SerialNum());

            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            return __pllstCursoring->c4ElemCount();
        }

        const TElem& objCur() const
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsCursoring(const TDeque<TElem>& colToCheck) const
        {
            return (__pcolCursoring == &colToCheck);
        }

        const TDequeNode<TElem>* pnodeCur() const
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }

        TDequeNode<TElem>* pnodeCur()
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TDeque<TElem>;


    private :
        // --------------------------------------------------------------------
        //  Unimplmented constructors and operators
        // --------------------------------------------------------------------
        TDequeCursor();


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TDequeCursorOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pcolCursoring
        //      This is the deque collection that we are cursoring.
        //
        //  __pllstCursoring
        //      This is the linked list we are cursoring. It is extracted from
        //      the deque during construction, for fast access.
        //
        //  __pnodeCur
        //      This is the current node that we are on in our iteration. It
        //      is 0 if we are at the end.
        // --------------------------------------------------------------------
        const TDeque<TElem>*    __pcolCursoring;
        const TDLinkedList*     __pllstCursoring;
        TDequeNode<TElem>*      __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a stack cursor of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TDequeCursor<TElem>,TBiElemCursor<TElem>)
};

template <class TElem> tCIDLib::TBoolean TDequeCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TDequeCursor<TElem>::clsThis;

#pragma pack(pop)



// ----------------------------------------------------------------------------
//  Some template functions to make duplication and copying of deques
//  typesafe
// ----------------------------------------------------------------------------
template <class TElem> tCIDLib::TBoolean
bCompCollections(const  TDeque<TElem>&  col1
                , const TDeque<TElem>&  col2
                , const TObjEq<TElem>&  compToUse)
{
    // Check for comparison against self
    if (&col1 == &col2)
        return kCIDLib::True;

    // Lock both of the bags
    TLocker lockDeque1(&col1);
    TLocker lockDeque2(&col2);

    //
    //  First see if the linked lists behind them have equal element counts,
    //  max element counts, and default delete mode. Note that this does
    //  compare the elements because the linked knows nothing about how
    //  to compare them.
    //
    if (col1.__llstDeque != col2.__llstDeque)
        return kCIDLib::False;

    // If the element count of either is 0, then they are both zero so equal
    if (!col1.c4ElemCount())
        return kCIDLib::True;

    //
    //  Ok so we have to compare the elements. So we just create cursors
    //  for both and compare each element. We know that they have the
    //  same number of elements and are locked so this safe.
    //
    TDeque<TElem>::TCursor curs1(&col1);
    TDeque<TElem>::TCursor curs2(&col2);
    do
    {
        if (!compToUse.bEqual(curs1.objCur(), curs2.objCur()))
            return kCIDLib::False;
    }   while (curs1.bNext() && curs2.bNext());
    return kCIDLib::True;
}

template <class TElem> tCIDLib::TVoid
CopyCollectionNodes(TDeque<TElem>& colDest, const TDeque<TElem>& colSource)
{
    // Check for copy to self
    if (&colDest == &colSource)
        return;

    // Lock both collections
    TLocker lockNew(&colDest);
    TLocker lockToDup(&colSource);

    if (colSource.c4ElemCount())
    {
        TDeque<TElem>::TCursor  cursSource(&colSource);
        do
        {
            colDest.PushTop(cursSource.objCur());
        }   while (cursSource.bNext());
    }
}

template <class TElem> TDeque<TElem>*
pDupCollection(const TDeque<TElem>& colToDup)
{
    // Lock the source collection
    TLocker lockToDup(&colToDup);

    // Create a new deque with the same max elem count
    TDeque<TElem>* const pcolNew = new TDeque<TElem>(colToDup);
    TJanitor<TDeque<TElem> > janCol(pcolNew);

    // If there are any elements in the source, copy them
    if (colToDup.c4ElemCount())
    {
        TDeque<TElem>::TCursor  cursDup(&colToDup);
        do
        {
            pcolNew->PushTop(cursDup.objCur());
        }   while (cursDup.bNext());
    }

    // Return a pointer to the new deque
    janCol.Orphan();
    return pcolNew;
}
