//
// NAME: CIDLib_LinkedList.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_LinkedList.Cpp module. This module
//  implements the concrete doubly linked list collection type. The linked
//  list is defined in terms of an abstract node class, from which a template
//  class will later be defined. That template class will be instantiated
//  many times for particular styles of collections built upon this
//  basic linked list functionality.
//
//  Note that this concrete linked list class is not derived from the
//  base collection classes itself. The template classes that use it are
//  derived from the collection classes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/25/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TDLstNode
//  PREFIX: node
// ----------------------------------------------------------------------------
class CIDLIBEXP TDLstNode : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        ~TDLstNode();


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        TDLstNode* pnodeNext() const;

        TDLstNode* pnodePrev() const;


    protected   :
        // --------------------------------------------------------------------
        //  Our container class is our friend. Can you say friend? Sure!
        //  That's real nice.
        // --------------------------------------------------------------------
        friend class TDLinkedList;


        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TDLstNode();


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TDLstNode(const TDLstNode&);

        tCIDLib::TVoid operator=(const TDLstNode&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pnodeNext
        //  __pnodePrev
        //      The next and previous nodes of this node
        // --------------------------------------------------------------------
        TDLstNode*      __pnodeNext;
        TDLstNode*      __pnodePrev;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TDLstNode,TObject)
};


// ----------------------------------------------------------------------------
//   CLASS: TDLinkedList
//  PREFIX: llst
// ----------------------------------------------------------------------------
class CIDLIBEXP TDLinkedList : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors/Destructors
        // --------------------------------------------------------------------
        TDLinkedList();

        TDLinkedList
        (
            const   tCIDLib::TCard4         c4MaxElems
        );

        ~TDLinkedList();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TDLinkedList&            llstToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TDLinkedList&            llstToTest
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid AppendNode
        (
                    TDLstNode* const        pnodeNew
        );

        tCIDLib::TBoolean bIsEmpty() const;

        tCIDLib::TBoolean bIsFirstNode
        (
                    TDLstNode* const        pnodeTest
        )   const;

        tCIDLib::TBoolean bIsLastNode
        (
                    TDLstNode* const        pnodeTest
        )   const;

        tCIDLib::TCard4 c4ElemCount() const;

        tCIDLib::TCard4 c4MaxElemCount() const;

        tCIDLib::TCard4 c4MaxElemCount
        (
            const   tCIDLib::TCard4         c4NewMax
        );

        tCIDLib::TVoid ExchangeNodes
        (
                    TDLstNode*              pnode1
            ,       TDLstNode*              pnode2
        );

        tCIDLib::TVoid Flush();

        tCIDLib::TVoid FlushNode
        (
                    TDLstNode* const        pnodeToFlush
        );

        tCIDLib::TVoid InsertNode
        (
                    TDLstNode* const        pnodeAfter
            ,       TDLstNode* const        pnodeNew
        );

        tCIDLib::TVoid MoveToHead
        (
                    TDLstNode* const        pnodeToMove
        );

        tCIDLib::TVoid MoveToTail
        (
                    TDLstNode* const        pnodeToMove
        );

        TDLstNode* pnodeHead() const;

        TDLstNode* pnodeTail() const;

        tCIDLib::TVoid PrependNode
        (
                    TDLstNode* const        pnodeNew
        );


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and destructors
        // --------------------------------------------------------------------
        TDLinkedList(const TDLinkedList&);

        TDLinkedList& operator=(const TDLinkedList&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4ElemCount
        //      The number of objects currently in the container. It is
        //      maintained as elems are added or removed.
        //
        //  __c4MaxElemCount
        //      The maximum number objects this array can hold. When the elem
        //      count exceeds this limit, a runtime error is generated.
        //
        //  __pnodeHead
        //      This is a pointer to the 1st node of the linked list. If it
        //      is 0, then there are no elements in the list.
        //
        //  __pnodTail
        //      This is the pointer to the last node of the linked list. If it
        //      is 0, then there are no elements in the list.
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4ElemCount;
        tCIDLib::TCard4         __c4MaxElemCount;
        TDLstNode*              __pnodeHead;
        TDLstNode*              __pnodeTail;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TDLinkedList,TObject)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  TDLstNode: Constructors and destructors
// ----------------------------------------------------------------------------
inline TDLstNode::TDLstNode() :

    __pnodeNext(0)
    , __pnodePrev(0)
{
}

inline TDLstNode::~TDLstNode()
{
}


// ----------------------------------------------------------------------------
//  TDLstNode: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline TDLstNode* TDLstNode::pnodeNext() const
{
    return __pnodeNext;
}

inline TDLstNode* TDLstNode::pnodePrev() const
{
    return __pnodePrev;
}


// ----------------------------------------------------------------------------
//  TDLinkedList: Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TDLinkedList::operator!=(const TDLinkedList& llstToTest) const
{
    return !operator==(llstToTest);
}


// ----------------------------------------------------------------------------
//  TDLinkedList: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TDLinkedList::bIsEmpty() const
{
    if (__c4ElemCount)
        return kCIDLib::False;
    return kCIDLib::True;
}

inline tCIDLib::TBoolean
TDLinkedList::bIsFirstNode(TDLstNode* const pnodeTest) const
{
    if (pnodeTest == __pnodeHead)
        return kCIDLib::False;
    else
        return kCIDLib::True;
}

inline tCIDLib::TBoolean
TDLinkedList::bIsLastNode(TDLstNode* const pnodeTest) const
{
    if (pnodeTest == __pnodeTail)
        return kCIDLib::False;
    return kCIDLib::True;
}

inline tCIDLib::TCard4 TDLinkedList::c4ElemCount() const
{
    return __c4ElemCount;
}

inline tCIDLib::TCard4 TDLinkedList::c4MaxElemCount() const
{
    return __c4MaxElemCount;
}

inline TDLstNode* TDLinkedList::pnodeHead() const
{
    return __pnodeHead;
}

inline TDLstNode* TDLinkedList::pnodeTail() const
{
    return __pnodeTail;
}
