//
// NAME: CIDLib_DLinkedList.Cpp
//
// DESCRIPTION:
//
//  This module implements the concrete collection class, TDLinkedList, and its
//  abstract linked list node class. The linked list class is defined in terms
//  of the abstract node class, from which many type specific versions are
//  instantiated via a template. The derived types determine the type of data
//  is in a node and what capabilities that the data must provide.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/02/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TDLstNode,TObject)
RTTIData(TDLinkedList,TObject)



// ----------------------------------------------------------------------------
//   CLASS: TDLinkedList
//  PREFIX: llst
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TDLinkedList: Constructors and Destructors
// ----------------------------------------------------------------------------

TDLinkedList::TDLinkedList(const tCIDLib::TCard4 c4MaxElems) :

    __c4ElemCount(0)
    , __c4MaxElemCount(c4MaxElems)
    , __pnodeHead(0)
    , __pnodeTail(0)
{
    // If the max items is 0, then set it to the max
    if (!__c4MaxElemCount)
        __c4MaxElemCount = kCIDLib::c4MaxCard;
}

TDLinkedList::TDLinkedList() :

    __c4ElemCount(0)
    , __c4MaxElemCount(kCIDLib::c4MaxCard)
    , __pnodeHead(0)
    , __pnodeTail(0)
{
}

TDLinkedList::~TDLinkedList()
{
    Flush();
}


// ----------------------------------------------------------------------------
//  TDLinkedList: Public operators
// ----------------------------------------------------------------------------

tCIDLib::TBoolean
TDLinkedList::operator==(const TDLinkedList& llstToTest) const
{
    if (__c4MaxElemCount != llstToTest.__c4MaxElemCount)
        return kCIDLib::False;

    if (__c4ElemCount != llstToTest.__c4ElemCount)
        return kCIDLib::False;

    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TDLinkedList: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TDLinkedList::AppendNode(TDLstNode* const pnodeNew)
{
    // Make sure that we are not overflowing here
    if (__c4ElemCount >= __c4MaxElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_TooManyNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }

    // Bump up the count of items
    __c4ElemCount++;

    //
    //  Check for special case of an empty list with no entries yet. If so,
    //  then just store away the new node.
    //
    if (!__pnodeHead)
    {
        // No current nodes, so point the head and tail at this node
        __pnodeHead  = pnodeNew;
        __pnodeTail  = pnodeNew;
    }
     else
    {
        // Point the current tail's next at the new node
        __pnodeTail->__pnodeNext = pnodeNew;

        // Set hte new node's prev to the current tail and 0 its next
        pnodeNew->__pnodePrev = __pnodeTail;
        pnodeNew->__pnodeNext = 0;

        // Now store it as the new tail
        __pnodeTail = pnodeNew;
    }
}


tCIDLib::TCard4
TDLinkedList::c4MaxElemCount(const tCIDLib::TCard4 c4NewMax)
{
    if (c4NewMax < __c4ElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NotEnoughNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4NewMax)
            , TCardinal(__c4ElemCount)
        );
    }

    __c4MaxElemCount = c4NewMax;
    return c4NewMax;
}


tCIDLib::TVoid
TDLinkedList::ExchangeNodes(TDLstNode* pnode1, TDLstNode* pnode2)
{
    tCIDLib::TCard4 c4Ind;

    // Handle freakout scenario of the 2 nodes being the same
    if (pnode1 == pnode2)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_SameNode
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_BadParms
        );
        return;
    }

    //
    //  If there are not at least 2 nodes, then an internal fatal error. The
    //  client app must be using at least one invalid node.
    //
    if (__c4ElemCount < 2)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NotEnoughNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }

    // If the head pointer is nul, then an internal error
    #if CID_DEBUG_ON
    if (!__pnodeHead)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NullHead
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
        );
    }
    #endif

    //
    //  We need to make sure that node 1 is before node 2, otherwise we need
    //  to flip them.
    //
    TDLstNode* pnodeTmp = __pnodeHead;
    for (c4Ind = 0; c4Ind < __c4ElemCount; c4Ind++)
    {
        //
        //  If we find the 1st node first, then just break out. If we find the
        //  second node first, then flip them.
        //
        if (pnodeTmp == pnode1)
        {
            break;
        }
         else if (pnodeTmp == pnode2)
        {
            pnodeTmp = pnode1;
            pnode1   = pnode2;
            pnode2   = pnodeTmp;
            break;
        }

        pnodeTmp = pnodeTmp->__pnodeNext;
    }

    // If we got here, then a node is missing
    if (c4Ind == __c4ElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NodeNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
        );
    }

    //
    //  Get local copies of the next/prev pointers of each node so that we
    //  won't have to worry about changing the original and still needing
    //  the old value.
    //
    TDLstNode* pnode1Prev  = pnode1->__pnodePrev;
    TDLstNode* pnode2Prev  = pnode2->__pnodePrev;
    TDLstNode* pnode1Next  = pnode1->__pnodeNext;
    TDLstNode* pnode2Next  = pnode2->__pnodeNext;

    //
    //  Handle the next node pointer of the node before pnode1. It should now
    //  point to pnode2, unless pnode1 is the 1st node.
    //
    if (pnode1Prev)
       pnode1->__pnodePrev->__pnodeNext = pnode2;

    //
    //  Handle the previous node pointer of the node after pnode2. It should
    //  now point to pnode1, unless pnode2 is the last node
    //
    if (pnode2Next)
       pnode2->__pnodeNext->__pnodePrev = pnode1;

    //
    //  Handle the previous node pointer of pnode2. It should point to the
    //  previous node of pnode1. If this guy is nul, then pnode2 is now the
    //  1st node, so point the list head at him.
    //
    pnode2->__pnodePrev = pnode1Prev;

    if (!pnode2->__pnodePrev)
        __pnodeHead = pnode2;

    //
    //  Handle the next node pointer of pnode1 and make it point to pnode2's
    //  next node, which could be nul if pnode2 was last. This would now make
    //  pnode1 last.
    //
    pnode1->__pnodeNext = pnode2Next;

    //
    //  Handle the next node pointer to pnode2 and make it point to pnode1's
    //  next node.
    //
    //  We also have a potential special case here if the nodes being
    //  exchanged are side by side, in which case pnode2's next node should
    //  point at pnode1. Otherwise pnode2 would end up pointing at itself.
    //
    if (pnode1Next == pnode2)
    {
        pnode2->__pnodeNext = pnode1;
    }
     else
    {
        pnode2->__pnodeNext = pnode1Next;
        pnode2Prev->__pnodeNext = pnode1;
    }

    //
    //  The same problem exists for the previous node of pnode1. If they are
    //  side by side, then just point it at pnode1; else, point it at the
    //  previous node of pnode2.
    //
    if (pnode2Prev == pnode1)
    {
       pnode1->__pnodePrev = pnode2;
    }
     else
    {
       pnode1->__pnodePrev = pnode2Prev;
       pnode1Next->__pnodePrev = pnode2;
    }
}


tCIDLib::TVoid
TDLinkedList::InsertNode(   TDLstNode* const    pnodeAfter
                            , TDLstNode* const  pnodeNew)
{
    //
    //  If the after node is 0, then that means insert it at the head,
    //  so we just turn it into a call to PrependNode().
    //
    if (!pnodeAfter)
    {
        PrependNode(pnodeNew);
        return;
    }

    // <TBD> Insure that new node is not already in the list!

    //
    //  If there are no current nodes, then an error because pnodeAfter
    //  cannot exist.
    //
    if (!__c4ElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NodeNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    // Make sure that we are not overflowing here
    if (__c4ElemCount >= __c4MaxElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_TooManyNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_OutResource
        );
    }

    // Get a pointer to the after node's next node
    TDLstNode* pnodeNext = pnodeAfter->__pnodeNext;

    //
    //  Insert the node into the list. Handle the special case of the
    //  insert after node being the last node.
    //
    if (__pnodeTail == pnodeAfter)
    {
        // Point the list tail at the new node
        __pnodeTail = pnodeNew;

        // Point new node's prev node at pnodeAFter and link forward
        pnodeNew->__pnodePrev = pnodeAfter;
        pnodeAfter->__pnodeNext = pnodeNew;
    }
     else
    {
       // No special case, just do the normal patch around
       pnodeAfter->__pnodeNext = pnodeNew;
       pnodeNew->__pnodePrev = pnodeAfter;
       pnodeNew->__pnodeNext = pnodeNext;
       pnodeNext->__pnodePrev = pnodeNew;
    }

    // Bump up the count of items
    __c4ElemCount++;
}


tCIDLib::TVoid TDLinkedList::MoveToHead(TDLstNode* const pnodeSrc)
{
    // <TBD> Insure its a member node if in debug build!

    // If there are not at least 2 nodes, then nothing to do
    if (__c4ElemCount < 2)
        return;

    // If this node is already the first, then nothing to do
    if (!pnodeSrc->__pnodePrev)
        return;

    //
    //  Pull the node out of the list. If its the last one, then handle that
    //  specially. We already checked for it already being at the head.
    //
    if (!pnodeSrc->__pnodeNext)
    {
        // Set our previous nodes's next node to 0
        pnodeSrc->__pnodePrev->__pnodeNext = 0;

        // And make it the new tail
        __pnodeTail = pnodeSrc->__pnodePrev;
    }
     else
    {
        // Set our previous node's next node to our next node
        pnodeSrc->__pnodePrev->__pnodeNext = pnodeSrc->__pnodeNext;

        // Set our next node's previous to our previous
        pnodeSrc->__pnodeNext->__pnodePrev = pnodeSrc->__pnodePrev;
    }

    // Bump down the node count
    __c4ElemCount--;

    // And insert the node at the start
    PrependNode(pnodeSrc);
}


tCIDLib::TVoid TDLinkedList::MoveToTail(TDLstNode* const pnodeSrc)
{
    // <TBD> Insure its a member node if in debug build!

    // If there are not at least 2 nodes, then nothng to do
    if (__c4ElemCount < 2)
        return;

    // If this node is already the last, then nothing to do
    if (!pnodeSrc->__pnodeNext)
        return;

    //
    //  Pull the node out of the list. Handle it being the first node
    //  separately. We already checked whether it was already at the tail.
    //
    if (!pnodeSrc->__pnodePrev)
    {
        // Set our next node's previous node to 0
        pnodeSrc->__pnodeNext->__pnodePrev = 0;

        // Store it as the new head
        __pnodeHead = pnodeSrc->__pnodeNext;
    }
     else
    {
        // Set our previous node's next node to our next node
        pnodeSrc->__pnodePrev->__pnodeNext = pnodeSrc->__pnodeNext;

        // Set our next node's previous to our previous
        pnodeSrc->__pnodeNext->__pnodePrev = pnodeSrc->__pnodePrev;
    }

    // Bump down the node count
    __c4ElemCount--;

    // And append it
    AppendNode(pnodeSrc);
}


tCIDLib::TVoid TDLinkedList::PrependNode(TDLstNode* const pnodeNew)
{
    // If there are no current nodes, then just call AppendNode()
    if (!__c4ElemCount)
    {
        AppendNode(pnodeNew);
        return;
    }

    // <TBD> Insure its not already in the list if in debug build!

    // Make sure that we are not overflowing here
    if (__c4ElemCount >= __c4MaxElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_TooManyNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_OutResource
        );
    }

    // Bump up the count of items
    __c4ElemCount++;

    //
    //  There are only two options here, one was an empty list and was
    //  handled above. The other is that we just patch in the new node.
    //  So get a pointer to the current first node. The new node's prev
    //  node pointer is left 0 to indicate it is the first.
    //
    //  If there was only 1 node, then __pnodeTail still points to the old
    //  head which is correct.
    //
    TDLstNode* pnodeCurFirst = __pnodeHead;

    __pnodeHead = pnodeNew;
    __pnodeHead->__pnodeNext = pnodeCurFirst;
    pnodeCurFirst->__pnodePrev = pnodeNew;
}


tCIDLib::TVoid TDLinkedList::Flush()
{
    //  
    //  If no head node, then an empty list so just return. But clear other
    //  members just in case!
    //
    if (!__pnodeHead)
    {
        __c4ElemCount   = 0;
        __pnodeHead     = 0;
        __pnodeTail     = 0;
        return;
    }

    // There is at least one node, so loop through them
    TDLstNode* pnodeCur  = __pnodeHead;
    TDLstNode* pnodeNext = 0;

    do
    {
        // Before we destroy this node, save its next node
        pnodeNext = pnodeCur->__pnodeNext;

        // Free the node object
        delete pnodeCur;

        // Put the saved next node into the current node pointer
        pnodeCur = pnodeNext;

    }   while (pnodeCur);

    // 0 out the current item count and the head/tail pointers
    __c4ElemCount   = 0;
    __pnodeHead     = 0;
    __pnodeTail     = 0;
}


tCIDLib::TVoid
TDLinkedList::FlushNode(TDLstNode* const pnodeDelete)
{
    // If there is no head node, then an empty list so just return
    if (!__pnodeHead)
        return;

    // If only 1 node, then just flush
    if (__c4ElemCount == 1)
    {
        Flush();
        return;
    }

    // Get a pointer to the next/prev nodes of the passed node
    TDLstNode* pnodePrev = pnodeDelete->__pnodePrev;
    TDLstNode* pnodeNext = pnodeDelete->__pnodeNext;

    //
    //  Adjust the next node of pnodePrev and previous node of the
    //  pnodeNext to skip over the current node, thus removing it from
    //  the chain. Check for the special case that this is the first or
    //  last node.
    //
    if (!pnodeNext)
    {
        //
        //  The delete record is end of the line, so this one is easy. Just
        //  make the pnodePrev's next node nul, which makes it the new end
        //  of the list.
        //
        pnodePrev->__pnodeNext = 0;
        __pnodeTail = pnodePrev;
    }
     else if (!pnodePrev)
    {
        //
        //  The delete record is the start of the line, so we have to point
        //  the list's head to nodeNext and set the next node's previous
        //  node to nul to indicate that it is now the first node.
        //
        __pnodeHead  = pnodeNext;
        pnodeNext->__pnodePrev = 0;
    }
     else
    {
        //
        //  No special conditions, just do the normal patch around the
        //  deleted node
        //
       pnodePrev->__pnodeNext = pnodeNext;
       pnodeNext->__pnodePrev = pnodePrev;
    }

    // Bump down the count of nodes
    __c4ElemCount--;

    // Delete the node object
    delete pnodeDelete;
}
