//
//  FILE NAME: CIDLib_CountedPointer.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This file defines a template class which wraps pointers to things. This
//  class makes it safe have multiple references to a pointer. It does
//  reference counting and destroys the thing pointed to when the last
//  reference is dropped.
//
//  This class is a template so that it can be typesafe and get all of
//  the benefits of a 'smart pointer' type of class, along with the ability
//  to reference count. This makes it convenient as well as safe.
//
//  A safe counter is used to insure that the counter manipulation is fully
//  atomic.
//
//  CAVEATS/GOTCHAS:
//
//  1)  The pointer can be reset to point at something else, which will
//      cause the old pointer to be deleted and the new one stored. However,
//      the reference count must be 1 in order to do it, i.e. there is only
//      one user of the pointer, so its safe to reset it.
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TCntPtr
//  PREFIX: cptr
// ----------------------------------------------------------------------------
template <class T> class TCntPtr :

    public TObject, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TCntPtr() :

            __pCountedData(new TCntPtr<T>::TData<T>)
        {
            // Set the class name on the first creation of a bag of this type
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp(L"TCntPtrOf", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        TCntPtr(T* const pToAdopt) :

            __pCountedData(new TCntPtr<T>::TData<T>(pToAdopt))
        {
            // Set the class name on the first creation of a bag of this type
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp(L"TCntPtrOf", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        TCntPtr(const TCntPtr<T>& cptrToCopy) :

            __pCountedData(0)
        {
            // Set the class name on the first creation of a bag of this type
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp(L"TCntPtrOf", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }

            // Lock the source object
            TKrnlCritSecLocker kcrslSync(&cptrToCopy.__kcrsSync);

            // Bump the ref count first, then copy the data object pointer
            cptrToCopy.__pCountedData->__c4Count++;
            __pCountedData = cptrToCopy.__pCountedData;
        }

        ~TCntPtr()
        {
            // Lock the object
            TKrnlCritSecLocker kcrslSync(&__kcrsSync);

            // If there is any count, then dec and delete if 0
            if (__pCountedData->__c4Count)
            {
                __pCountedData->__c4Count--;
                if (!__pCountedData->__c4Count)
                {
                    delete __pCountedData;
                    __pCountedData = 0;
                }
            }
        }


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        T* operator->()
        {
            // Lock the object and get the data pointer out
            T* pRet;
            {
                TKrnlCritSecLocker kcrslSync(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return pRet;
        }

        const T* operator->() const
        {
            // Lock the object
            const T* pRet;
            {
                TKrnlCritSecLocker kcrslSync(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return pRet;
        }

        const T& operator()() const
        {
            // Lock the object
            T* pRet;
            {
                TKrnlCritSecLocker kcrslThis(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return *pRet;
        }

        T& operator()()
        {
            // Lock the object
            T* pRet;
            {
                TKrnlCritSecLocker kcrslThis(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return *pRet;
        }

        TCntPtr& operator=(const TCntPtr<T>& cptrToAssign)
        {
            if (this == &cptrToAssign)
                return *this;

            // Lock this object and the source object
            TKrnlCritSecLocker kcrslSync(&__kcrsSync);
            TKrnlCritSecLocker kcrslAssign(&cptrToAssign.__kcrsSync);

            //
            //  Bump down the ref count on our object and delete if this is
            //  the last reference.
            //
            if (__pCountedData->__c4Count)
            {
                __pCountedData->__c4Count--;
                if (!__pCountedData->__c4Count)
                {
                    delete __pCountedData;
                    __pCountedData = 0;
                }
            }

            // Bump up the source's ref count, then copy its pointer
            cptrToAssign.__pCountedData->__c4Count++;
            __pCountedData = cptrToAssign.__pCountedData;
            return *this;
        }

        tCIDLib::TBoolean operator==(const TCntPtr<T>& cptrToCompare)
        {
            // Lock both objects
            TKrnlCritSecLocker kcrslSync(&__kcrsSync);
            TKrnlCritSecLocker kcrslAssign(&cptrToCompare.__kcrsSync);

            if ((__pCountedData->__c4Count != cptrToCompare.__pCountedData->__c4Count)
            ||  (__pCountedData->__pData != cptrToCompare.__pCountedData->__pData))
            {
                return kCIDLib::False;
            }
            return kCIDLib::True;
        }

        tCIDLib::TBoolean operator!=(const TCntPtr<T>& cptrToCompare)
        {
            return !operator==(cptrToCompare);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TCard4 c4RefCount() const
        {
            tCIDLib::TCard4 c4Count;
            {
                TKrnlCritSecLocker kcrslSync(&__kcrsSync);
                c4Count = __pCountedData->__c4Count;
            }
            return c4Count;
        }

        T& objData()
        {
            T* pRet;
            {
                TKrnlCritSecLocker kcrslSync(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return *pRet;
        }

        const T& objData() const
        {
            T* pRet;
            {
                TKrnlCritSecLocker kcrslSync(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return *pRet;
        }

        T* pobjData()
        {
            T* pRet;
            {
                TKrnlCritSecLocker kcrslSync(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return pRet;
        }

        const T* pobjData() const
        {
            const T* pRet;
            {
                TKrnlCritSecLocker kcrslSync(&__kcrsSync);
                pRet = __pCountedData->__pData;
            }
            return pRet;
        }

        tCIDLib::TVoid SetPointer(T* const pToAdopt)
        {
            TKrnlCritSecLocker kcrslSync(&__kcrsSync);

            //
            //  The count cannot be greater than 1 when we do this because that
            //  would replace the pointer when there are references to it.
            //
            if (__pCountedData->__c4Count > 1)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCPtr_InUse
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , TCardinal(__pCountedData->__c4Count)
                );
            }

            // Set the count down to 0 and delete the current data
            delete __pCountedData->__pData;
            __pCountedData->__c4Count = 0;

            //
            //  Store the new data and set the usage to 1 (since it could have
            //  been 0.
            //
            __pCountedData->__pData = pToAdopt;
            __pCountedData->__c4Count = 1;
        }


    private :
        // --------------------------------------------------------------------
        //  Private data types
        // --------------------------------------------------------------------
        template <class T> class TData
        {
            public  :
                // ------------------------------------------------------------
                //  Our parent class is our friend
                // ------------------------------------------------------------
                friend class TCntPtr<T>;

                // ------------------------------------------------------------
                //  Constructors and destructors
                // ------------------------------------------------------------
                TData(T* const pToAdopt) :

                    __pData(pToAdopt)
                    , __c4Count(1)
                {
                }

                TData() :

                    __pData(0)
                    , __c4Count(0)
                {
                }

                ~TData()
                {
                    delete __pData;
                    __pData = 0;
                }

            private :
                // ------------------------------------------------------------
                //  Private data
                //
                //  __c4Count
                //      This is the usage count for the data.
                //
                //  __pData
                //      This is the pointer being reference counted. It is
                //      set during construction and deleted and cleared 
                //      during destruction of the last reference.
                // ------------------------------------------------------------
                tCIDLib::TCard4     __c4Count;
                T*                  __pData;
        };


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __kcrsSync
        //      This is a critical section that is used to protect access to
        //      the data and count. This guy is mutable because its just an
        //      internal implementation detail and not counted in any equality
        //      or externally visible test.
        //
        //  __pCountedData
        //      This the pointer to the counted data 
        // --------------------------------------------------------------------
        mutable TKrnlCritSec    __kcrsSync;
        TData<T>*               __pCountedData;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a bag cursor of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any standard macros
        // --------------------------------------------------------------------
        RTTIMacros(TCntPtr<T>,TObject)
        DefPolyDup(TCntPtr<T>)
};

template <class T> tCIDLib::TBoolean TCntPtr<T>::__bFirstOfType = kCIDLib::False;
template <class T> const TClass TCntPtr<T>::clsThis;

#pragma pack(pop)
