//
// NAME: CIDLib_Console.Cpp
//
// DESCRIPTION:
//
//	This module implements the TConsole class, a specialized stream for
//  getting interactive input from the console, though it can be redirectable
//  also.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TConsoleImpl,TTextStreamImpl)
RTTIData2(TConsole,TTextStream)


// ----------------------------------------------------------------------------
//  Local static data
//
//  __c4DefRecall
//      This is the size of the recall buffer used by the default constructor.
//      This will generally be sufficient.
// ----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4DefRecall   = 32;



// ----------------------------------------------------------------------------
//   CLASS: TConsoleImpl
//  PREFIX: strmi
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TConsoleImpl: Constructors and destructors
// ----------------------------------------------------------------------------

TConsoleImpl::TConsoleImpl()
{
    try
    {
        __kconImpl.Open(kCIDLib::False, 0, tCIDLib::ERedir_Allow);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_Open
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}

TConsoleImpl::TConsoleImpl( const   tCIDLib::TBoolean   bInsertMode
                            , const tCIDLib::TCard4     c4MaxRecall
                            , const tCIDLib::ERedir     eRedirect)
{
    try
    {
        __kconImpl.Open(bInsertMode, c4MaxRecall, eRedirect);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_Open
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}

TConsoleImpl::TConsoleImpl( const   tCIDLib::TBoolean       bInsertMode
                            , const tCIDLib::TCard4         c4MaxRecall
                            , const tCIDLib::ETextFormats   eTextFmt
                            , const tCIDLib::ERedir         eRedirect) :

    TTextStreamImpl(eTextFmt)
    , __kconImpl(eTextFmt)
{
    try
    {
        __kconImpl.Open(bInsertMode, c4MaxRecall, eRedirect);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_Open
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}

TConsoleImpl::~TConsoleImpl()
{
}


// ----------------------------------------------------------------------------
//  TConsoleImpl: Public, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TCard4
TConsoleImpl::c4GetLine(        tCIDLib::Tch* const     pszBufToFill
                        , const tCIDLib::TCard4         c4BufSize
                        , const tCIDLib::TBoolean       bStripWhitespace)
{
    tCIDLib::TCard4 c4Len = 0;
    try
    {
        c4Len = __kconImpl.c4GetLine
        (
            pszBufToFill
            , c4BufSize
            , bStripWhitespace
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if (kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
        {
            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStrm_EndOfStream
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_OutResource
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcCon_TextReadError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
    return c4Len;
}


tCIDLib::Tch TConsoleImpl::chGet()
{
    try
    {
        return __kconImpl.chRead();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if ((kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
        ||  (kerrToCatch.errcId() == kKrnlErrors::errcNotAllData))
        {
            tCIDLib::TErrCode errcThrow;
            if (kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
                errcThrow = kCIDErrs::errcStrm_EndOfStream;
            else
                errcThrow = kCIDErrs::errcStrm_NotAllData;

            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , errcThrow
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_OutResource
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcCon_TextReadError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
}


tCIDLib::TBoolean TConsoleImpl::bEndOfStream() const
{
    try
    {
        return __kconImpl.bAtEnd();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_QueryAtEnd
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TFilePos TConsoleImpl::fposCurPos() const
{
    try
    {
        return __kconImpl.fposCurPos();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_QueryPosition
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(L"current")
        );
    }
}


tCIDLib::TFilePos TConsoleImpl::fposLogicalEnd() const
{
    try
    {
        return __kconImpl.fposLogicalEnd();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_QueryPosition
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(L"logical end")
        );
    }
}

tCIDLib::TFilePos TConsoleImpl::fposPhysicalEnd() const
{
    try
    {
        return __kconImpl.fposPhysicalEnd();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_QueryPosition
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(L"physical end")
        );
    }
}

tCIDLib::TFilePos
TConsoleImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    try
    {
        return __kconImpl.fposSkipForwardBy(c4SkipBy);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_SeekError
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TFilePos TConsoleImpl::fposSeekToEnd()
{
    try
    {
        return __kconImpl.fposSeekToEnd();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_SeekError
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TVoid TConsoleImpl::PutCh(const tCIDLib::Tch chToWrite)
{
    try
    {
        __kconImpl.PutCh(chToWrite);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if ((kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
        ||  (kerrToCatch.errcId() == kKrnlErrors::errcNotAllData))
        {
            tCIDLib::TErrCode errcThrow;
            if (kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
                errcThrow = kCIDErrs::errcStrm_EndOfStream;
            else
                errcThrow = kCIDErrs::errcStrm_NotAllData;

            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , errcThrow
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_OutResource
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcCon_TextWriteError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
}

tCIDLib::TVoid TConsoleImpl::PutCh(const tCIDLib::Tsch schToWrite)
{
    try
    {
        __kconImpl.PutCh(schToWrite);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if ((kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
        ||  (kerrToCatch.errcId() == kKrnlErrors::errcNotAllData))
        {
            tCIDLib::TErrCode errcThrow;
            if (kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
                errcThrow = kCIDErrs::errcStrm_EndOfStream;
            else
                errcThrow = kCIDErrs::errcStrm_NotAllData;

            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , errcThrow
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_OutResource
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcCon_TextWriteError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
}


tCIDLib::TVoid
TConsoleImpl::PutLine(const tCIDLib::Tch* const pszBufToWrite)
{
    try
    {
        __kconImpl.PutLine(pszBufToWrite);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if ((kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
        ||  (kerrToCatch.errcId() == kKrnlErrors::errcNotAllData))
        {
            tCIDLib::TErrCode errcThrow;
            if (kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
                errcThrow = kCIDErrs::errcStrm_EndOfStream;
            else
                errcThrow = kCIDErrs::errcStrm_NotAllData;

            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , errcThrow
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_OutResource
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcCon_TextWriteError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
}

tCIDLib::TVoid
TConsoleImpl::PutLine(const tCIDLib::Tsch* const pszBufToWrite)
{
    try
    {
        __kconImpl.PutLine(pszBufToWrite);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if ((kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
        ||  (kerrToCatch.errcId() == kKrnlErrors::errcNotAllData))
        {
            tCIDLib::TErrCode errcThrow;
            if (kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
                errcThrow = kCIDErrs::errcStrm_EndOfStream;
            else
                errcThrow = kCIDErrs::errcStrm_NotAllData;

            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , errcThrow
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_OutResource
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcCon_TextWriteError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
}


tCIDLib::TVoid TConsoleImpl::Reset()
{
    try
    {
        __kconImpl.Reset();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCon_SeekError
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::Tsch TConsoleImpl::schGet()
{
    try
    {
        return __kconImpl.schRead();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if ((kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
        ||  (kerrToCatch.errcId() == kKrnlErrors::errcNotAllData))
        {
            tCIDLib::TErrCode errcThrow;
            if (kerrToCatch.errcId() == kKrnlErrors::errcEndOfData)
                errcThrow = kCIDErrs::errcStrm_EndOfStream;
            else
                errcThrow = kCIDErrs::errcStrm_NotAllData;

            facCIDLib.ThrowKrnlErr
            (
                __FILE__
                , __LINE__
                , errcThrow
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_OutResource
            );
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcCon_TextReadError
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
}


tCIDLib::TVoid TConsoleImpl::TruncateAtZero()
{
    facCIDLib.LogErr
    (
        __FILE__
        , __LINE__
        , kCIDErrs::errcStrm_NotSupported
        , tCIDLib::ESev_APIFailed
        , tCIDLib::EClass_CantDo
        , clsIsA()
        , TString(L"truncation")
    );
}



// ----------------------------------------------------------------------------
//   CLASS: TConsole
//  PREFIX: con
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TConsole: Constructors and Destructors
// ----------------------------------------------------------------------------

TConsole::TConsole()
{
    try
    {
        _AdoptImplObject(new TConsoleImpl(kCIDLib::False, __c4DefRecall));
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_CreateImpl
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TConsoleImpl::clsThis
        );
    }

}

TConsole::TConsole( const   tCIDLib::TBoolean       bInsertMode
                    , const tCIDLib::TCard4         c4RecallBufSize
                    , const tCIDLib::ERedir         eRedirect)
{
    try
    {
        _AdoptImplObject
        (
            new TConsoleImpl(bInsertMode, c4RecallBufSize, eRedirect)
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_CreateImpl
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TConsoleImpl::clsThis
        );
    }
}

TConsole::TConsole( const   tCIDLib::TBoolean       bInsertMode
                    , const tCIDLib::TCard4         c4RecallBufSize
                    , const tCIDLib::ETextFormats   eTextFmt
                    , const tCIDLib::ERedir         eRedirect)
{
    try
    {
        _AdoptImplObject
        (
            new TConsoleImpl
            (
                bInsertMode
                , c4RecallBufSize
                , eTextFmt
                , eRedirect
            )
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_CreateImpl
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TConsoleImpl::clsThis
        );
    }
}


TConsole::~TConsole()
{
}
