//
//  FILE NAME: CIDLib_Comparator.Hpp
//
//  DESCRIPTION:
//
//  This file implements abstract base classes which are the base for all
//  'comparator' classes. These classes are used in collections (and in
//  other places as well, though collections are their primary clients) to
//  compare collection elements. Often times, the equality or < > operators
//  of an element are not appropriate, because not the whole object is of
//  interest for sorting or searching.
//
//  The base equality class is TObjEq, which defines the virtual bEqual()
//  method. It returns a boolean that indicates whether the elements are
//  equal or not. Derivatives override this and do the desired element
//  comparison.
//
//  The base relative magnitude class is TObjComp which defines the virtual
//  eCompare() method. Derivatives override this class and do the desired
//  comparison. It returns a value that indicates the relative magnitude of
//  the two passed objects.
//
//  Also defined here are the obvious derivatives which use the standard
//  < and == operators. TStdObjEq provides the standard equality test using
//  the == operator. TStdObjComp provides the standard relative magnitude
//  test, using the < and == operators.
//
//  AUTHOR: Dean Roddey
//
//  CREATED: 07/25/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  CAVEATS/GOTCHAS:
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TObjComp
//  PREFIX: comp
// ----------------------------------------------------------------------------
template <class T> class TObjComp : public TObject, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TObjComp()
        {
            __CheckFirstOfType();
        }

        TObjComp(const TObjComp<T>& compToCopy)
        {
            __CheckFirstOfType();
        }

        ~TObjComp() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TObjComp<T>& operator=(const TObjComp<T>&)
        {
            // Just a dummy to make life consistent for derived classes
            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, virtual methods
        // --------------------------------------------------------------------
        virtual tCIDLib::ESortComps eCompare
        (
            const   T&                  obj1
            , const T&                  obj2
        )   const = 0;


    private :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TObjCompFor", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Do any needed RTTI macros
        // --------------------------------------------------------------------
        RTTIMacros(TObjComp<T>,TObject)


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;
};

template <class T> tCIDLib::TBoolean TObjComp<T>::__bFirstOfType = kCIDLib::False;
template <class T> const TClass TObjComp<T>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TStdObjComp
//  PREFIX: comp
// ----------------------------------------------------------------------------
template <class T> class TStdObjComp : public TObjComp<T>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TStdObjComp()
        {
            __CheckFirstOfType();
        }

        TStdObjComp(const TStdObjComp<T>& compToCopy)
        {
            __CheckFirstOfType();
        }

        ~TStdObjComp() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TStdObjComp<T>& operator=(const TStdObjComp<T>&)
        {
            // Just a dummy to make life consistent for derived classes
            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::ESortComps eCompare(const T& obj1, const T& obj2) const
        {
            if (obj1 < obj2)
                return tCIDLib::ESort_FirstLess;
            else if (obj1 == obj2)
                return tCIDLib::ESort_Equal;
            return tCIDLib::ESort_FirstGreater;
        }


    private :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TStdObjCompFor", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        // --------------------------------------------------------------------
        //  Do any needed RTTI macros
        // --------------------------------------------------------------------
        RTTIMacros(TStdObjComp<T>,TObjComp<T>)
        DefPolyDup(TStdObjComp<T>)


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;
};

template <class T> tCIDLib::TBoolean TStdObjComp<T>::__bFirstOfType = kCIDLib::False;
template <class T> const TClass TStdObjComp<T>::clsThis;




// ----------------------------------------------------------------------------
//   CLASS: TObjEq
//  PREFIX: comp
// ----------------------------------------------------------------------------
template <class T> class TObjEq : public TObject, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TObjEq()
        {
            __CheckFirstOfType();
        }

        TObjEq(const TObjEq<T>& compToCopy)
        {
            __CheckFirstOfType();
        }

        ~TObjEq() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TObjEq<T>& operator=(const TObjEq<T>&)
        {
            // Just a dummy to make life consistent for derived classes
            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, virtual methods
        // --------------------------------------------------------------------
        virtual tCIDLib::TBoolean bEqual
        (
            const   T&                  obj1
            , const T&                  obj2
        )   const = 0;


    private :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TObjEqFor", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        // --------------------------------------------------------------------
        //  Do any needed RTTI macros
        // --------------------------------------------------------------------
        RTTIMacros(TObjEq<T>,TObject)


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;
};

template <class T> tCIDLib::TBoolean TObjEq<T>::__bFirstOfType = kCIDLib::False;
template <class T> const TClass TObjEq<T>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TStdObjEq
//  PREFIX: comp
// ----------------------------------------------------------------------------
template <class T> class TStdObjEq : public TObjEq<T>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TStdObjEq()
        {
            __CheckFirstOfType();
        }

        TStdObjEq(const TStdObjEq<T>& compToCopy)
        {
            __CheckFirstOfType();
        }

        ~TStdObjEq() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TStdObjEq<T>& operator=(const TStdObjEq<T>&)
        {
            // Just a dummy to make life consistent for derived classes
            return *this;
        }

        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bEqual(const T& obj1, const T& obj2) const
        {
            return (obj1 == obj2);
        }


    private :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TStdObjEqFor", 64);
                strTmp.Append(T::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }

        // --------------------------------------------------------------------
        //  Do any needed RTTI macros
        // --------------------------------------------------------------------
        RTTIMacros(TStdObjEq<T>,TObjEq<T>)
        DefPolyDup(TStdObjEq<T>)


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;
};

template <class T> tCIDLib::TBoolean TStdObjEq<T>::__bFirstOfType = kCIDLib::False;
template <class T> const TClass TStdObjEq<T>::clsThis;

#pragma pack(pop)
