//
// NAME: CIDLib_ColorPalette.Cpp
//
// DESCRIPTION:
//
//	This module implements the TClrPalette class, which supports color
//  palettes of arbitary size up to 8K colors.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/24/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"
#include    "CIDLib_RawPalettes_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TClrPalette,TObject)



// -----------------------------------------------------------------------------
//   CLASS: TClrPalette
//  PREFIX: pal
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TClrPalette: Constructors and Destructors
// -----------------------------------------------------------------------------

TClrPalette::TClrPalette() :

    __c4ClrCount(0)
    , __pargbClrs(0)
{
    __c4ClrCount = 2;
    __pargbClrs = new TRGBClr[2];
    __pargbClrs[0] = TRGBClr(0,0,0);
    __pargbClrs[1] = TRGBClr(255,255,255);
}

TClrPalette::TClrPalette(const tCIDLib::EDefPalettes ePalette) :

    __c4ClrCount(0)
    , __pargbClrs(0)
{
    if (ePalette == tCIDLib::EDefPalette_BlackAndWhite)
    {
        __c4ClrCount = 2;
        __pargbClrs = new TRGBClr[2];
        __pargbClrs[0] = TRGBClr(0,0,0);
        __pargbClrs[1] = TRGBClr(255,255,255);
    }
     else if (ePalette == tCIDLib::EDefPalette_16Default)
    {
        __c4ClrCount = 16;
        __pargbClrs = new TRGBClr[__c4ClrCount];
        for (tCIDLib::TCard4 c4Val = 0; c4Val < __c4ClrCount; c4Val++)
            __pargbClrs[c4Val] = __argbDefault16[c4Val];
    }
     else if (ePalette == tCIDLib::EDefPalette_16GrayScale)
    {
        __c4ClrCount = 16;
        __pargbClrs = new TRGBClr[__c4ClrCount];
        for (tCIDLib::TCard1 c1Val = 0; c1Val < __c4ClrCount; c1Val++)
            __pargbClrs[c1Val] = TRGBClr(c1Val, c1Val, c1Val);
    }
     else if (ePalette == tCIDLib::EDefPalette_256Default)
    {
        __c4ClrCount = 256;
        __pargbClrs = new TRGBClr[__c4ClrCount];
        for (tCIDLib::TCard4 c4Val = 0; c4Val < __c4ClrCount; c4Val++)
            __pargbClrs[c4Val] = __argbDefault256[c4Val];
    }
     else if (ePalette == tCIDLib::EDefPalette_256GrayScale)
    {
        __c4ClrCount = 256;
        __pargbClrs = new TRGBClr[__c4ClrCount];
        for (tCIDLib::TCard4 c4Val = 0; c4Val < __c4ClrCount; c4Val++)
            __pargbClrs[c4Val] = __argbGrayScale[c4Val];
    }
    #if CID_DEBUG_ON
     else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPal_Type
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(ePalette)
        );
    }
    #endif
}

TClrPalette::TClrPalette(const tCIDLib::TCard4 c4ClrCnt) :

    __c4ClrCount(c4ClrCnt)
    , __pargbClrs(0)
{
    // The count cannot be 0
    if (!__c4ClrCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPal_ZeroElems
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    __pargbClrs = new TRGBClr[c4ClrCnt];
}

TClrPalette::TClrPalette(   const   tCIDLib::TCard4         c4ClrCnt
                            ,       TElemCursor<TRGBClr>&   cursInitList) :

    __c4ClrCount(c4ClrCnt)
    , __pargbClrs(0)
{
    // Cannot be zero
    if (!c4ClrCnt)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPal_ZeroElems
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }

    __pargbClrs = new TRGBClr[c4ClrCnt];

    // If not enough colors in list, then fatal error
    if (__c4ClrCount > cursInitList.c4ElemCount())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPal_InitCount
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(cursInitList.c4ElemCount())
            , TCardinal(__c4ClrCount)
        );
    }

    // Iterate the container and init the array
    tCIDLib::TCard4 c4Ind   = 0;
    do
    {
        // Copy the color to our array
        __pargbClrs[c4Ind] = cursInitList.objCur();

        // Bump up the index
        c4Ind++;

    }   while (cursInitList.bNext());
}

TClrPalette::TClrPalette(const TClrPalette& palToCopy) :

    __c4ClrCount(palToCopy.__c4ClrCount)
    , __pargbClrs(new TRGBClr[palToCopy.__c4ClrCount])
{
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4ClrCount; c4Ind++)
        __pargbClrs[c4Ind] = palToCopy.__pargbClrs[c4Ind];
}

TClrPalette::~TClrPalette()
{
    delete [] __pargbClrs;
    __pargbClrs = 0;
}


// -----------------------------------------------------------------------------
//  TClrPalette: Public operators
// -----------------------------------------------------------------------------

TClrPalette& TClrPalette::operator=(const TClrPalette& palToAssign)
{
    // If the source is a different size, then reallocate
    if (__c4ClrCount != palToAssign.__c4ClrCount)
    {
        // Free the current color list
        delete [] __pargbClrs;
        __pargbClrs = 0;

        // Allocate the new list
        __c4ClrCount = palToAssign.__c4ClrCount;
        __pargbClrs = new TRGBClr[__c4ClrCount];
    }

    // Loop though and copy the colors
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4ClrCount; c4Ind++)
        __pargbClrs[c4Ind] = palToAssign.__pargbClrs[c4Ind];

    return *this;
}


tCIDLib::TBoolean
TClrPalette::operator==(const TClrPalette& palToTest) const
{
    if (__c4ClrCount != palToTest.__c4ClrCount)
        return kCIDLib::False;

    // Look through the colors
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4ClrCount; c4Ind++)
    {
        if (__pargbClrs[c4Ind] != palToTest.__pargbClrs[c4Ind])
            return kCIDLib::False;
    }
    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TClrPalette: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4
TClrPalette::c4ParseFromText(   const   TString&                strFile
                                , const tCIDLib::EPalTxtFormats eFormat
                                , const tCIDLib::ETextFormats   eTextFmt)
{
    //
    //  Create a text stream based on a file, and open it for exclusive
    //  read/write. Fail if it does not exist.
    //
    TTextFileStream strmSource
    (
        strFile
        , eTextFmt
        , tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_OpenIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    //
    //  Create a stream parser for the file. We pass it by reference, not
    //  pointer, so it knows that it does not adopt the stream.
    //
    TStreamParser   prsrSource(&strmSource, tCIDLib::EAdoptOpt_NoAdopt);

    // And call the other version
    return c4ParseFromText(prsrSource, eFormat);
}

tCIDLib::TCard4
TClrPalette::c4ParseFromText(           TStreamParser&          prsrSrc
                                , const tCIDLib::EPalTxtFormats eFormat)
{
    tCIDLib::TCard4 c4Ind;
    TString strSeparators(kCIDLib::szWhitespace, 16);

    // Add to the correct list of separators if needed
    if ((eFormat == tCIDLib::EPalFmt_CommaDelimited)
    ||  (eFormat == tCIDLib::EPalFmt_CommaDelimited_OnePer))
        strSeparators.Append(kCIDLib::chComma);

    try
    {
        TString strTmp(kCIDLib::pszEmptyZStr, 64);
        for (c4Ind = 0; c4Ind < __c4ClrCount; c4Ind++)
        {
            //
            //  Get the next value, which should be the red component. Try to
            //  convert it to a TCard1. If this fails, then it will log an
            //  error. Do the same for the green and blue. Check for nothing
            //  coming back on the red, which is legal and means we hit the
            //  end.
            //
            prsrSrc.GetNextToken(strSeparators, strTmp);
            if (!strTmp.c4Length())
                break;
            __pargbClrs[c4Ind].c1Red(tCIDLib::TCard1(strTmp.c4Val()));

            prsrSrc.GetNextToken(strSeparators, strTmp);
            if (!strTmp.c4Length())
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcPal_TextFormat
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_BadParms
                    , TCardinal(prsrSrc.c4CurLine())
                    , TString(L"green color component")
                );
            }
            __pargbClrs[c4Ind].c1Green(tCIDLib::TCard1(strTmp.c4Val()));

            prsrSrc.GetNextToken(strSeparators, strTmp);
            if (!strTmp.c4Length())
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcPal_TextFormat
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_BadParms
                    , TCardinal(prsrSrc.c4CurLine())
                    , TString(L"blue color component")
                );
            }
            __pargbClrs[c4Ind].c1Blue(tCIDLib::TCard1(strTmp.c4Val()));

            // If its a one-per type format, then flush the rest of this line
            if ((eFormat == tCIDLib::EPalFmt_CommaDelimited_OnePer)
            ||  (eFormat == tCIDLib::EPalFmt_SpaceDelimited_OnePer))
            {
                prsrSrc.FlushLine();
            }
        }
    }

    catch(const TError& errToCatch)
    {
        if (errToCatch.bCheckError(facCIDLib, kCIDErrs::errcPal_TextFormat))
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcPal_ParseError
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_BadParms
                , TCardinal(prsrSrc.c4CurLine())
                , errToCatch.strErrText()
            );
        }
    }
    return c4Ind;
}


const TRGBClr& TClrPalette::rgbAt(const tCIDLib::TCard4 c4Ind) const
{
    if (c4Ind >= __c4ClrCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPal_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Ind)
            , TCardinal(__c4ClrCount-1)
        );
    }
    return __pargbClrs[c4Ind];
}

TRGBClr& TClrPalette::rgbAt(const tCIDLib::TCard4 c4Ind)
{
    if (c4Ind >= __c4ClrCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPal_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Ind)
            , TCardinal(__c4ClrCount-1)
        );
    }
    return __pargbClrs[c4Ind];
}


tCIDLib::TVoid
TClrPalette::RotateBackward(const tCIDLib::TCard4 c4Count)
{
    // Check for a count same as palette or 0. If so, nothing to do
    if ((c4Count == __c4ClrCount) || !c4Count)
        return;

    // Check for a count larger than palette size. If so, do modulo
    tCIDLib::TCard4 c4ActualCnt = c4Count;
    if (c4ActualCnt > __c4ClrCount)
        c4ActualCnt = (c4ActualCnt % __c4ClrCount);

    // Allocate a new buffer
    TRGBClr* const pargbTmp = new TRGBClr[__c4ClrCount];

    try
    {
        //
        //  Start filling in the new buffer at the 0th element, from the
        //  c4ActualCnt'th element in the current buffer. Go to the end of the
        //  current array.
        //
        tCIDLib::TCard4 c4NewInd = 0;
        tCIDLib::TCard4 c4Ind;
        for (c4Ind = c4ActualCnt; c4Ind < __c4ClrCount; c4Ind++)
            pargbTmp[c4NewInd++] = __pargbClrs[c4ActualCnt];

        //
        //  Now copy from the old array at the 0'th element into the new
        //  array where the c4NewInd left off.
        //
        for (c4Ind = 0; c4Ind < c4ActualCnt; c4Ind++)
            pargbTmp[c4NewInd++] = __pargbClrs[c4Ind];
    }

    catch (...)
    {
        delete [] pargbTmp;
        throw;
    }

    // Delete the old array and keep the new one
    delete [] __pargbClrs;
    __pargbClrs = pargbTmp;
}

tCIDLib::TVoid TClrPalette::RotateForward(const tCIDLib::TCard4 c4Count)
{
    // Check for a count same as palette or 0. If so, nothing to do
    if ((c4Count == __c4ClrCount) || !c4Count)
        return;

    // Check for a count larger than palette size. If so, do modulo
    tCIDLib::TCard4 c4ActualCnt = c4Count;
    if (c4ActualCnt > __c4ClrCount)
        c4ActualCnt = (c4ActualCnt % __c4ClrCount);

    // Allocate a new buffer
    TRGBClr* const pargbTmp = new TRGBClr[__c4ClrCount];

    try
    {
        //
        //  Start filling in the new buffer at the c4ActualCnt'th element,
        //  from the 0'th element in the current buffer. Go to the end of
        //  the current array.
        //
        tCIDLib::TCard4 c4OldInd = 0;
        tCIDLib::TCard4 c4Ind;
        for (c4Ind = c4ActualCnt; c4Ind < __c4ClrCount; c4Ind++)
            pargbTmp[c4Ind] = __pargbClrs[c4OldInd++];

        //
        //  Now start copying from the c4OldInd'th element of the old array to
        //  the c4Ind'th element of the new array.
        //
        for (c4Ind = 0; c4Ind < c4ActualCnt; c4Ind++)
            pargbTmp[c4Ind] = __pargbClrs[c4OldInd++];
    }

    catch (...)
    {
        delete [] pargbTmp;
        throw;
    }

    // Delete the old array and keep the new one
    delete [] __pargbClrs;
    __pargbClrs = pargbTmp;
}


// -----------------------------------------------------------------------------
//  TClrPalette: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TClrPalette::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Get the color count, saving the current count first
    tCIDLib::TCard4  c4OldCount = __c4ClrCount;
    strmToReadFrom >> __c4ClrCount;

    //
    //  If the color count has changed, then delete the current array
    //  and reallocate.
    //
    if (c4OldCount != __c4ClrCount)
    {
        delete [] __pargbClrs;
        __pargbClrs = new TRGBClr[__c4ClrCount];
    }

    // Loop through the colors
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4ClrCount; c4Ind++)
        strmToReadFrom >> __pargbClrs[c4Ind];
}

tCIDLib::TVoid TClrPalette::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the color count
    strmToWriteTo << __c4ClrCount;

    // Loop through the colors and write them
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4ClrCount; c4Ind++)
        strmToWriteTo << __pargbClrs[c4Ind];
}
