//
// NAME: CIDLib_Color.Hpp
//
// DESCRIPTION:
//
//  This is the header file for the CIDLib_Color.Cpp module, which
//  implements the TRGBClr and TFRGBClr classes. The TRGBClr class is a
//  pretty simple class that is used for all colors in CIDLib color output
//  to the underlying operating system. It contains a value from 0..255 for
//  the red, green, and blue components of a color value. It has methods for
//  manipulating those components in interesting ways.
//
//  TFRGBClr is a floating point version of TRGBClr that is used in ray
//  tracing and some other applications. It contains a value from 0.0
//  to 1.0 for each color component. It is always translated to a regular
//  TRGBClr value for output. It has a lot of methods for manipulating it
//  in various ways. It also supports an opacity value that controls its
//  transparency.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 10/05/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TRGBClr
//  PREFIX: rgb
// ----------------------------------------------------------------------------
class CIDLIBEXP TRGBClr :

    public TObject, public MDuplicable, public MFormattable, public MStreamable
{
    public  :
        #pragma pack(push, 1)
        struct TRawRGB
        {
            tCIDLib::TCard1 c1Red;
            tCIDLib::TCard1 c1Green;
            tCIDLib::TCard1 c1Blue;
        };

        union TCodedRGB
        {
            tCIDLib::TCard1 c1Red;
            tCIDLib::TCard1 c1Green;
            tCIDLib::TCard1 c1Blue;
            tCIDLib::TCard1 c1Dummy;
        };
        #pragma pack(pop)


        // --------------------------------------------------------------------
        //  Constructors and Destructors.
        // --------------------------------------------------------------------
        TRGBClr();

        TRGBClr
        (
            const   tCIDLib::TCard1         c1Red
            , const tCIDLib::TCard1         c1Green
            , const tCIDLib::TCard1         c1Blue
        );

        TRGBClr
        (
            const   TRawRGB&                rgbRaw
        );

        TRGBClr
        (
            const   TRGBClr&                rgbToCopy
        );

        TRGBClr
        (
            const   tCIDLib::TCard4         c4Color
        );

        TRGBClr
        (
            const   tCIDLib::ESysColors     eSysColor
        );

        ~TRGBClr() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TRGBClr& operator=
        (
            const   tCIDLib::TCard4         c4Color
        );

        TRGBClr& operator=
        (
            const   TRGBClr::TRawRGB&       rgbRaw
        );

        TRGBClr& operator=
        (
            const   TRGBClr::TCodedRGB&     rgbCoded
        );

        TRGBClr& operator=
        (
            const   TRGBClr&                rgbToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TRGBClr&                rgbToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TRGBClr&                rgbToTest
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Adjust
        (
            const   tCIDLib::TInt4          i4RedOffset
            , const tCIDLib::TInt4          i4GreenOffset
            , const tCIDLib::TInt4          i4BlueOffset
        );

        tCIDLib::TCard1 c1AdjustBlue
        (
            const   tCIDLib::TInt4          i4Offset
        );

        tCIDLib::TCard1 c1AdjustGreen
        (
            const   tCIDLib::TInt4          i4Offset
        );

        tCIDLib::TCard1 c1AdjustRed
        (
            const   tCIDLib::TInt4          i4Offset
        );

        tCIDLib::TCard1 c1Blue() const;

        tCIDLib::TCard1 c1Blue
        (
            const   tCIDLib::TCard1         c1Blue
        );

        tCIDLib::TCard1 c1Green() const;

        tCIDLib::TCard1 c1Green
        (
            const   tCIDLib::TCard1         c1Green
        );

        tCIDLib::TCard1 c1Red() const;

        tCIDLib::TCard1 c1Red
        (
            const   tCIDLib::TCard1         c1Red
        );

        tCIDLib::TCard4 c4Color() const;

        tCIDLib::TCard4 c4Magnitude() const;

        tCIDLib::TVoid Set
        (
            const   tCIDLib::TCard1         c1Red
            , const tCIDLib::TCard1         c1Green
            , const tCIDLib::TCard1         c1Blue
        );

        tCIDLib::TVoid ToBlack();

        tCIDLib::TVoid ToRGB
        (
                    TRGBClr::TRawRGB&       rgbTarget
        )   const;


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private         :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c1Red
        //  __c1Green
        //  __c1Blue
        //      These are the color components, stored in the correct order
        //      to match the system format.
        //
        //  __c1Dummy
        //      This is a dummy byte to make the format of the data come out
        //      just like a color encoded in a 32 bit value.
        // --------------------------------------------------------------------
        #pragma pack(1)
        tCIDLib::TCard1     __c1Red;
        tCIDLib::TCard1     __c1Green;
        tCIDLib::TCard1     __c1Blue;
        tCIDLib::TCard1     __c1Dummy;
        #pragma pack()


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRGBClr,TObject)
        DefPolyDup(TRGBClr)
};



// ----------------------------------------------------------------------------
//  CLASS: TFRGBClr
// PREFIX: frgb
// ----------------------------------------------------------------------------
class CIDLIBEXP TFRGBClr :

    public TObject, public MDuplicable, public MStreamable, public MFormattable
{
    public  :
        // --------------------------------------------------------------------
        // Constructors and Destructors
        // --------------------------------------------------------------------
        TFRGBClr();

        TFRGBClr
        (
            const   tCIDLib::TFloat8&       f8Red
            , const tCIDLib::TFloat8&       f8Green
            , const tCIDLib::TFloat8&       f8Blue
            , const tCIDLib::TFloat8&       f8Alpha = 0.0
        );

        TFRGBClr
        (
            const   TFRGBClr&               frgbToCopy
        );

        ~TFRGBClr();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TFRGBClr& operator=
        (
            const   TFRGBClr&               frgbToAssign
        );

        friend TFRGBClr CIDLIBEXP operator+
        (
            const   TFRGBClr&               frgb1
            , const TFRGBClr&               frgb2
        );

        friend TFRGBClr CIDLIBEXP operator+
        (
            const   TFRGBClr&               frgbSrc
            , const tCIDLib::TFloat8&       f8Inc
        );

        tCIDLib::TVoid operator+=
        (
            const   TFRGBClr&               frgbSrc
        );

        tCIDLib::TVoid operator+=
        (
            const   tCIDLib::TFloat8&       f8Inc
        );

        friend TFRGBClr CIDLIBEXP operator-
        (
            const   TFRGBClr&               frgb1
            , const TFRGBClr&               frgb2
        );

        friend TFRGBClr CIDLIBEXP operator-
        (
            const   TFRGBClr&               frgbSrc
            , const tCIDLib::TFloat8&       f8Dec
        );

        tCIDLib::TVoid operator-=
        (
            const   TFRGBClr&               frgbSrc
        );

        tCIDLib::TVoid operator-=
        (
            const   tCIDLib::TFloat8&       f8Dec
        );

        friend TFRGBClr CIDLIBEXP operator*
        (
            const   TFRGBClr&               frgb1
            , const TFRGBClr&               frgb2
        );

        friend TFRGBClr CIDLIBEXP operator*
        (
            const   TFRGBClr&               frgb1
            , const tCIDLib::TFloat8&       f8Scale
        );

        tCIDLib::TVoid operator*=
        (
            const   TFRGBClr&               frgbSrc
        );

        tCIDLib::TVoid operator*=
        (
            const   tCIDLib::TFloat8&       f8Scale
        );

        friend TFRGBClr CIDLIBEXP operator/
        (
            const   TFRGBClr&               frgb1
            , const TFRGBClr&               frgb2
        );

        friend TFRGBClr CIDLIBEXP operator/
        (
            const   TFRGBClr&               frgb1
            , const tCIDLib::TFloat8&       f8Divisor
        );

        tCIDLib::TVoid operator/=
        (
            const   TFRGBClr&               frgbSrc
        );

        tCIDLib::TVoid operator/=
        (
            const   tCIDLib::TFloat8&       f8Divisor
        );

        tCIDLib::TBoolean operator==
        (
            const   TFRGBClr&               frgbToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TFRGBClr&               frgbToTest
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid AddScaled
        (
            const   TFRGBClr&               frgbBase
            , const tCIDLib::TFloat8&       f8Scale
        );

        tCIDLib::TVoid Adjust
        (
            const   tCIDLib::TFloat8&       f8Red
            , const tCIDLib::TFloat8&       f8Green
            , const tCIDLib::TFloat8&       f8Blue
            , const tCIDLib::TFloat8&       f8Alpha = 0.0
        );

        tCIDLib::TVoid Clip();

        tCIDLib::TVoid ConvertToRGB
        (
                    TRGBClr&                rgbToFill
        )   const;

        tCIDLib::TVoid FadeToBlack();

        tCIDLib::TFloat8 f8Alpha() const;

        tCIDLib::TFloat8 f8Alpha
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Blue() const;

        tCIDLib::TFloat8 f8Blue
        (
            const   tCIDLib::TFloat8&       f8New
        );

        friend tCIDLib::TFloat8 CIDLIBEXP f8Distance
        (
            const   TFRGBClr&               frgb1
            , const TFRGBClr&               frgb2
        );

        tCIDLib::TFloat8 f8Distance
        (
            const   TFRGBClr&               frgb2
        );

        tCIDLib::TFloat8 f8Green() const;

        tCIDLib::TFloat8 f8Green
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TFloat8 f8Magnitude() const;

        tCIDLib::TFloat8 f8Red() const;

        tCIDLib::TFloat8 f8Red
        (
            const   tCIDLib::TFloat8&       f8New
        );

        tCIDLib::TVoid Interpolate
        (
            const   tCIDLib::TFloat8&       f8Frac
            , const TFRGBClr&               frgbStartClr
            , const TFRGBClr&               frgbEndClr
        );

        tCIDLib::TVoid SetToScaled
        (
            const   TFRGBClr&               frgbBase
            , const tCIDLib::TFloat8&       f8Scale
        );


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private             :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __f8Red
        //  __f8Green
        //  __f8Blue
        //      The red,green, and blue values as percentage values from 0.0
        //      to 1.0.
        //
        //  __f8Alpha
        //      The alpha component is a measure of transparency, which 0.0
        //      be totally opaque and 1.0 being totally transparent.
        // --------------------------------------------------------------------
        tCIDLib::TFloat8        __f8Red;
        tCIDLib::TFloat8        __f8Green;
        tCIDLib::TFloat8        __f8Blue;
        tCIDLib::TFloat8        __f8Alpha;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFRGBClr,TObject)
        DefPolyDup(TFRGBClr)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  TRGBClr: Constructors and destructors
// ----------------------------------------------------------------------------
inline TRGBClr::TRGBClr() :

    __c1Blue(0)
    , __c1Green(0)
    , __c1Red(0)
    , __c1Dummy(0)
{
}

inline TRGBClr::TRGBClr(const TRGBClr::TRawRGB& rgbRaw) :

    __c1Blue(rgbRaw.c1Blue)
    , __c1Green(rgbRaw.c1Green)
    , __c1Red(rgbRaw.c1Red)
    , __c1Dummy(0)
{
}

inline TRGBClr::TRGBClr(const   tCIDLib::TCard1 c1Red
                        , const tCIDLib::TCard1 c1Green
                        , const tCIDLib::TCard1 c1Blue) :
    __c1Blue(c1Blue)
    , __c1Green(c1Green)
    , __c1Red(c1Red)
    , __c1Dummy(0)
{
}

inline TRGBClr::TRGBClr(const TRGBClr& rgbSrc) :

    __c1Blue(rgbSrc.__c1Blue)
    , __c1Green(rgbSrc.__c1Green)
    , __c1Red(rgbSrc.__c1Red)
    , __c1Dummy(0)
{
}

// ----------------------------------------------------------------------------
//  TRGBClr: Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TRGBClr::operator!=(const TRGBClr& rgbToTest) const
{
    return !operator==(rgbToTest);
}


// ----------------------------------------------------------------------------
//  TRGBClr: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TCard1 TRGBClr::c1Blue() const
{
    return __c1Blue;
}

inline tCIDLib::TCard1 TRGBClr::c1Blue(const tCIDLib::TCard1 c1Blue)
{
    __c1Blue = c1Blue;
    return c1Blue;
}

inline tCIDLib::TCard1 TRGBClr::c1Green() const
{
    return __c1Green;
}

inline tCIDLib::TCard1 TRGBClr::c1Green(const tCIDLib::TCard1 c1Green)
{
    __c1Green = c1Green;
    return c1Green;
}

inline tCIDLib::TCard1 TRGBClr::c1Red() const
{
    return __c1Red;
}

inline tCIDLib::TCard1 TRGBClr::c1Red(const tCIDLib::TCard1 c1Red)
{
    __c1Red = c1Red;
    return c1Red;
}

inline tCIDLib::TCard4 TRGBClr::c4Color() const
{
    return *((const tCIDLib::TCard4*)&__c1Red);
}

inline tCIDLib::TVoid TRGBClr::ToBlack()
{
    __c1Red = 0;
    __c1Green = 0;
    __c1Blue = 0;
}


// ----------------------------------------------------------------------------
//  TFRGBClr: Constructors and destructors
// ----------------------------------------------------------------------------
inline TFRGBClr::TFRGBClr(  const   tCIDLib::TFloat8&   f8Red
                            , const tCIDLib::TFloat8&   f8Green
                            , const tCIDLib::TFloat8&   f8Blue
                            , const tCIDLib::TFloat8&   f8Alpha) :
    __f8Red(f8Red)
    , __f8Green(f8Green)
    , __f8Blue(f8Blue)
    , __f8Alpha(f8Alpha)
{
}

inline TFRGBClr::TFRGBClr(const TFRGBClr& frgbToCopy) :

    __f8Red(frgbToCopy.__f8Red)
    , __f8Green(frgbToCopy.__f8Green)
    , __f8Blue(frgbToCopy.__f8Blue)
    , __f8Alpha(frgbToCopy.__f8Alpha)
{
}

inline TFRGBClr::TFRGBClr() :

    __f8Red(0.0)
    , __f8Green(0.0)
    , __f8Blue(0.0)
    , __f8Alpha(0.0)
{
}

inline TFRGBClr::~TFRGBClr()
{
}


// ----------------------------------------------------------------------------
//  TFRGBClr: Public operators
// ----------------------------------------------------------------------------
inline TFRGBClr& TFRGBClr::operator=(const TFRGBClr& frgbToAssign)
{
    if (this == &frgbToAssign)
        return *this;

    __f8Red     = frgbToAssign.__f8Red;
    __f8Green   = frgbToAssign.__f8Green;
    __f8Blue    = frgbToAssign.__f8Blue;
    __f8Alpha   = frgbToAssign.__f8Alpha;

    return *this;
}

inline tCIDLib::TVoid TFRGBClr::operator+=(const tCIDLib::TFloat8& f8Inc)
{
    __f8Red += f8Inc;
    __f8Green += f8Inc;
    __f8Blue += f8Inc;
    __f8Alpha += f8Inc;
}

inline tCIDLib::TVoid TFRGBClr::operator+=(const TFRGBClr& frgbSrc)
{
    __f8Red += frgbSrc.__f8Red;
    __f8Green += frgbSrc.__f8Green;
    __f8Blue += frgbSrc.__f8Blue;
    __f8Alpha += frgbSrc.__f8Alpha;
}

inline tCIDLib::TVoid TFRGBClr::operator*=(const tCIDLib::TFloat8& f8Scale)
{
    __f8Red *= f8Scale;
    __f8Green *= f8Scale;
    __f8Blue *= f8Scale;
    __f8Alpha *= f8Scale;
}

inline tCIDLib::TVoid TFRGBClr::operator*=(const TFRGBClr& frgbSrc)
{
    __f8Red *= frgbSrc.__f8Red;
    __f8Green *= frgbSrc.__f8Green;
    __f8Blue *= frgbSrc.__f8Blue;
    __f8Alpha *= frgbSrc.__f8Alpha;
}

inline tCIDLib::TVoid TFRGBClr::operator-=(const TFRGBClr& frgbSrc)
{
    __f8Red -= frgbSrc.__f8Red;
    __f8Green -= frgbSrc.__f8Green;
    __f8Blue -= frgbSrc.__f8Blue;
    __f8Alpha -= frgbSrc.__f8Alpha;
}

inline tCIDLib::TVoid TFRGBClr::operator-=(const tCIDLib::TFloat8& f8Dec)
{
    __f8Red -= f8Dec;
    __f8Green -= f8Dec;
    __f8Blue -= f8Dec;
    __f8Alpha -= f8Dec;
}

inline tCIDLib::TVoid TFRGBClr::operator/=(const tCIDLib::TFloat8& f8Divisor)
{
    __f8Red /= f8Divisor;
    __f8Green /= f8Divisor;
    __f8Blue /= f8Divisor;
    __f8Alpha /= f8Divisor;
}

inline tCIDLib::TVoid TFRGBClr::operator/=(const TFRGBClr& frgbSrc)
{
    __f8Red /= frgbSrc.__f8Red;
    __f8Green /= frgbSrc.__f8Green;
    __f8Blue /= frgbSrc.__f8Blue;
    __f8Alpha /= frgbSrc.__f8Alpha;
}

inline tCIDLib::TBoolean
TFRGBClr::operator!=(const TFRGBClr& frgbToTest) const
{
    return !operator==(frgbToTest);
}


// ----------------------------------------------------------------------------
//  TFRGBClr: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TVoid TFRGBClr::Adjust( const   tCIDLib::TFloat8&   f8Red
                                        , const tCIDLib::TFloat8&   f8Green
                                        , const tCIDLib::TFloat8&   f8Blue
                                        , const tCIDLib::TFloat8&   f8Alpha)
{
    __f8Red += f8Red;
    __f8Green += f8Green;
    __f8Blue += f8Blue;
    __f8Alpha += f8Alpha;
}

inline tCIDLib::TVoid TFRGBClr::FadeToBlack()
{
    __f8Red = 0.0;
    __f8Green = 0.0;
    __f8Blue = 0.0;
    __f8Alpha = 0.0;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Alpha() const
{
    return __f8Alpha;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Alpha(const tCIDLib::TFloat8& f8New)
{
    __f8Alpha = f8New;
    return __f8Alpha;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Blue() const
{
    return __f8Blue;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Blue(const tCIDLib::TFloat8& f8New)
{
    __f8Blue = f8New;
    return __f8Blue;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Green() const
{
    return __f8Green;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Green(const tCIDLib::TFloat8& f8New)
{
    __f8Green = f8New;
    return __f8Green;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Red() const
{
    return __f8Red;
}

inline tCIDLib::TFloat8 TFRGBClr::f8Red(const tCIDLib::TFloat8& f8New)
{
    __f8Red = f8New;
    return __f8Red;
}
