//
// NAME: CIDLib_Color.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRGBClr and TFRGBClr classes. TRGBClr is
//  the usual color type, with cardinal values from 0 to kCIDLib::c1MaxCard
//  for each color component. TFRGBClr uses a floating point representation
//  (from 0.0 to 1.0) and provides an opacity term.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 10/05/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  CIDLib includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TRGBClr,TObject)
RTTIData2(TFRGBClr,TObject)


// -----------------------------------------------------------------------------
//   CLASS: TRGBClr
//  PREFIX: rgb
//
//  This class is used for all color graphic output.
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRGBClr: Constructors and operators
// -----------------------------------------------------------------------------

TRGBClr::TRGBClr(const tCIDLib::TCard4 c4Color) :

    __c1Red(((const TRGBClr::TRawRGB*)&c4Color)->c1Red)
    , __c1Green(((const TRGBClr::TRawRGB*)&c4Color)->c1Green)
    , __c1Blue(((const TRGBClr::TRawRGB*)&c4Color)->c1Blue)
    , __c1Dummy(0)
{
}

TRGBClr::TRGBClr(const tCIDLib::ESysColors eSysColor) :

    __c1Red(0)
    , __c1Green(0)
    , __c1Blue(0)
    , __c1Dummy(0)
{
    *this = TSysInfo::rgbSystemColor(eSysColor);
}


// -----------------------------------------------------------------------------
//  TRGBClr: Public operators
// -----------------------------------------------------------------------------

TRGBClr& TRGBClr::operator=(const tCIDLib::TCard4 c4Color)
{
    __c1Red     = ((const TRGBClr::TRawRGB*)&c4Color)->c1Red;
    __c1Green   = ((const TRGBClr::TRawRGB*)&c4Color)->c1Green;
    __c1Blue    = ((const TRGBClr::TRawRGB*)&c4Color)->c1Blue;
    __c1Dummy   = 0;

    return *this;
}

TRGBClr& TRGBClr::operator=(const TRGBClr& rgbToAssign)
{
    if (this == &rgbToAssign)
        return *this;

    __c1Red     = rgbToAssign.__c1Red;
    __c1Green   = rgbToAssign.__c1Green;
    __c1Blue    = rgbToAssign.__c1Blue;
    __c1Dummy   = 0;

    return *this;
}

TRGBClr& TRGBClr::operator=(const TRGBClr::TRawRGB& rgbRaw)
{
    __c1Red     = rgbRaw.c1Red;
    __c1Green   = rgbRaw.c1Green;
    __c1Blue    = rgbRaw.c1Blue;
    __c1Dummy   = 0;

    return *this;
}

TRGBClr& TRGBClr::operator=(const TRGBClr::TCodedRGB& rgbCoded)
{
    __c1Red     = rgbCoded.c1Red;
    __c1Green   = rgbCoded.c1Green;
    __c1Blue    = rgbCoded.c1Blue;
    __c1Dummy   = 0;

    return *this;
}

tCIDLib::TBoolean TRGBClr::operator==(const TRGBClr& rgbToTest) const
{
    if (rgbToTest.__c1Red != __c1Red)
        return kCIDLib::False;

    if (rgbToTest.__c1Green != __c1Green)
        return kCIDLib::False;

    if (rgbToTest.__c1Blue != __c1Blue)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TRGBClr: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TCard1 TRGBClr::c1AdjustBlue(const tCIDLib::TInt4 i4Offset)
{
    #if CID_DEBUG_ON
    if ((i4Offset < -kCIDLib::c1MaxCard) || (i4Offset > kCIDLib::c1MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcClr_RGBRange
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    #endif

    tCIDLib::TInt4   i4Cur = __c1Blue;

    i4Cur += i4Offset;
    if (i4Cur < 0)
        __c1Blue = 0;
    else if (i4Cur > kCIDLib::c1MaxCard)
        __c1Blue = kCIDLib::c1MaxCard;
    else
        __c1Blue = tCIDLib::TCard1(i4Cur);

    return __c1Blue;
}

tCIDLib::TCard1 TRGBClr::c1AdjustGreen(const tCIDLib::TInt4 i4Offset)
{
    #if     CID_DEBUG_ON
    if ((i4Offset < -kCIDLib::c1MaxCard) || (i4Offset > kCIDLib::c1MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcClr_RGBRange
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    #endif

    tCIDLib::TInt4   i4Cur = __c1Green;

    i4Cur += i4Offset;
    if (i4Cur < 0)
        __c1Green = 0;
    else if (i4Cur > kCIDLib::c1MaxCard)
        __c1Green = kCIDLib::c1MaxCard;
    else
        __c1Green = tCIDLib::TCard1(i4Cur);

    return __c1Green;
}

tCIDLib::TCard1 TRGBClr::c1AdjustRed(const tCIDLib::TInt4 i4Offset)
{
    #if     CID_DEBUG_ON
    if ((i4Offset < -kCIDLib::c1MaxCard) || (i4Offset > kCIDLib::c1MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcClr_RGBRange
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    #endif

    tCIDLib::TInt4   i4Cur = __c1Red;

    i4Cur += i4Offset;
    if (i4Cur < 0)
        __c1Red = 0;
    else if (i4Cur > kCIDLib::c1MaxCard)
        __c1Red = kCIDLib::c1MaxCard;
    else
        __c1Red = tCIDLib::TCard1(i4Cur);

    return __c1Red;
}


tCIDLib::TVoid TRGBClr::Adjust( const   tCIDLib::TInt4  i4RedOffset
                                , const tCIDLib::TInt4  i4GreenOffset
                                , const tCIDLib::TInt4  i4BlueOffset)
{
    tCIDLib::TInt4   i4Cur;
    
    i4Cur = __c1Red;
    i4Cur += i4RedOffset;
    if (i4Cur < 0)
        __c1Red = 0;
    else if (i4Cur > kCIDLib::c1MaxCard)
        __c1Red = kCIDLib::c1MaxCard;
    else
        __c1Red = tCIDLib::TCard1(i4Cur);

    i4Cur = __c1Green;
    i4Cur += i4GreenOffset;
    if (i4Cur < 0)
        __c1Green = 0;
    else if (i4Cur > kCIDLib::c1MaxCard)
        __c1Green = kCIDLib::c1MaxCard;
    else
        __c1Green = tCIDLib::TCard1(i4Cur);

    i4Cur = __c1Blue;
    i4Cur += i4BlueOffset;
    if (i4Cur < 0)
        __c1Blue = 0;
    else if (i4Cur > kCIDLib::c1MaxCard)
        __c1Blue = kCIDLib::c1MaxCard;
    else
        __c1Blue = tCIDLib::TCard1(i4Cur);
}


tCIDLib::TCard4 TRGBClr::c4Magnitude() const
{
    return tCIDLib::TCard4(__c1Red + __c1Green + __c1Blue);
}


tCIDLib::TVoid TRGBClr::Set(const   tCIDLib::TCard1 c1Red
                            , const tCIDLib::TCard1 c1Green
                            , const tCIDLib::TCard1 c1Blue)
{
    __c1Red     = c1Red;
    __c1Green   = c1Green;
    __c1Blue    = c1Blue;
}


tCIDLib::TVoid TRGBClr::ToRGB(TRGBClr::TRawRGB& rgbTarget) const
{
    rgbTarget.c1Red      = __c1Red;
    rgbTarget.c1Green    = __c1Green;
    rgbTarget.c1Blue     = __c1Blue;
}


// -----------------------------------------------------------------------------
//  TRGBClr: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TRGBClr::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo << __c1Red << L"," << __c1Green << L"," << __c1Blue;
}


tCIDLib::TVoid TRGBClr::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __c1Red;
    strmToReadFrom >> __c1Green;
    strmToReadFrom >> __c1Blue;
    __c1Dummy = 0;
}

tCIDLib::TVoid TRGBClr::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __c1Red;
    strmToWriteTo << __c1Green;
    strmToWriteTo << __c1Blue;
}




// -----------------------------------------------------------------------------
//  CLASS: TFRGBClr
// PREFIX: frgb
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFRGBClr: Public operators
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: operator+-*/
//
// DESCRIPTION:
//
//  These operators add two colors or will add a scalar to each component
//  of a color.
// ---------------------------------------
//   INPUT: frgb1, frgb2 are the two colors when adding two colors
//          f8Inc is the scalar increment value
//
//  OUTPUT: None
//
//  RETURN: None
//
TFRGBClr operator+(const TFRGBClr& frgb1, const TFRGBClr& frgb2)
{
    return TFRGBClr( frgb1.__f8Red + frgb2.__f8Red
                    , frgb1.__f8Green + frgb2.__f8Green
                    , frgb1.__f8Blue + frgb2.__f8Blue
                    , frgb1.__f8Alpha + frgb2.__f8Alpha);
}

TFRGBClr operator+(const TFRGBClr& frgb1, const tCIDLib::TFloat8& f8Inc)
{
    return TFRGBClr( frgb1.__f8Red + f8Inc
                    , frgb1.__f8Green + f8Inc
                    , frgb1.__f8Blue + f8Inc
                    , frgb1.__f8Alpha);
}

TFRGBClr operator-(const TFRGBClr& frgb1, const TFRGBClr& frgb2)
{
    return TFRGBClr( frgb1.__f8Red - frgb2.__f8Red
                    , frgb1.__f8Green - frgb2.__f8Green
                    , frgb1.__f8Blue - frgb2.__f8Blue
                    , frgb1.__f8Alpha - frgb2.__f8Alpha);
}

TFRGBClr operator-(const TFRGBClr& frgb1, const tCIDLib::TFloat8& f8Dec)
{
    return TFRGBClr( frgb1.__f8Red - f8Dec
                    , frgb1.__f8Green - f8Dec
                    , frgb1.__f8Blue - f8Dec
                    , frgb1.__f8Alpha);
}

TFRGBClr operator*(const TFRGBClr& frgb1, const TFRGBClr& frgb2)
{
    return TFRGBClr( frgb1.__f8Red * frgb2.__f8Red
                    , frgb1.__f8Green * frgb2.__f8Green
                    , frgb1.__f8Blue * frgb2.__f8Blue
                    , frgb1.__f8Alpha * frgb2.__f8Alpha);
}

TFRGBClr
operator*(const TFRGBClr& frgb1, const tCIDLib::TFloat8& f8Scale)
{
    return TFRGBClr( frgb1.__f8Red * f8Scale
                    , frgb1.__f8Green * f8Scale
                    , frgb1.__f8Blue * f8Scale
                    , frgb1.__f8Alpha * f8Scale);
}

TFRGBClr operator/(const TFRGBClr& frgb1, const TFRGBClr& frgb2)
{
    return TFRGBClr( frgb1.__f8Red / frgb2.__f8Red
                    , frgb1.__f8Green / frgb2.__f8Green
                    , frgb1.__f8Blue / frgb2.__f8Blue
                    , frgb1.__f8Alpha / frgb2.__f8Alpha);
}

TFRGBClr
operator/(const TFRGBClr& frgb1, const tCIDLib::TFloat8& f8Divisor)
{
    return TFRGBClr( frgb1.__f8Red / f8Divisor
                    , frgb1.__f8Green / f8Divisor
                    , frgb1.__f8Blue / f8Divisor
                    , frgb1.__f8Alpha);
}

//
// FUNCTION/METHOD NAME: operator==
//
// DESCRIPTION:
//
//  This method will compare this object to the passed one.
// ---------------------------------------
//   INPUT: rgbToTest is the object to compare against
//
//  OUTPUT: None
//
//  RETURN: eTrue if equal, else eFalse
//
tCIDLib::TBoolean TFRGBClr::operator==(const TFRGBClr& frgbToTest) const
{
    // Compare the values of the objects
    if (__f8Red != frgbToTest.__f8Red)
        return kCIDLib::False;

    if (__f8Green != frgbToTest.__f8Green)
        return kCIDLib::False;

    if (__f8Blue != frgbToTest.__f8Blue)
        return kCIDLib::False;

    if (__f8Alpha != frgbToTest.__f8Alpha)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TFRGBClr: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: AddScaled
//
// DESCRIPTION:
//
//  This method will add a scaled color value to this color.
// ---------------------------------------
//   INPUT: frgbBase is the base color to scale
//          f8Scale is the value to scale by
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFRGBClr::AddScaled(const TFRGBClr& frgbBase, const tCIDLib::TFloat8& f8Scale)
{
    __f8Red     += (frgbBase.__f8Red * f8Scale);
    __f8Green   += (frgbBase.__f8Green * f8Scale);
    __f8Blue    += (frgbBase.__f8Blue * f8Scale);
    __f8Alpha   += (frgbBase.__f8Alpha * f8Scale);
}


//
// FUNCTION/METHOD NAME: Clip
//
// DESCRIPTION:
//
//  This method will clip the floating point color components to be within
//  the valid range of 0.0 to 1.0.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFRGBClr::Clip()
{
    if (__f8Red > 1.0)
        __f8Red = 1.0;
    else if (__f8Red < 0.0)
        __f8Red = 0.0;

    if (__f8Green > 1.0)
        __f8Green = 1.0;
    else if (__f8Green < 0.0)
        __f8Green = 0.0;

    if (__f8Blue > 1.0)
        __f8Blue = 1.0;
    else if (__f8Blue < 0.0)
        __f8Blue = 0.0;

    if (__f8Alpha > 1.0)
        __f8Alpha = 1.0;
    else if (__f8Alpha < 0.0)
        __f8Alpha = 0.0;
}


//
// FUNCTION/METHOD NAME: ConvertToRGB
//
// DESCRIPTION:
//
//  This method will convert the floating point RGB values used in ray
//  tracing into a normal RGB value. It clips the float values if needed to
//  keep the RGB values calculated from wrapping.
//
//  The alpha component cannot be represented in a regular RGB format, so it
//  is discarded.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: rgbFill is the TRGBClr object to fill
//
//  RETURN: None
//
tCIDLib::TVoid TFRGBClr::ConvertToRGB(TRGBClr& rgbToFill) const
{
    tCIDLib::TFloat8 f8Red, f8Green, f8Blue;

    if (__f8Red < 0.0)
        f8Red = 0.0;
     else if (__f8Red > 1.0)
        f8Red = 1.0;
     else
        f8Red = __f8Red;

    if (__f8Green < 0.0)
        f8Green = 0.0;
     else if (__f8Green > 1.0)
        f8Green = 1.0;
     else
        f8Green = __f8Green;

    if (__f8Blue < 0.0)
        f8Blue = 0.0;
     else if (__f8Blue > 1.0)
        f8Blue = 1.0;
     else
        f8Blue = __f8Blue;

    rgbToFill.Set
    (
        tCIDLib::TCard1(f8Red * kCIDLib::c1MaxCard)
        , tCIDLib::TCard1(f8Green * kCIDLib::c1MaxCard)
        , tCIDLib::TCard1(f8Blue * kCIDLib::c1MaxCard)
    );
}


//
// FUNCTION/METHOD NAME: f8Distance
//
// DESCRIPTION:
//
//  These methods will return the sum of the absolute values of the
//  differences of the color components.
// ---------------------------------------
//   INPUT: frgb1, frgb2 are the colors to operate on.
//
//  OUTPUT: None
//
//  RETURN: The color distance
//
tCIDLib::TFloat8
f8Distance(const TFRGBClr& frgb1, const TFRGBClr& frgb2)
{
    return  TMathLib::f8Abs(frgb1.__f8Red - frgb2.__f8Red)
            + TMathLib::f8Abs(frgb1.__f8Green - frgb2.__f8Green)
            + TMathLib::f8Abs(frgb1.__f8Blue - frgb2.__f8Blue);
}

tCIDLib::TFloat8 TFRGBClr::f8Distance(const TFRGBClr& frgb2)
{
    return  TMathLib::f8Abs(__f8Red - frgb2.__f8Red)
            + TMathLib::f8Abs(__f8Green - frgb2.__f8Green)
            + TMathLib::f8Abs(__f8Blue - frgb2.__f8Blue);
}


//
// FUNCTION/METHOD NAME: f8Magnitude
//
// DESCRIPTION:
//
//  Returns the 'magnitude' of the color, which is the sum of the color
//  component values. It is for telling the absolute difference between
//  color values, i.e. its not 'weighted' for human eye sensitivity.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: The magnitude value.
//
tCIDLib::TFloat8 TFRGBClr::f8Magnitude() const
{
    return tCIDLib::TFloat8(__f8Red + __f8Green + __f8Blue);
}


//
// FUNCTION/METHOD NAME: Interpolate
//
// DESCRIPTION:
//
//  This method will interpolate a color between the two colors. 
// ---------------------------------------
//   INPUT: f8Frac is the percentage of interpolation between the start and
//              end colors.
//          frgbStartClr, frgbEndClr are the colors to interpolate between.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFRGBClr::Interpolate(  const   tCIDLib::TFloat8&   f8Frac
                        , const TFRGBClr&           frgbStartClr
                        , const TFRGBClr&           frgbEndClr)
{
    __f8Red   = frgbStartClr.__f8Red
                + (f8Frac * (frgbEndClr.__f8Red - frgbStartClr.__f8Red));

    __f8Green = frgbStartClr.__f8Green
                + (f8Frac * (frgbEndClr.__f8Green - frgbStartClr.__f8Green));

    __f8Blue  = frgbStartClr.__f8Blue
                + (f8Frac * (frgbEndClr.__f8Blue - frgbStartClr.__f8Blue));

    __f8Alpha = frgbStartClr.__f8Alpha
                + (f8Frac * (frgbEndClr.__f8Alpha - frgbStartClr.__f8Alpha));
}


//
// FUNCTION/METHOD NAME: SetToScaled
//
// DESCRIPTION:
//
//  This method will set this color to another color scaled by a scale
//  factor. If a component is scaled over 1.0, it will be clipped.
// ---------------------------------------
//   INPUT: fgbBase is the color to scale
//          f8Scale is the scale factor
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFRGBClr::SetToScaled(  const    TFRGBClr&          frgbBase
                        , const tCIDLib::TFloat8&   f8Scale)
{
    __f8Red     = frgbBase.__f8Red * f8Scale;
    __f8Green   = frgbBase.__f8Green * f8Scale;
    __f8Blue    = frgbBase.__f8Blue * f8Scale;
    __f8Alpha   = frgbBase.__f8Alpha * f8Scale;
}

// -----------------------------------------------------------------------------
//  TFRGBClr: Protected, inherited methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _FormatTo
//
// DESCRIPTION:
//
//  This method will format this object's value into the passed stream.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: strmToWriteTo is the stream to format to.
//
//  RETURN: None
//
tCIDLib::TVoid TFRGBClr::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    strmToWriteTo   << TFloat(__f8Red, 5)
                    << L"," << TFloat(__f8Green, 5)
                    << L"," << TFloat(__f8Blue, 5);
}


//
// FUNCTION/METHOD NAME: _StreamFrom
//                       _StreamTo
//
// DESCRIPTION:
//
//  These methods handle streaming floating point color objects in and out.
//  This one is pretty danged simple since it just has 4 fundamental members.
// ---------------------------------------
//   INPUT: strmToReadFrom is the stream to read in from
//          stramToWriteTo is the stream to stream out to
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFRGBClr::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> __f8Red;
    strmToReadFrom >> __f8Green;
    strmToReadFrom >> __f8Blue;
    strmToReadFrom >> __f8Alpha;
}


tCIDLib::TVoid TFRGBClr::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __f8Red;
    strmToWriteTo << __f8Green;
    strmToWriteTo << __f8Blue;
    strmToWriteTo << __f8Alpha;
}
